package com.xdja.platform.service.rmi.core;

import java.rmi.AccessException;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.util.HashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * rmi服务端。
 * @author 武宗品
 * @时间 2013年10月23日
 */
public class RmiServer<RMIInterface extends IRmiCommonRemote> {
	
	private static final Logger logger = LoggerFactory.getLogger(RmiServer.class);
	
	/**
	 * 已经绑定的服务端
	 */
	@SuppressWarnings("rawtypes")
	private static HashMap<String, RmiServer> RMI_SERVERS = new HashMap<String, RmiServer>();
	
	/**
	 * ip地址。为空时默认本机
	 */
	private final String serverIp;
	
	/**
	 * 端口号
	 */
	private final int port;
	
	/**
	 * 服务名
	 */
	private final String name;
	
	/**
	 * 绑定的远程方法
	 */
	private final Remote remote;

	/**
	 * 构造函数
	 * @param serverIp 服务器ip地址，为空时默认本机。
	 * @param port 端口号
	 * @param name 服务名
	 * @param remote 绑定的服务类
	 */
	private RmiServer(String serverIp, int port, String name, Remote remote) {
		this.serverIp = serverIp;
		this.port = port;
		this.name = name;
		this.remote = remote;
	}
	
	/**
	 * 根据服务器ip地址、端口号、服务名开启rmiServer。一个服务器ip地址+端口号+服务名只能挂载一个Remote，如果同一个已经存在，则不再挂载，并返回之前挂载的服务。<br/>
	 * 调用方必须保证传入参数的正确性，并且RMIImpl必须是一个真正的实现类。否则出错将返回null。运行时异常将不由本方法负责。
	 * @param serverIp 服务器ip地址，为空时默认本机。
	 */
	public static <RMIInterface extends IRmiCommonRemote, RMIImpl extends RMIInterface> RmiServer<RMIInterface> startRMIServer(
			final String serverIp, final int port, final String name, final Class<RMIImpl> clazz) 
			throws AccessException, RemoteException {
		RMIImpl remote = null;
		String rmiServerKey = serverIp + port + name;
		@SuppressWarnings("unchecked")
		RmiServer<RMIInterface> rmiServer = (RmiServer<RMIInterface>)RMI_SERVERS.get(rmiServerKey);
		if(rmiServer == null) {
			try {
				remote = clazz.newInstance();
			} catch (InstantiationException e) {
				logger.error("构造RMIImpl失败", e);
				return null;
			} catch (IllegalAccessException e) {
				logger.error("构造RMIImpl失败", e);
				return null;
			}
			rmiServer = new RmiServer<RMIInterface>(serverIp, port, name, remote);
			getRegistry(serverIp, port).rebind(name, remote);
			RMI_SERVERS.put(rmiServerKey, rmiServer);
		}
		return rmiServer;
	}
	
	
	
	 /**
     * 创建一个Registry对象.
	 * @param serverIp 服务器ip地址，为空时默认本机。
     */
    private static Registry getRegistry(String serverIp, int port) throws RemoteException {
		Registry registry = null;
		try {
			if(serverIp == null) {
				registry = LocateRegistry.getRegistry(port);
			} else {
				registry = LocateRegistry.getRegistry(serverIp, port);
			}
			registry.list();
		} catch (RemoteException e) {
			if(serverIp == null) {
				registry = LocateRegistry.createRegistry(port);
			} else {
				registry = LocateRegistry.createRegistry(port);
				registry = LocateRegistry.getRegistry(serverIp, port);
				registry.list();
			}
		}
		return registry;
    }
    
	public int getPort() {
		return port;
	}

	public String getName() {
		return name;
	}

	public Remote getRemote() {
		return remote;
	}

	public String getServerIp() {
		return serverIp;
	}
}
