package com.xdja.plugin.shadow.nonedynamic;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.os.Parcel;
import android.os.StrictMode;
import android.text.TextUtils;
import android.util.Log;

import com.tencent.shadow.core.common.InstalledApk;
import com.tencent.shadow.core.common.LoggerFactory;
import com.tencent.shadow.core.load_parameters.LoadParameters;
import com.tencent.shadow.core.loader.ShadowPluginLoader;
import com.tencent.shadow.core.runtime.container.ContentProviderDelegateProviderHolder;
import com.tencent.shadow.core.runtime.container.DelegateProviderHolder;
import com.xdja.plugin.shadow.nonedynamic.bean.InstalledShadowPluginInfo;
import com.xdja.plugin.shadow.nonedynamic.jar.ErrorCode;
import com.xdja.plugin.shadow.nonedynamic.param.ParamKeywords;
import com.xdja.plugin.shadow.nonedynamic.storage.StorageInstalledPlugin;
import com.xdja.plugin.shadow.nonedynamic.utils.ApkParser;
import com.xdja.plugin.shadow.nonedynamic.utils.ConvertUtil;
import com.xdja.plugin.shadow.nonedynamic.utils.FileUtil;
import com.xdja.plugin.shadow.nonedynamic.utils.PkgInfoUtil;
import com.xdja.plugin.shadow.nonedynamic.utils.Utils;

import org.apache.commons.io.FileUtils;

import java.io.File;
import java.util.ArrayList;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * @author: zhangxiaolong@xdja.com <br/>
 * date:   2020/2/14 <br/>
 */
public class ShadowManager {

    static {
        detectNonSdkApiUsageOnAndroidP();
        LoggerFactory.setILoggerFactory(new XdjaLoggerFactory());
    }

    private static final String PATH_PLUGIN = "ShadowNoneDynamicPlugin";

    private static ShadowManager instance;

    private ShadowPluginLoader mPluginLoader;
    private Context context;

    private ShadowManager() {

    }

    public static ShadowManager getInstance() {
        if (instance == null) {
            synchronized (ShadowManager.class) {
                if (instance == null) {
                    instance = new ShadowManager();
                }
            }
        }
        return instance;
    }

    ExecutorService cachedThreadPool = Executors.newCachedThreadPool();


    public void init(Context context) {
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }

        ShadowPluginLoader loader = mPluginLoader = new XdjaPluginLoader(context);
        loader.onCreate();
        DelegateProviderHolder.setDelegateProvider(loader.getDelegateProviderKey(), loader);
        ContentProviderDelegateProviderHolder.setContentProviderDelegateProvider(loader);
    }

    /**
     * 获取所有安装插件
     *
     * @return
     */
    public ArrayList<InstalledShadowPluginInfo> getAll() {
        ArrayList<InstalledShadowPluginInfo> all = StorageInstalledPlugin.getAll(context);
        return all;
    }

    /**
     * 卸载
     *
     * @param partKey
     * @return
     */
    public int uninstall(final String partKey) {
        if (TextUtils.isEmpty(partKey)) {
            return -1;
        }
        final InstalledShadowPluginInfo plugin = StorageInstalledPlugin.get(context, partKey);
        if (plugin == null) {
            return -1;
        }
        StorageInstalledPlugin.delete(context, partKey);
        File path = new File(context.getFilesDir(), PATH_PLUGIN);

        File dir = new File(path, "/" + partKey + "/" + plugin.version);
        FileUtil.deleteFile(dir);

        return 0;
    }

    /**
     * 启动插件中的activity
     *
     * @param partKey
     * @param activityName
     * @return
     */
    public int startPluginActivity(final String partKey, String activityName) {
        //判断参数
        if (TextUtils.isEmpty(partKey)) {
            return ErrorCode.RET_PARAM_ERROR;
        }

        final InstalledShadowPluginInfo plugin = StorageInstalledPlugin.get(context, partKey);
        if (plugin == null) {
            return ErrorCode.RET_NOT_FIND_PLUGIN;
        }

        final String startActivityName;
        if (TextUtils.isEmpty(activityName)) {
            startActivityName = plugin.mainActivityName;
        } else {
            startActivityName = activityName;
        }
        if (TextUtils.isEmpty(startActivityName)) {
            return ErrorCode.RET_NOT_FIND_PLUGIN;
        }

        if (mPluginLoader.getPluginParts(partKey) == null) {
            cachedThreadPool.submit(new Callable<Integer>() {
                @Override
                public Integer call() {
                    try {
                        mPluginLoader.loadPlugin(ConvertUtil.convert(plugin.installedApk)).get(10, TimeUnit.SECONDS);
                    } catch (Exception e) {
                        e.printStackTrace();
                        return ErrorCode.RET_LOAD_PLUGIN_FAIL;
                    }
                    mPluginLoader.callApplicationOnCreate(partKey, XdjaPluginContainerProvider.getContentProvider());
                    return startActivity(startActivityName);
                }
            });
        } else {
            return startActivity(startActivityName);
        }
        return 0;
    }

    /**
     * 安装插件
     *
     * @param partKey
     * @param version
     * @param pluginFile
     * @return
     */
    public int install(String partKey, String version, File pluginFile) {
        int ret;
        boolean flag;
        //判断参数
        if (TextUtils.isEmpty(partKey) || TextUtils.isEmpty(version) || pluginFile == null || !pluginFile.exists()) {
            return ErrorCode.RET_PARAM_ERROR;
        }

        //判断版本号大小。如果没有安装，表示可以往下进行
        InstalledShadowPluginInfo pluginInfo = StorageInstalledPlugin.get(context, partKey);
        if (pluginInfo != null) {
            flag = Utils.judgeVersion(version, pluginInfo.version);
            if (!flag) {
                return ErrorCode.RET_VERSION_LOW;
            }
        }

        //判断插件能否安装
        flag = judgePluginCanInstall(pluginFile);
        if (!flag) {
            return ErrorCode.RET_NOT_ALLOW_INSTALL;
        }

        //安装
        ret = installPlugin(partKey, version, pluginFile);
        //如果返回值为0，通知安装成功
        if (ret == 0) {
            notifyInstalledPlugin(partKey, version);
        }

        return ret;
    }

    /**
     * 判断插件能否安装。<br>
     * 比如: 完整性；签名等
     *
     * @param pluginFile
     * @return
     */
    private boolean judgePluginCanInstall(File pluginFile) {
        if (PkgInfoUtil.isApkInDebug(context)) {
            return true;
        }
        String pluginCert = PkgInfoUtil.getCertFingerStrFromApkFile(context, pluginFile.getAbsolutePath());
        String selfCert = PkgInfoUtil.getCertFingerStr(context, context.getPackageName());
        try {
            if (selfCert.equals(pluginCert)) {
                return true;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return false;
    }

    /**
     * 安装插件
     *
     * @param pluginFile
     * @return
     */
    private int installPlugin(String partKey, String version, File pluginFile) {
        File path = new File(context.getFilesDir(), PATH_PLUGIN);
        String hash = FileUtil.getCrc32(pluginFile);

        File dir = new File(path, "/" + partKey + "/" + version + "/" + hash);
        File objFile = new File(dir, "plugin.apk");
        File libDir = new File(dir, "lib");
        File oDexDir = new File(dir, "odex");
        oDexDir.mkdirs();

        try {
            //copy文件
            FileUtils.copyFile(pluginFile, objFile);
            //解压so库
            FileUtil.extraSo(context, objFile, libDir);

            InstalledShadowPluginInfo installedPluginInfo = StorageInstalledPlugin.get(context, partKey);
            if (installedPluginInfo != null) {
                String apkFilePath = installedPluginInfo.installedApk.apkFilePath;
                File installedFileDir = new File(apkFilePath).getParentFile();
                String installedFileDirPath = installedFileDir.getAbsolutePath();
                if (!installedFileDirPath.equals(dir.getAbsolutePath())) {
                    FileUtil.deleteFile(installedFileDir);
                }
            }

            LoadParameters loadParameters = new LoadParameters(null, partKey, null, null);
            Parcel parcel = Parcel.obtain();
            loadParameters.writeToParcel(parcel, 0);
            InstalledApk installedApk = new InstalledApk(objFile.getAbsolutePath(), oDexDir.getAbsolutePath(), libDir.getAbsolutePath(), parcel.marshall());
            InstalledShadowPluginInfo pluginInfo = new InstalledShadowPluginInfo(partKey, version, ConvertUtil.convert(installedApk));
            ApkParser apkParser = ApkParser.build(context, objFile.getAbsolutePath());
            if (apkParser != null) {
                pluginInfo.mainActivityName = apkParser.getMainActivityName();
                pluginInfo.info.put("activity", apkParser.getActivityList());
                pluginInfo.info.put("service", apkParser.getServiceList());
                pluginInfo.info.put("provider", apkParser.getProviderList());
                pluginInfo.info.put("receiver", apkParser.getReceiverList());
            }

            StorageInstalledPlugin.add(context, partKey, pluginInfo);
            parcel.recycle();
            return 0;
        } catch (Exception e) {
            e.printStackTrace();
        }

        return ErrorCode.RET_INSTALL_ERROR;
    }

    /**
     * 通知已安装好了插件
     */
    private void notifyInstalledPlugin(String partKey, String version) {
        //TODO: 还未想好怎么通知
        Log.e("test", partKey + "_" + version + " : installed");
    }

    private int startActivity(String activityName) {
        Intent pluginIntent = new Intent();
        pluginIntent.setClassName(context.getPackageName(), activityName);
        pluginIntent.putStringArrayListExtra(ParamKeywords.KEY_StringArrayList_activities, XdjaComponentManager.sActivities);
        Intent intent = mPluginLoader.getMComponentManager().convertPluginActivityIntent(pluginIntent);
        if (!(context instanceof Activity)) {
            intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        }
        context.startActivity(intent);
        return 0;
    }

    private static void detectNonSdkApiUsageOnAndroidP() {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.P) {
            return;
        }
        boolean isRunningEspressoTest;
        try {
            Class.forName("android.support.test.espresso.Espresso");
            isRunningEspressoTest = true;
        } catch (Exception ignored) {
            isRunningEspressoTest = false;
        }
        if (isRunningEspressoTest) {
            return;
        }
        StrictMode.VmPolicy.Builder builder = new StrictMode.VmPolicy.Builder();
        builder.detectNonSdkApiUsage();
        StrictMode.setVmPolicy(builder.build());
    }
}
