package com.xdja.plugin.shadow.nonedynamic.utils;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.IntentFilter;
import android.content.res.AssetManager;
import android.content.res.XmlResourceParser;
import android.os.PatternMatcher;
import android.text.TextUtils;

import org.xmlpull.v1.XmlPullParser;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;

/**
 * @author zhangxiaolong@xdja.com <br/>
 * @date 2019/10/22 <br/>
 */
public class ApkParser {

    private Context context;
    String apkPath;

    private ArrayList<String> activityList = new ArrayList<>();
    private ArrayList<String> serviceList = new ArrayList<>();
    private ArrayList<String> receiverList = new ArrayList<>();
    private ArrayList<String> providerList = new ArrayList<>();
    private HashMap<String, IntentFilter> intentFilters = new HashMap();
    private String mainActivityName;

    private ApkParser(Context context, String apkPath) {
        this.context = context;
        this.apkPath = apkPath;
    }

    public String getApkPath() {
        return apkPath;
    }

    public ArrayList<String> getActivityList() {
        return activityList;
    }

    public ArrayList<String> getServiceList() {
        return serviceList;
    }

    public ArrayList<String> getReceiverList() {
        return receiverList;
    }

    public ArrayList<String> getProviderList() {
        return providerList;
    }

    public HashMap<String, IntentFilter> getIntentFilters() {
        return intentFilters;
    }

    public String getMainActivityName() {
        return mainActivityName;
    }

    //==================================================================================================

    /**
     * 解析apk，得到想要的信息
     *
     * @param context
     * @param apkPath
     * @return
     */
    public static ApkParser build(Context context, String apkPath) {
        if (context == null || TextUtils.isEmpty(apkPath)) {
            return null;
        }

        ApkParser apkParser = new ApkParser(context, apkPath);
        apkParser.parse();
        return apkParser;
    }

    /**
     * 开始解析
     */
    private void parse() {
        AssetManager assetManager = null;
        XmlResourceParser parser = null;
        try {
            assetManager = AssetManager.class.newInstance();
            Method method = AssetManager.class.getDeclaredMethod("addAssetPath", String.class);
            method.invoke(assetManager, apkPath);
            parser = assetManager.openXmlResourceParser(ANDROID_MANIFEST_FILENAME);
            parseManifest(parser);
            parser.close();
            assetManager.close();
            parser = null;
            assetManager = null;
        } catch (Exception e) {

        } finally {
            if (parser != null) {
                parser.close();
                parser = null;
            }
            if (assetManager != null) {
                assetManager.close();
                assetManager = null;
            }
        }
    }

    /**
     * 解析AndroidMenifest.xml
     *
     * @param parser
     * @throws Exception
     */
    private void parseManifest(XmlResourceParser parser) throws Exception {
        int outerDepth = parser.getDepth();
        int type;
        while ((type = parser.next()) != XmlPullParser.END_DOCUMENT && (type != XmlPullParser.END_TAG || parser.getDepth() > outerDepth)) {
            if (type == XmlPullParser.END_TAG || type == XmlPullParser.TEXT) {
                continue;
            }
            String tagName = parser.getName();
            if (TextUtils.isEmpty(tagName)) {
                continue;
            }

            if (tagName.equals("activity") || tagName.equals("receiver") || tagName.equals("service") || tagName.equals("provider")) {
                String name = parser.getAttributeValue(ANDROID_RESOURCES, "name");
                if (!TextUtils.isEmpty(name)) {
                    writeToList(tagName, name);
                    parseIntentFilter(name, parser);
                }
            }
        }
    }

    /**
     * 写入到list中
     *
     * @param tagName
     * @param name
     */
    private void writeToList(String tagName, String name) {
        if (tagName.equals("activity")) {
            activityList.add(name);
        } else if (tagName.equals("service")) {
            serviceList.add(name);
        } else if (tagName.equals("provider")) {
            providerList.add(name);
        } else if (tagName.equals("receiver")) {
            receiverList.add(name);
        }
    }

    /**
     * 找到Intent-Filter字段，并进行解析
     *
     * @param name
     * @param parser
     * @throws Exception
     */
    private void parseIntentFilter(String name, XmlResourceParser parser) throws Exception {
        int outerDepth = parser.getDepth();
        int type;
        while ((type = parser.next()) != XmlPullParser.END_DOCUMENT && (type != XmlPullParser.END_TAG || parser.getDepth() > outerDepth)) {
            if (type == XmlPullParser.END_TAG || type == XmlPullParser.TEXT) {
                continue;
            }
            String tagName = parser.getName();
            if (tagName == null) {
                continue;
            }

            if (tagName.equals("intent-filter")) {
                IntentFilter mFilter = new IntentFilter();
                intentFilters.put(name, mFilter);
                parseIntentFilter(mFilter, parser);
                //找出mainActivity所在的name
                //注意：这里默认apk是只有一个mainActivity的，找到一个就不再找MainActivity了。
                if (TextUtils.isEmpty(mainActivityName)) {
                    boolean mainFlag = mFilter.hasAction("android.intent.action.MAIN");
                    boolean launcherFlag = mFilter.hasCategory("android.intent.category.LAUNCHER");
                    if (mainFlag && launcherFlag) {
                        mainActivityName = name;
                    }
                }
            }
        }
    }

    /**
     * 解析Intent-Filter字段
     *
     * @param mFilter
     * @param attrs
     * @throws Exception
     */
    @SuppressLint("NewApi")
    private void parseIntentFilter(IntentFilter mFilter, XmlResourceParser attrs) throws Exception {
        int outerDepth = attrs.getDepth();
        int type;
        while ((type = attrs.next()) != XmlPullParser.END_DOCUMENT && (type != XmlPullParser.END_TAG || attrs.getDepth() > outerDepth)) {
            if (type == XmlPullParser.END_TAG || type == XmlPullParser.TEXT) {
                continue;
            }
            String nodeName = attrs.getName();
            if (nodeName == null) {
                continue;
            }

            if (nodeName.equals("action")) {
                String value = attrs.getAttributeValue(ANDROID_RESOURCES, "name");
                if (!TextUtils.isEmpty(value)) {
                    mFilter.addAction(value);
                }
            } else if (nodeName.equals("category")) {
                String value = attrs.getAttributeValue(ANDROID_RESOURCES, "name");
                if (!TextUtils.isEmpty(value)) {
                    mFilter.addCategory(value);
                }
            } else if (nodeName.equals("data")) {
                String host = attrs.getAttributeValue(ANDROID_RESOURCES, "host");
                String port = attrs.getAttributeValue(ANDROID_RESOURCES, "port");
                if (!TextUtils.isEmpty(host) && !TextUtils.isEmpty(port)) {
                    mFilter.addDataAuthority(host, port);
                }

                String scheme = attrs.getAttributeValue(ANDROID_RESOURCES, "scheme");
                if (!TextUtils.isEmpty(scheme)) {
                    mFilter.addDataScheme(scheme);
                }

                String path = attrs.getAttributeValue(ANDROID_RESOURCES, "path");
                if (!TextUtils.isEmpty(path)) {
                    mFilter.addDataPath(path, PatternMatcher.PATTERN_LITERAL);
                }

                String pathPrefix = attrs.getAttributeValue(ANDROID_RESOURCES, "pathPrefix");
                if (!TextUtils.isEmpty(pathPrefix)) {
                    mFilter.addDataPath(pathPrefix, PatternMatcher.PATTERN_PREFIX);
                }

                String pathPattern = attrs.getAttributeValue(ANDROID_RESOURCES, "pathPattern");
                if (!TextUtils.isEmpty(pathPattern)) {
                    mFilter.addDataPath(pathPattern, PatternMatcher.PATTERN_SIMPLE_GLOB);
                }

                String mimeType = attrs.getAttributeValue(ANDROID_RESOURCES, "mimeType");
                if (!TextUtils.isEmpty(mimeType)) {
                    mFilter.addDataType(mimeType);
                }

                if (android.os.Build.VERSION.SDK_INT >= 19) {
                    String ssp = attrs.getAttributeValue(ANDROID_RESOURCES, "ssp");
                    mFilter.addDataSchemeSpecificPart(ssp, PatternMatcher.PATTERN_LITERAL);

                    String sspPrefix = attrs.getAttributeValue(ANDROID_RESOURCES, "sspPrefix");
                    mFilter.addDataSchemeSpecificPart(sspPrefix, PatternMatcher.PATTERN_PREFIX);

                    String sspPattern = attrs.getAttributeValue(ANDROID_RESOURCES, "sspPattern");
                    mFilter.addDataSchemeSpecificPart(sspPattern, PatternMatcher.PATTERN_SIMPLE_GLOB);
                }
            }
        }
    }

    private static final String ANDROID_RESOURCES = "http://schemas.android.com/apk/res/android";
    private static final String ANDROID_MANIFEST_FILENAME = "AndroidManifest.xml";

}
