package com.xdja.prs.authentication;

import java.beans.PropertyVetoException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.mchange.v2.c3p0.ComboPooledDataSource;
import com.xdja.prs.authentication.support.csagent.AppUserArea;

/**
 * @description: 数据库查询Manager
 * 				共同处理CSAgent鉴权数据和DRS鉴权数据
 * 				CSAgent鉴权数据维护SN-资源关系，DRS鉴权数据维护SN-本地表关系（资源和本地表关联，比CSAgent多关联出来本地表名）
 * @2015年9月8日 下午6:56:08
 * @author: RPP
 * @note:
 */
public class DataManager {
	
	private NamedParameterJdbcTemplate mdpJdbc;
    private NamedParameterJdbcTemplate prsJdbc;
    private NamedParameterJdbcTemplate pamsJdbc;
    
    /**
     * 缓存数据源
     */
    private static Map<String, ComboPooledDataSource> dsMap = Maps.newHashMap();
    
    /**
     * 初始化MDP、PRS、PAMS数据库连接
     * @param props
     * @throws PropertyVetoException
     */
    public DataManager(Properties props) throws PropertyVetoException{
    	
    	 mdpJdbc = initJdbcTemplate(props, "mdp");
         prsJdbc = initJdbcTemplate(props, "prs");
         pamsJdbc = initJdbcTemplate(props, "pams");
    }

    /**
     * @description: 初始化指定数据库连接
     * @param props 数据库配置信息property
     * @param tag 指定数据库（与配置文件中相应）
     * @return SpringJdbc
     * @throws PropertyVetoException
     * @2015年9月8日 下午7:51:09
     * @author: RPP
     */
	private NamedParameterJdbcTemplate initJdbcTemplate(Properties props, String tag) throws PropertyVetoException {
        String urlKey = "auth." + tag + ".jdbc.url";
        String url = props.getProperty(urlKey);
        Assert.hasText(url, urlKey);
        String driverKey = "auth." + tag + ".jdbc.driver";
        String driver = props.getProperty(driverKey);
        Assert.hasText(driver, driver);
        String usernameKey = "auth." + tag + ".jdbc.username";
        String username = props.getProperty(usernameKey);
        Assert.hasText(username, usernameKey);
        String passwordKey = "auth." + tag + ".jdbc.password";
        String password = props.getProperty(passwordKey);
        Assert.hasText(password, passwordKey);
        ComboPooledDataSource ds = new ComboPooledDataSource();
        ds.setDriverClass(driver);
        ds.setJdbcUrl(url);
        ds.setUser(username);
        ds.setPassword(password);
        ds.setMinPoolSize(1);
        ds.setMaxPoolSize(3);
        ds.setInitialPoolSize(1);
        ds.setMaxIdleTime(60);
        dsMap.put(tag, ds);
        return new NamedParameterJdbcTemplate(ds);
    }
	
    /**
     * @description: 获取所有 应用服务器IP-资源包 关系(DRS和CSAgent共用)
     * @return 应用服务器IP-资源包 关系
     * @2015年9月8日 下午8:21:17
     * @author: RPP
     */
    public Map<String, Set<String>> getIp2Group() {
    	
    	final Map<String, Set<String>> ip2Group = Maps.newHashMap();
    	String sql = "select ar.RES_GROUP_ID as groupid, a.SERVER_IP as ip " + 
    			"from T_APP_RES ar " +
    			"left join T_APP a on ar.APP_ID=a.APP_ID " +
    			"where a.SERVER_IP is not null " +
    			"union " +
    			"select tar.RES_GROUP_ID as groupid, b.SERVER_IP as ip " +
    			"from T_TMP_APP_RES tar " +
    			"left join T_TMP_APP b on tar.APP_ID=b.APP_ID " +
    			"where b.SERVER_IP is not null and b.PUBLISH_STATUS='0' and b.APPLY_STATUS='1'";
    	mdpJdbc.getJdbcOperations().query(sql, new RowCallbackHandler() {
    		@Override
    		public void processRow(ResultSet rs) throws SQLException {
                String groupId = rs.getString("groupId");
                String ip = rs.getString("ip");
                if (!ip2Group.containsKey(ip)) {
                    ip2Group.put(ip, Sets.<String>newHashSet());
                }
                ip2Group.get(ip).add(groupId);
            }
    	});
    	
    	return ip2Group;
    }
    
    /**
     * @description: 由所有 应用服务器IP-资源包 关系 获取所有 应用服务器IP-资源组 关系(CSAgent独用;资源组为资源包中所有资源Set集合)
     * @param ip2Group 应用服务器IP-资源包关系
     * @return 应用服务器IP-资源组 关系
     * @2015年9月8日 下午8:22:52
     * @author: RPP
     */
    public Map<String, Set<String>> getServerIp2ResMappings(Map<String, Set<String>> ip2Group) {
    	
    	Map<String, Set<String>> ip2Res = Maps.newHashMap();
    	Map<String,Object> params = Maps.newHashMap();
    	List<String> rids = Lists.newArrayList();
        String sql = "select RES_ID as rid from T_PRS_RESOURCE where RES_PKG_ID in (:resGroupId)";
        for (String ip : ip2Group.keySet()) {
            Set<String> groupSet = ip2Group.get(ip);
            params.put("resGroupId", groupSet);
            
            List<Map<String, Object>> rsList = prsJdbc.queryForList(sql, params);
            for(Map<String, Object> map : rsList) {
            	rids.add((String)map.get("rid"));
            }
            
            ip2Res.put(ip, Sets.newHashSet(rids));
            params.clear();
            groupSet.clear();
        }
        ip2Group.clear();

        return ip2Res;
    }
    
    /**
     * @description: 由所有 应用服务器IP-资源包 关系 获取所有 应用服务器IP-本地表 关系(DRS独用)
     * @param ip2Res 应用服务器IP-资源组 关系
     * @return 应用服务器IP-本地表名 关系
     * @2015年9月9日 上午9:44:18
     * @author: RPP
     */
    public Map<String, Set<String>> getServerIp2TabMappings(Map<String, Set<String>> ip2Group) {
    	
    	Map<String, Set<String>> ip2Tab = Maps.newHashMap();
    	String sql = "select RES_LOCAL_TABLE_NAME as rLocTabName from T_PRS_RESOURCE where RES_PKG_ID in (:resGroupId)";
    	
    	List<String> tabList = Lists.newArrayList();
    	Map<String,Object> params = Maps.newHashMap();
    	Set<String> groupSet = Sets.newHashSet();
    	for (String ip : ip2Group.keySet()) {
    		
            groupSet = ip2Group.get(ip);
            params.put("resGroupId", groupSet);

            List<Map<String, Object>> rsList = prsJdbc.queryForList(sql, params);
            for(Map<String, Object> map : rsList) {
            	tabList.add((String)map.get("rLocTabName"));
            }
            ip2Tab.put(ip, Sets.newHashSet(tabList));
            
            tabList.clear();
            params.clear();
            groupSet.clear();
        }
        ip2Group.clear();

        return ip2Tab;
    }
    
    /**
     * @description: 获取 所有应用-SN 关系（CSAgent、DRS共用）
     * 				type:person-人;dep-单位;police-警种;sex-性别
     * @return 所有应用-SN 关系
     * @2015年9月8日 下午7:57:08
     * @author: RPP
     */
    public Map<String, Set<String>> getAppSnMappings() {
        final Map<String, Set<String>> app2person = Maps.newHashMap();
        Map<String, Set<String>> app2sn = Maps.newHashMap();
        //mdp 应用和范围的对应关系
        final Map<String, Set<AppUserArea>> app2area = Maps.newHashMap();
        String sql = "select aua.APP_ID as appId,aua.Type as type,aua.AREA_ID as areaId from T_APP_USE_AREA aua " +
        		"union " +
        		"select taua.APP_ID as appId,taua.Type as type,taua.AREA_ID as areaId from T_TMP_APP_USE_AREA taua";
        mdpJdbc.getJdbcOperations().query(sql, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                String appId = rs.getString("appId");
                String type = rs.getString("type");
                String areaId = rs.getString("areaId");
                if ("person".equalsIgnoreCase(type)) {
                    putPerson2App(app2person, appId, Arrays.asList(areaId));
                } else {
                    AppUserArea one = new AppUserArea(appId, type, areaId);
                    if (!app2area.containsKey(appId)) {
                        app2area.put(appId, Sets.<AppUserArea>newHashSet());
                    }
                    app2area.get(appId).add(one);
                }
            }
        });

        //应用与可使用的人员之前的关联
        Map<String,Object> params = Maps.newHashMap();
        for (String app : app2area.keySet()) {
            Set<AppUserArea> conditions = app2area.get(app);
            Set<String> deptSet = getQueryDeptSet(conditions);
            Set<String> policeSet = getQueryProperty(conditions, "police");
            Set<String> sexSet = getQueryProperty(conditions, "sex");
            List<String> persons = Lists.newArrayList();
            StringBuffer tmpSql = new StringBuffer("select p.id as id from T_PERSON p where flag=0 ");
            if (!CollectionUtils.isEmpty(deptSet)) {
                tmpSql.append(" and p.dep_id in (:deptIds)");
                params.put("deptIds", deptSet);
            }
            if (!CollectionUtils.isEmpty(policeSet)) {
                tmpSql.append(" and p.police in (:police)");
                params.put("police", policeSet);
            }
            if (!CollectionUtils.isEmpty(sexSet)) {
                tmpSql.append(" and p.sex in (:sex)");
                params.put("sex", sexSet);
            }
            
            List<Map<String, Object>> rsList = pamsJdbc.queryForList(tmpSql.toString(), params);
            for(Map<String, Object> map : rsList) {
            	persons.add((String)map.get("id"));
            }
            
            putPerson2App(app2person, app, persons);
            
            params.clear();
        }

        //人员与sn之前的关联
        for (String app : app2person.keySet()) {
            Set<String> persons = app2person.get(app);

            Set<String> sns = querySns(persons);
            app2sn.put(app, sns);

            persons.clear();
        }
        app2person.clear();

        return app2sn;
    }
    
    /**
     * @description: 获取 应用-资源包 关系（CSAgent、DRS共用）
     * @return 应用-资源包 关系
     * @2015年9月9日 下午2:17:06
     * @author: RPP
     */
    public Map<String, Set<String>> getAppResGroupMappings() {
    	String sql = "select ar.RES_GROUP_ID as groupId,ar.APP_ID as app from T_APP_RES ar " +
    			"union " +
    			"select tar.RES_GROUP_ID as groupId,tar.APP_ID as app from T_TMP_APP_RES tar";
        final Map<String, Set<String>> app2Group = Maps.newHashMap();
        mdpJdbc.getJdbcOperations().query(sql, new RowCallbackHandler() {
            @Override
            public void processRow(ResultSet rs) throws SQLException {
                String groupId = rs.getString("groupId");
                String app = rs.getString("app");
                if (!app2Group.containsKey(app)) {
                	app2Group.put(app, Sets.<String>newHashSet());
                }
                app2Group.get(app).add(groupId);
            }
        });
        return app2Group;
    }
    
    /**
     * @description: 由 应用-资源包 关系 获取 应用-资源 关系(CSAgent独用)
     * @param app2Group 应用-资源包 关系
     * @return 应用-资源 关系
     * @2015年9月9日 下午2:32:27
     * @author: RPP
     */
    public Map<String, Set<String>> getAppResMappings(Map<String, Set<String>> app2Group) {
    	
    	Map<String, Set<String>> app2res = Maps.newHashMap();
    	List<String> rids = Lists.newArrayList();
    	Map<String, Object> params = Maps.newHashMap();
        String sql = "select RES_ID as rid from T_PRS_RESOURCE where RES_PKG_ID in (:resGroupIds)";
        for (String app : app2Group.keySet()) {
            Set<String> groupSet = app2Group.get(app);
            params.put("resGroupIds", groupSet);
            
            List<Map<String, Object>> rsList = prsJdbc.queryForList(sql, params);
            for(Map<String, Object> map : rsList) {
            	rids.add((String)map.get("rid"));
            }
            app2res.put(app, Sets.newHashSet(rids));

            rids.clear();
            params.clear();
            groupSet.clear();
        }
        app2Group.clear();
        return app2res;
    }
    
    /**
     * @description: 由 应用-资源包 关系 获取 应用-本地表 关系(DRS独用)
     * @param app2Group 应用-资源包 关系
     * @return 应用-本地表关系
     * @2015年9月9日 下午2:33:29
     * @author: RPP
     */
    public Map<String, Set<String>> getAppTabMappings(Map<String, Set<String>> app2Group) {
    	
    	Map<String, Set<String>> app2res = Maps.newHashMap();
        String sql = "select RES_LOCAL_TABLE_NAME as rLocTabName from T_PRS_RESOURCE where RES_PKG_ID in (:resGroupIds)";
        
        Map<String, Object> params = Maps.newHashMap();
        List<String> tabList = Lists.newArrayList();
        for (String app : app2Group.keySet()) {
            Set<String> groupSet = app2Group.get(app);
            params.put("resGroupIds", groupSet);

            List<Map<String, Object>> rsList = prsJdbc.queryForList(sql, params);
            for(Map<String, Object> map : rsList) {
            	tabList.add((String)map.get("rLocTabName"));
            }
            app2res.put(app, Sets.newHashSet(tabList));

            tabList.clear();
            groupSet.clear();
        }
        app2Group.clear();
        return app2res;
    }
    
    /**
     * 使用in的时候每100个查一次，防止db出错
     *
     * @param persons
     * @return
     */
    private Set<String> querySns(Set<String> persons) {
        Set<String> rst = Sets.newHashSet();
        String tmpSql = "select d.sn as sn from T_DEVICE d where d.state=3 and d.person_id in (:personIds)";
        Map<String, Object> params = Maps.newHashMap();
        params.put("personIds", persons);
        List<Map<String, Object>> rsList = pamsJdbc.queryForList(tmpSql, params);
        for(Map<String, Object> map : rsList) {
        	rst.add((String)map.get("sn"));
        }
        return rst;
    }

    private Set<String> getQueryProperty(Set<AppUserArea> conditions, String type) {
    	
    	Set<String> conditionIds = Sets.newHashSet();
        for (AppUserArea one : conditions) {
            if (type.equalsIgnoreCase(one.getType())) {
                conditionIds.add(one.getAreaId());
            }
        }
        return conditionIds;
    }

    @SuppressWarnings("unchecked")
	private Set<String> getQueryDeptSet(Set<AppUserArea> conditions) {
        Set<String> depIds = getQueryProperty(conditions, "dep");
        Set<String> depIdSet = Sets.newHashSet();
        for(String depId : depIds) {
        	if(StringUtils.hasText(depId)) {
        		List<String> ids = Lists.newArrayList();
            	String sql = "select id from t_department start with id =:id connect by prior parent_id=id AND flag=0";
            	Map<String, Object> params = Maps.newHashMap();
            	params.put("id", depId);
            	
            	List<Map<String, Object>> rsList = pamsJdbc.queryForList(sql, params);
            	for(Map<String, Object> map : rsList) {
            		ids.add((String)map.get("id"));
            	}
            	depIdSet.addAll(Sets.newHashSet(ids));
        	}
        }
        return depIdSet;
    }
    
    private void putPerson2App(Map<String, Set<String>> app2person, String appId, Collection<String> persons) {
        if (!app2person.containsKey(appId)) {
            app2person.put(appId, Sets.<String>newHashSet());
        }
        app2person.get(appId).addAll(persons);
    }
    
    public void closeDataSource() {
    	for(String tag : dsMap.keySet()) {
    		dsMap.get(tag).close();
    	}
    }
}
