package com.xdja.prs.authentication.gateway;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.Map;

/**
 * Project Name：prs-authentication<br/>
 * ClassName： com.xdja.prs.authentication.gateway.GatewayProcess<br/>
 * Description：<br/>
 *
 * @author: 黄地
 * @date: 2015/08/24 14:41
 * note:
 */
public class GatewayProcess implements UdpCallback {
    private Logger logger = LoggerFactory.getLogger(getClass());
    private Map<String, String> ipToSnMappings = Maps.newConcurrentMap();
    private Map<String, Map<Integer, String>> gatewayPortToSnMappings = Maps.newConcurrentMap();
    private List<String> gatewayIpList = Lists.newArrayList();
    private UdpServer udpServer;

    /**
     * 方法描述：启动网关交互模块
     *
     * @param gatewayIpList
     * @param gatewayPortList
     * @param gatewayReceivePort @throws Exception
     * @author: 黄地
     * @date: 2015-08-24 14:58:53
     */
    public void startup(List<String> gatewayIpList, List<Integer> gatewayPortList, int gatewayReceivePort) throws Exception {
        this.gatewayIpList = Lists.newArrayList(gatewayIpList);
        logger.debug("gateway ip list : {}", this.gatewayIpList);
        udpServer = new UdpServer(gatewayReceivePort, this);
        udpServer.startOnlineInfoListenServer();
        udpServer.startGetCurrentOnlineInfo(gatewayIpList, gatewayPortList);
    }

    /**
     * 方法描述：检查指定ip是否是网关ip
     *
     * @param ip
     * @return
     * @author: 黄地
     * @date: 2015-08-24 15:13:03
     */
    public boolean isGatewayIp(String ip) {
        return gatewayIpList.contains(ip);
    }

    /**
     * 方法描述：通过ip获取sn
     *
     * @param ip
     * @return
     * @author: 黄地
     * @date: 2015-08-24 15:15:03
     */
    public String getSnByIp(String ip) {
        logger.debug("隧道模式，尝试通过ip获取sn");
        if (logger.isDebugEnabled()) {
            logger.debug("当前所有的隧道IP列表:{}", ipToSnMappings.keySet());
        }
        return ipToSnMappings.get(ip);
    }

    /**
     * 方法描述：通过端口号获取sn
     *
     * @param port
     * @return
     * @author: 黄地
     * @date: 2015-08-24 15:15:18
     */
    public String getSnByPort(String ip, int port) {
        logger.debug("转发模式，尝试通过端口获取sn");
        if (gatewayPortToSnMappings.containsKey(ip)) {
            Map<Integer, String> map = gatewayPortToSnMappings.get(ip);
            if (logger.isDebugEnabled()) {
                logger.debug("IP:{}对应网关已使用端口列表:{}", ip, map.keySet());
            }
            return map.get(port);
        } else {
            logger.debug("IP:{}不在网关列表中", ip);
        }
        return null;
    }

    public void allOffline(String gatewayIp) {
        Map<Integer, String> portToSn = gatewayPortToSnMappings.remove(gatewayIp);
        if (portToSn != null) {
            portToSn.clear();
        }

    }


    public void offline(String gatewayIp, String ip, Integer forwardPort, String sn) {
        logger.debug("收到下线通知. gateway:{},ip:{},port:{},sn:{}", gatewayIp, ip, forwardPort, sn);
        Assert.hasText(sn, "sn empty!");
        if (StringUtils.hasText(ip)) {
            ipToSnMappings.remove(ip);
        } else if (forwardPort != null) {
            if (gatewayPortToSnMappings.containsKey(gatewayIp)) {
                Map<Integer, String> portToSn = gatewayPortToSnMappings.get(gatewayIp);
                portToSn.remove(forwardPort);
            }

        }
    }


    public void online(String gatewayIp, String ip, Integer forwardPort, String sn) {
        logger.debug("收到上线通知. gateway:{},ip:{},port:{},sn:{}", gatewayIp, ip, forwardPort, sn);
        Assert.hasText(sn, "sn empty!");
        if (StringUtils.hasText(ip)) {
            ipToSnMappings.put(ip, sn);
        } else if (forwardPort != null) {
            if (!gatewayPortToSnMappings.containsKey(gatewayIp)) {
                gatewayPortToSnMappings.put(gatewayIp, Maps.<Integer, String>newHashMap());
            }
            gatewayPortToSnMappings.get(gatewayIp).put(forwardPort, sn);
        }
    }


    /**
     * 停止网关交互模块
     *
     * @throws Exception
     */
    public void shutdown() throws Exception {
        udpServer.shutdown();
    }


}
