/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package com.xdja.prs.authentication.sync;

import io.netty.bootstrap.ServerBootstrap;
import io.netty.channel.*;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.handler.codec.http.*;
import io.netty.util.concurrent.DefaultThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URL;
import java.util.HashMap;
import java.util.Map;

/**
 * Embedded HTTP/1.1 file server based on a classic (blocking) I/O model.
 */
public class CustomHttpServer {
    private Logger logger = LoggerFactory.getLogger(getClass());
    private Map<String, MyRequestHandler> requestHandlerMap = new HashMap<String, MyRequestHandler>();
    private ServerBootstrap serverBootstrap;
    private Channel serverChannel;

    public CustomHttpServer registerHandler(String path, MyRequestHandler requestHandler) {
        requestHandlerMap.put(path, requestHandler);
        return this;
    }

    public void start(int port) throws Exception {
        EventLoopGroup bossGroup = new NioEventLoopGroup(0, new DefaultThreadFactory("authentication-sync-Boss"));
        EventLoopGroup workerGroup = new NioEventLoopGroup(0, new DefaultThreadFactory("authentication-sync-Worker"));
        serverBootstrap = new ServerBootstrap();
        serverBootstrap.group(bossGroup, workerGroup)
                .channel(NioServerSocketChannel.class)
//                .handler(new LoggingHandler(LogLevel.INFO))
                .option(ChannelOption.TCP_NODELAY, true)
                .option(ChannelOption.SO_BACKLOG, 4096)
                .childHandler(new ChannelInitializer<SocketChannel>() {
                    @Override
                    protected void initChannel(SocketChannel ch) throws Exception {
//                        ch.pipeline().addLast(new LoggingHandler(LogLevel.INFO));
                        ch.pipeline().addLast(new HttpServerCodec());
                        ch.pipeline().addLast(new HttpObjectAggregator(Integer.MAX_VALUE));
                        ch.pipeline().addLast(new InnerServerHandler());
                    }
                });
        serverChannel = serverBootstrap.bind(port).sync().channel();
    }

    public void stop() throws InterruptedException {
        requestHandlerMap.clear();
        if (serverChannel != null) {
            serverChannel.close().sync();
        }
        if (serverBootstrap != null) {
            serverBootstrap.group().shutdownGracefully();
            serverBootstrap.childGroup().shutdownGracefully();
        }
    }

    public void awaitTermination() throws InterruptedException {
        serverChannel.closeFuture().sync();
    }

    private class InnerServerHandler extends SimpleChannelInboundHandler<FullHttpMessage> {
        @Override
        protected void channelRead0(ChannelHandlerContext channelHandlerContext, FullHttpMessage fullHttpMessage) throws Exception {
            if (fullHttpMessage instanceof FullHttpRequest) {
                FullHttpRequest request = (FullHttpRequest) fullHttpMessage;
                String uri = request.getUri();
                try {
                    uri = new URL(uri).getPath();
                } catch (Exception e) {
                }
                logger.debug("receive request uri is {}", uri);
                for (String one : requestHandlerMap.keySet()) {
                    if (uri.startsWith(one)) {
                        requestHandlerMap.get(one).handle(request, channelHandlerContext);
                        return;
                    }
                }
                DefaultFullHttpResponse response = new DefaultFullHttpResponse(HttpVersion.HTTP_1_1, HttpResponseStatus.NOT_FOUND);
                channelHandlerContext.writeAndFlush(response).addListener(ChannelFutureListener.CLOSE);
            }
        }
    }
}