package com.xdja.api;

import android.content.Context;
import android.util.Log;
import android.util.SparseArray;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.cryptodev.CryptoDevInfo;
import com.xdja.cryptodev.CryptoDevManager;
import com.xdja.cryptodev.driver.chipmanager.ChipManagerDriver;
import com.xdja.sslvpn.SSLVPNErrorCode;
import com.xdja.sslvpn.sslvpnJNI;
import com.xdja.sslvpn.sslvpn_acl_version;
import com.xdja.sslvpn.sslvpn_callback;
import com.xdja.sslvpn.sslvpn_config;
import com.xdja.sslvpn.sslvpn_device_config;
import com.xdja.sslvpn.sslvpn_device_type;
import com.xdja.sslvpn.sslvpn_log_decor;
import com.xdja.sslvpn.sslvpn_logging_config;
import com.xdja.sslvpn.sslvpn_session_config;
import com.xdja.sslvpn.sslvpn_session_mode;
import com.xdja.sslvpn.sslvpn_transport_config;
import com.xdja.sslvpn.sslvpn_transport_protocol;
import com.xdja.timer.TimerWrapper;
import com.xdja.timer.timerJNI;
import com.xdja.timer.timer_callback;
import com.xdja.tun.TunProxy;

import java.io.File;
import java.util.HashMap;
import java.util.List;

/**
 * Created by xingjianqiang on 2019/1/30.
 */

public class SSLVPNApi {
    private static final String THIS_FILE = "sslvpnapi";
    private SSLVPNApi(){}
    private static SSLVPNApi sslvpnApi = new SSLVPNApi();
    public static SSLVPNApi getInstance() {
        return sslvpnApi;
    }

    private timer_callback timercb = new timer_callback();

    private static final int[] ROLE_ARRAY = new int[]{JNIAPI.ROLE_A, JNIAPI.ROLE_A, JNIAPI.ROLE_A, JNIAPI.ROLE_A, JNIAPI.ROLE_D,
                                         JNIAPI.ROLE_A, JNIAPI.ROLE_D, JNIAPI.ROLE_A, JNIAPI.ROLE_A, JNIAPI.ROLE_A};


    private HashMap<Integer, SessionConfig> sessionMap = new HashMap<>();

    HashMap<Integer, SessionConfig> getSessionMap() {
        return sessionMap;
    }

    /**
     * 删除该目录下时间点之前的文件
     * @param path
     * @param miliseconds
     */
    private static void deleteFilesBefore(String path, long miliseconds) {
        File file = new File(path);
        if (!file.exists()) {
            Log.e(THIS_FILE, "deleteFilesBefore: Directory not exists, please check");
            return;
        }
        //获得文件里面所有的文件及文件夹
        File[] files = file.listFiles();
        if(files==null || files.length<= 0)return;
        //遍历files里面的所有文件及文件夹
        for (File f : files) {
            //获得绝对路径下的文件及文件夹
            File absFile = f.getAbsoluteFile();
            long currTime = System.currentTimeMillis();   //当前时间
            long lastTime = absFile.lastModified();     //文件被最后一次修改的时间
            //时间差
            long diffen = currTime - lastTime;
            if (diffen >= miliseconds) {
                absFile.delete();
                if (absFile.isDirectory()) {
                    absFile.delete();
                }
            } else { // 删除size为0的日志文件
                if (absFile.length() == 0) {
                    absFile.delete();
                }
            }
        }
    }

    private void cleanLogFiles(String rootPath) {
        String path = rootPath + File.separator + "sslvpn" + File.separator;
        // 删除两天前的日志文件
        deleteFilesBefore(path, 2 * 24 * 60 * 60 * 1000);
    }

    /**
     * 初始化sslvpn模块
     * @param context 上下文
     * @param config 全局配置
     * @param callback 状态回调
     * @return 0 成功，其他失败
     */
    public int init(Context context, SSLVPNConfig config, SSLVPNCallback callback) {
        int status = -1;

        sessionMap.clear();
        cleanLogFiles(config.getLogRootPath());
        CryptoDevManager.getInstance().setContext(context);
        CryptoDevManager.getInstance().registerDriver(new ChipManagerDriver());
        boolean isScanning = true;
        while (isScanning) {
            try {
                Thread.sleep(10);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
            isScanning = CryptoDevManager.getInstance().isScanning();
        }

        List<CryptoDevInfo> devInfoList = CryptoDevManager.getInstance().getCryptoDevInfos();
        Log.d(THIS_FILE, "dev info list" + devInfoList.toString());


        TimerWrapper.create(context);
        TunProxy.setAppContext(context);
        timerJNI.timer_init(timercb);
        sslvpnJNI.sslvpn_create();

        sslvpn_config vpnconfig = new sslvpn_config();
        vpnconfig.setMax_session_cnt(4);
        vpnconfig.setWorker_thread_cnt(1);
        vpnconfig.setWakeup(config.isWakeupSystem()?1:0);

        sslvpn_logging_config logging_config = new sslvpn_logging_config();
        logging_config.setLogging_message(true);
        logging_config.setDecor(sslvpn_log_decor.PJ_LOG_HAS_SENDER.getValue() | sslvpn_log_decor.PJ_LOG_HAS_TIME.getValue() |
                sslvpn_log_decor.PJ_LOG_HAS_MICRO_SEC.getValue() | sslvpn_log_decor.PJ_LOG_HAS_NEWLINE.getValue() |
                sslvpn_log_decor.PJ_LOG_HAS_SPACE.getValue() | sslvpn_log_decor.PJ_LOG_HAS_THREAD_SWC.getValue() |
                sslvpn_log_decor.PJ_LOG_HAS_INDENT.getValue());
        logging_config.setLevel(sslvpn_logging_config.LEVEL_ALL);
        // create log directory
        String path = config.getLogRootPath() + File.separator + "sslvpn" + File.separator;
        File logPath = new File(path);
        boolean result = true;
        if (!logPath.exists()) {
            result = logPath.mkdir();
        }

        if (!result) {
            Log.e(THIS_FILE, "Create dir return false");
        }
        logging_config.setLog_filepath(path + File.separator);

        status = sslvpnJNI.sslvpn_init(vpnconfig, logging_config, callback);
        return status;
    }


    /**
     * 创建session
     * @param sessionConfig session配置
     * @return session id，用于后面调用连接和停止
     */
    public int sessionCreate(SessionConfig sessionConfig) {
        int[] sid = new int[1];
        sslvpn_session_config session_config = new sslvpn_session_config();
        session_config.setAcl_version(sslvpn_acl_version.SSLVPN_ACL_VERSION_V30.getValue());
        if (sessionConfig.getMode() == SessionConfig.MODE_TUN) {
            session_config.setTransport_mode(sslvpn_session_mode.SSLVPN_TRANSPORT_MODE_TUN.getValue());
        } else {
            session_config.setTransport_mode(sslvpn_session_mode.SSLVPN_TRANSPORT_MODE_FORWARD.getValue());
        }
        sslvpn_transport_config transport_config = new sslvpn_transport_config();
        transport_config.setCa_list(sessionConfig.getCa_list());
        if (sessionConfig.getProtocol() == SessionConfig.PROTOCOL_TCP) {
            transport_config.setProtocol(sslvpn_transport_protocol.SSLVPN_TRANSPORT_PROTOCOL_TCP.getValue());
            transport_config.setCipher_suit("TLS_RSA_WITH_SM4_CBC_SHA;TLS_RSA_WITH_SM4_ECB_SHA;TLS_SM2_WITH_SM4_ECB_SM3;TLS_SM2_WITH_SM4_CBC_SM3");

        } else {
            transport_config.setProtocol(sslvpn_transport_protocol.SSLVPN_TRANSPORT_PROTOCOL_UDP.getValue());
            transport_config.setCipher_suit("GM_RSA_SM4_SHA1;GM_RSA_SM4_SM3;GM_ECC_SM4_SM3");
        }

        sslvpn_device_config[] device_config = new sslvpn_device_config[3];
        // 贴膜卡
        device_config[0] = new sslvpn_device_config();
        device_config[0].setDeviceInfo(sslvpn_device_type.SSLVPN_DEVICE_COVER.getValue(),
                sessionConfig.getCertType() == SessionConfig.CERT_SIGN?2:3,
                String.valueOf(sessionConfig.getContainer()));
        device_config[0].setRoleId(ROLE_ARRAY[sessionConfig.getContainer()]);

        // tf卡
        device_config[1] = new sslvpn_device_config();
        device_config[1].setDeviceInfo(sslvpn_device_type.SSLVPN_DEVICE_TF.getValue(),
                sessionConfig.getCertType() == SessionConfig.CERT_SIGN?2:3,
                String.valueOf(sessionConfig.getContainer()));
        device_config[1].setRoleId(ROLE_ARRAY[sessionConfig.getContainer()]);

        // 芯片
        device_config[2] = new sslvpn_device_config();
        device_config[2].setDeviceInfo(sslvpn_device_type.SSLVPN_DEVICE_CHIP.getValue(),
                sessionConfig.getCertType() == SessionConfig.CERT_SIGN?2:3,
                String.valueOf(sessionConfig.getContainer()));
        device_config[2].setRoleId(ROLE_ARRAY[sessionConfig.getContainer()]);

        int status = sslvpnJNI.sslvpn_session_create(session_config, transport_config, device_config, sid);

        if (status == 0) {
            // create success, return sid
            sslvpnJNI.sslvpn_set_sessionIpList(sid[0], sessionConfig.getServerList());
            sessionMap.put(sid[0], sessionConfig);
            return sid[0];
        }

        Log.e(THIS_FILE, "Create session error. return " + status);
        return status;
    }

    /**
     * session连接
     * @param sessionId id用于标识session
     */
    public void sessionConnect(int sessionId) {
        int status = sslvpnJNI.sslvpn_session_connect(sessionId);
        Log.d(THIS_FILE, "Connect session " + sessionId + " return " + status);
    }

    /**
     * session停止
     * @param sessionId id用于标识session
     */
    public void sessionStop(int sessionId) {
        int status = sslvpnJNI.sslvpn_session_stop(sessionId, SSLVPNErrorCode.SSLVPN_USER_CANCELED);
        Log.d(THIS_FILE, "Stop session " + sessionId + " return " + status);
    }


    /**
     * sslvpn模块销毁，会停止所有session
     */
    public void destroy() {
        sslvpnJNI.sslvpn_destroy();
        timerJNI.timer_destroy();
        CryptoDevManager.getInstance().unregisterDriver(ChipManagerDriver.DRIVER_NAME);
    }

}
