package com.xdja.api;

import android.util.Log;

import com.xdja.sslvpn.sslvpn_cert;
import com.xdja.sslvpn.sslvpn_serverip_config;
import com.xdja.util.CertUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.transform.sax.TemplatesHandler;

/**
 * Created by xingjianqiang on 2019/1/31.
 */

/**
 * session配置，用于创建session时传入
 */
public class SessionConfig {

    private static final String THIS_FILE = "SessionConfig";
    public static final int PROTOCOL_TCP = 0;
    public static final int PROTOCOL_UDP = 1;
    public static final int CERT_SIGN = 0;
    public static final int CERT_EXCH = 1;
    public static final int MODE_TUN = 0;
    public static final int MODE_FORWARD = 1;

    private List<sslvpn_cert> ca_list = new ArrayList<>();
    private int mMode = MODE_TUN;
    private int mProtocol = PROTOCOL_TCP;
    private int mContainer = 0;
    private int mCertType = CERT_EXCH;
    List<sslvpn_serverip_config> serverip_configs = new ArrayList<>();
    /**
     * 设置CA（如果有CA链，需要添加所有子CA）
     * @param caBuf ca数据
     * @param length ca长度
     */
    public void addCA(byte[] caBuf, int length) {
        sslvpn_cert cert = sslvpn_cert.create_sslvpn_cert();
        cert.setCert_len(length);
        cert.setCert_buf(caBuf);
        int certType = CertUtil.parseCertType(caBuf, length);
        if (certType == CertUtil.CERT_TYPE_RSA) {
            cert.setPubkey_bits(CertUtil.getRsaPubkeyBits(caBuf, length));
        } else {
            cert.setPubkey_bits(256);
        }
        ca_list.add(cert);
    }

    /**
     * 设置CA（如果有CA链，需要添加所有子CA）
     * @param file CA文件
     */
    public void addCA(File file) {
        if (!file.exists()) {
            Log.e(THIS_FILE, "CA file " + file.getAbsolutePath() + " not exists!");
            return;
        }

        long length = file.length();
        byte[] buf = new byte[(int) length];
        FileInputStream fileInputStream = null;
        try {
            int readLen = 0;
            fileInputStream = new FileInputStream(file);
            readLen = fileInputStream.read(buf);

            if (readLen > 0) {
                addCA(buf, readLen);
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (fileInputStream != null) {
                try {
                    fileInputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    /**
     * 设置CA（如果有CA链，需要添加所有子CA）
     * @param filePath CA文件路径
     */
    public void addCA(String filePath) {
        File file = new File(filePath);
        if (file.exists()) {
            addCA(file);
        } else {
            Log.e(THIS_FILE, "CA file " + filePath + " not exists!");
        }
    }

    public List<sslvpn_cert> getCa_list() {
        return ca_list;
    }

    public int getMode() {
        return mMode;
    }

    public int getProtocol() {
        return mProtocol;
    }

    public int getContainer() {
        return mContainer;
    }

    public int getCertType() {
        return mCertType;
    }

    /**
     * 设置通道类型（隧道还是转发）
     * @param mode 类型（默认为隧道)
     * @see SessionConfig#MODE_TUN
     * @see SessionConfig#MODE_FORWARD
     */
    public void setMode(int mode) {
        this.mMode = mode;
    }
    /**
     * 设置传输协议tcp or udp，默认值为tcp
     * @param protocol 协议类型
     *                 @see SessionConfig#PROTOCOL_TCP
     *                 @see SessionConfig#PROTOCOL_UDP
     */
    public void setProtocol(int protocol) {
        this.mProtocol = protocol;
    }

    /**
     * 设置容器号和证书类型（交换还是签名）
     * @param container 容器号（默认为0号）
     * @param certType 证书类型（默认为交换）
     * @see SessionConfig#CERT_SIGN
     * @see SessionConfig#CERT_EXCH
     */
    public void setCertPosition(int container, int certType) {
        this.mContainer = container;
        this.mCertType = certType;
    }

    /**
     * 配置网关ip和端口，支持调用多次添加多个服务器做备用地址
     * @param ip 地址
     * @param port 端口
     */
    public void addServer(String ip, int port) {
        sslvpn_serverip_config config = new sslvpn_serverip_config();
        config.setGateway_ip(ip);
        config.setGateway_port(port);
        serverip_configs.add(config);
    }

    public List<sslvpn_serverip_config> getServerList() {
        return serverip_configs;
    }


}
