package com.xdja.key;

import android.bluetooth.BluetoothAdapter;
import android.content.Context;
import android.inputmethodservice.Keyboard;
import android.os.Build;
import android.os.CpuUsageInfo;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;
import android.text.TextUtils;

import com.xdja.SafeKey.XDJA_SM2_PARAM;
import com.xdja.key.covercard.CoverCardStateChangedListener;
import com.xdja.key.covercard.CoverCardWrapper;
import com.xdja.key.koal.KoalCfg;
import com.xdja.key.longmai.LongmaiCfg;
import com.xdja.key.xdjakey.XdjaKeyCfg;
import com.xdja.key.zhongfu.ZhongfuCfg;
import com.xdja.safeclient.Function;
import com.xdja.safeclient.MyApplication;
import com.xdja.safeclient.R;
import com.xdja.safeclient.event.EventManager;
import com.xdja.safeclient.event.SDCardStateChangeListener;
import com.xdja.safeclient.utils.Log;
import com.xdja.safeclient.utils.ToastUtil;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by xingjianqiang on 2017/8/19.
 * Project : safeclient.android.origin
 * Email : xingjianqiang@xdja.com
 */

public class KeyWrapper implements SDCardStateChangeListener {
    public static final int XDJA_KEY = 1<<0;
    public static final int ZHONGFU_KEY = 1<<1;
    public static final int COVER_KEY = 1<<2;
    public static final int CHIP_MANAGER_KEY = 1<<3;
    public static final int CONTENT_PROVIDER_KEY = 1<<4;
    public static final int LONGMAI_KEY = 1<<5;
    public static final int SANWEI_KEY = 1<<6;
    public static final int KOAL_KEY = 1<<7;

    public static final int KEY_TYPE_CHIP = 0;
    public static final int KEY_TYPE_TF = 1;
    public static final int KEY_TYPE_USBKEY = 2;
    public static final int KEY_TYPE_SIM = 3;
    public static final int KEY_TYPE_SOFT = 4;
    public static final int KEY_TYPE_BLUETOOTH = 5;

    public static final String COVER_KEY_DRIVER_NAME = "cover_card";
    public static final String XDJA_KEY_DRIVER_NAME = "xdja_key";
    public static final String CHIP_MANAGER_DRIVER_NAME = "chip_manager";
    public static final String PROVIDER_DRIVER_NAME = "content_provider";
    public static final String LONGMAI_DRIVER_NAME = "longmai_key";
    public static final String SANWEI_DRIVER_NAME = "sanwei_key";
    public static final String KOAL_DRIVER_NAME = "koal_key";

    private final static String THIS_FILE = "KeyWrapper";

    public static int currentIndex = -1;
    public static String currentKeyTypeComment = null;

    public static long currentKey = 0;

    private int currentKeyFlag = 0;

    private int state = State.NULL;

    public int getState() {
        return state;
    }

    private ArrayList<KeyChangedListener> keyChangedListenerArrayList = new ArrayList<>();


    public synchronized void registerKeyListener(KeyChangedListener listener) {
        if (!keyChangedListenerArrayList.contains(listener)) {
            keyChangedListenerArrayList.add(listener);
        }
    }

    public synchronized void unregisterKeyListener(KeyChangedListener listener) {
        keyChangedListenerArrayList.remove(listener);
    }

    public synchronized void notifyKeyAdded(int type, String cardId) {
        if (keyChangedListenerArrayList.isEmpty()) {
            return;
        }

        for (KeyChangedListener listener:keyChangedListenerArrayList) {
            listener.onKeyAdded(type, cardId);
        }
    }

    public synchronized void notifyKeyRemoved(int type, String cardId) {
        if (keyChangedListenerArrayList.isEmpty()) {
            return;
        }

        for (KeyChangedListener listener:keyChangedListenerArrayList) {
            listener.onKeyRemoved(type, cardId);
        }
    }

    public boolean isKeyError() {
        return state == State.INIT_KEY_FAILED ||state == State.NO_KEY;
    }
    public void setState(int state) {
        if (state != State.INITIALIZING && state != State.READY) {
            if (hasPendingReinit) {
                hasPendingReinit = false;
                return;
            }
        }
        this.state = state;
        for(KeyModuleStateListener listener:listeners) {
            listener.onStateChanged(state);
        }
    }


    public String buildTypeComment(String driver, int type) {
        StringBuffer sb = new StringBuffer();

        switch (driver) {
            case COVER_KEY_DRIVER_NAME:
                sb.append(context.getString(R.string.xdja_coverkey));
                break;
            case XDJA_KEY_DRIVER_NAME:
                if (type == KEY_TYPE_TF)
                    sb.append(context.getString(R.string.xdja_tf));
                else if (type == KEY_TYPE_CHIP) {
                    sb.append(context.getString(R.string.xdja_chip));
                }
                break;
            case CHIP_MANAGER_DRIVER_NAME:
//                sb.append(context.getString(R.string.chip_manager));
                if (type == KEY_TYPE_TF)
                    sb.append(context.getString(R.string.xdja_tf));
                else if (type == KEY_TYPE_CHIP) {
                    sb.append(context.getString(R.string.xdja_chip));
                } else if (type == KEY_TYPE_SIM) {
                    sb.append(context.getString(R.string.xdja_coverkey));
                } else {
                    sb.append(context.getString(R.string.chip_manager));
                }
                break;
            case PROVIDER_DRIVER_NAME:
                sb.append(context.getString(R.string.forward_client));
                break;
            case LONGMAI_DRIVER_NAME:
                sb.append(context.getString(R.string.longmai_card));
                break;
            case KOAL_DRIVER_NAME:
                sb.append(context.getString(R.string.koal_card));
                break;
            default:
                sb.append(context.getString(R.string.unkown_device));
                break;
        }
        return sb.toString();
    }

    public static class StatusCode {
        public static final int SUCCESS = 0;
        public static final int NO_VALID_KEY = 1000; // 无可用的卡
        public static final int NOT_INITIALIZED = 1001; // 卡模块未初始化
        public static final int INIT_FAILED = 1002; // 初始化卡失败
        public static final int INIT_IN_PROGRESS = 1003; // 初始化中


        public static String getComment(int code) {
            String comment = "";

            switch (code) {
                case SUCCESS:
                    comment = "成功";
                    break;
                case NO_VALID_KEY:
                    comment = "无可用的加密卡";
                    break;
                case NOT_INITIALIZED:
                    comment = "卡模块未初始化";
                    break;
                case INIT_FAILED:
                    comment = "初始化卡模块失败";
                    break;
                case INIT_IN_PROGRESS:
                    comment = "正在检测系统中的加密设备...";
                    break;
                default:
                    comment = KeyWrapper.getInstance().keyGetError(code);
                    break;
            }

            return comment;
        }
    }

    private boolean hasPendingReinit = false;
    @Override
    public void onSDCardStateChanged(int sdstate) {
        int ret;

        Log.e(THIS_FILE, "Detect sd card state changed, state = " + sdstate + " current key system state " + this.state);
        if (sdstate == SDCardStateChangeListener.SDCARD_MOUNTED) {
            switch (this.state) {
                case State.INITIALIZING:
                    // check if tf is now available
                    if ((currentKeyFlag & XDJA_KEY) == XDJA_KEY) {
                        if (!TextUtils.isEmpty(MyApplication.myApplication.getDevPath()) || Function.hasXdjaChip()) {

                            // stop cover key detecting
                            if ((currentKeyFlag & COVER_KEY) == COVER_KEY) {
                                CoverCardWrapper.getInstance().destroy();
                            }

                            hasPendingReinit = true;
                            Function.runOnMainThread(new Runnable() {
                                @Override
                                public void run() {
                                    sysDestroy();
                                    Function.initKeyModule(true);
                                }
                            });
                        }
                    }
                    break;
                default:
                    Function.runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            sysDestroy();
                            Function.initKeyModule(true);
                        }
                    });
                    break;
            }
        } else {
            if (currentIndex != -1) {
                KeyDevInfo keyDevInfo = new KeyDevInfo();
                ret = KeyWrapper.getInstance().getDevInfo(currentIndex, keyDevInfo);

                if (ret != 0) {
                    Log.e(THIS_FILE, "Get key dev info failed. ret " + ret);
                    return;
                }

                if (keyDevInfo.getType() == KEY_TYPE_TF && keyDevInfo.getDriver().equals(XDJA_KEY_DRIVER_NAME)) {
                    if (sdstate == SDCardStateChangeListener.SDCARD_MOUNTED) {
                        refreshDevice();
                    }
                }

            }
        }
    }


    public static class State {
        public static final int NULL = 0; //当前未启用
        public static final int INITIALIZING = 1; // 初始化中
        public static final int READY = 2; // 当前可用
        public static final int NO_KEY = 3; // 没有可用的加密设备
        public static final int INIT_KEY_FAILED = 4; // 初始化加密卡失败

        public static String getComment(int state) {
            switch (state) {
                case NULL:
                    return "NULL";
                case INITIALIZING:
                    return "INITIALIZING";
                case READY:
                    return "READY";
                case NO_KEY:
                    return "NO KEY";
                case INIT_KEY_FAILED:
                    return "INIT_KEY_FAILED";
                default:
                    return "UNKONW STATE";
            }
        }
    }

    static {
        System.loadLibrary("sm2");
        System.loadLibrary("key_mgr");
        System.loadLibrary("safekey");
        System.loadLibrary("key_xdja");

        System.loadLibrary("SM4");
        System.loadLibrary("SimKeyMCmdApi");
        System.loadLibrary("mSmart_SimKey");
        System.loadLibrary("mSmartAPI");
        System.loadLibrary("SymAlgSoftImp");
        System.loadLibrary("key_zhongfu");

        System.loadLibrary("key_covercard");

        System.loadLibrary("key_provider");


        System.loadLibrary("key_longmai");

        System.loadLibrary("swskf");
        System.loadLibrary("key_sanwei");

        System.loadLibrary("key_chipmanager");

        System.loadLibrary("key_koal");

        System.loadLibrary("key_wrapper");
        System.loadLibrary("key_jni");
    }


    private KeySelector keySelector = new KeySelector() {
        @Override
        public int select(ArrayList<KeyDevInfo> keyDevInfoArrayList) {
            if (keyDevInfoArrayList.size() > 0) {
                Log.d(THIS_FILE, "Default selector select first device.");
                return 0;
            }
            Log.e(THIS_FILE, "No key deivce in system.");
            return -1;
        }
    };

    private KeyWrapper(){
        handlerThread.start();
        myHandler = new MyHandler(handlerThread.getLooper());
    }

    class MyHandler extends Handler{
        public MyHandler(Looper looper){
            super(looper);
        }
        @Override
        public void handleMessage(Message msg) {
            switch (msg.what) {
                case INIT_DEVICE_AUTO:
                    if (MyApplication.myApplication.propertiesConfig.hasXdjaKey() == 1) {
                        EventManager.getInstance().removeSDCardStateChangedListener(instance);
                        EventManager.getInstance().addSDCardStateChangedListener(instance);
                    }
                    initKeySystem();
                    break;
            }
        }
    };

    private List<KeyModuleStateListener> listeners = new ArrayList<>();

    public boolean addKeyModuleStateListener(KeyModuleStateListener listener) {
        return listeners.add(listener);
    }

    public boolean removeKeyModuleStateListener(KeyModuleStateListener listener) {
        return listeners.remove(listener);
    }

    private int initKeySystem() {
        int ret;
        setState(State.INITIALIZING);

        ret = sysInit(currentKeyFlag);
        if (ret != 0) {
            Log.e(THIS_FILE, "Key sys init error. ret " + ret);
            setState(State.INIT_KEY_FAILED);
            return ret;
        }

        ArrayList<KeyDevInfo> keyList = new ArrayList<>();
        int devCnt = KeyWrapper.getInstance().getDevCount();
        Log.d(THIS_FILE, "Total " + devCnt + " device in system.");

        KeyDevInfo keyDevInfo;
        if (devCnt > 0) {
            for(int i=0;i<devCnt;i++) {
                keyDevInfo = new KeyDevInfo();
                ret = KeyWrapper.getInstance().getDevInfo(i, keyDevInfo);
                if (ret != 0){
                    Log.e(THIS_FILE, "Get device " + i + " info failed.");
                    break;
                }
                Log.d(THIS_FILE, "Device " + i + " : " + keyDevInfo);
                keyList.add(i, keyDevInfo);
            }
        }

        currentIndex = getKeySelector().select(keyList);


        // 当前卡类型描述信息
        keyDevInfo = new KeyDevInfo();
        ret = KeyWrapper.getInstance().getDevInfo(currentIndex, keyDevInfo);
        if (ret == 0) {
            currentKeyTypeComment = buildTypeComment(keyDevInfo.getDriver(), keyDevInfo.getType());
        }

        Log.d(THIS_FILE, "Selected key index = " + currentIndex);

        if (currentIndex == -1) {
            Log.e(THIS_FILE, "Select return -1 not valid.");
            setState(State.NO_KEY);
            return -1;
        }

        setCurrentDevIndex(currentIndex);

        long[] handle = new long[1];
        ret = getKeyInstance(currentIndex, handle);
        if (ret != 0) {
            Log.e(THIS_FILE, "Get key instance " + currentIndex + " failed.");
            setState(State.INIT_KEY_FAILED);
            return ret;
        }

        currentKey = handle[0];
        ret = keyOpen(currentKey);
        if (ret != 0) {
            Log.e(THIS_FILE, "Open key instance " + currentIndex + " failed.");
            setState(State.INIT_KEY_FAILED);
        } else {
            Log.d(THIS_FILE, "Init key module success. State READY!");
            setState(State.READY); // init success
        }

        return ret;
    }


    private static KeyWrapper instance = null;
    private Context context = null;

    public Context getContext() {
        return context;
    }

    public void setContext(Context context) {
        this.context = context;
    }

    public static KeyWrapper getInstance() {
        if (instance == null) {
            instance = new KeyWrapper();
            instance.setContext(MyApplication.myApplication);
        }
        return instance;
    }


    public native int configXdjaKey(XdjaKeyCfg xdjaKeyCfg);
    public native int configZhongfuKey(ZhongfuCfg zhongfuCfg);
    public native int configLongmaiKey(LongmaiCfg longmaiCfg);
    public native int configKoalKey(KoalCfg koalCfg);
    public native int sysInit(int keyFlag);
    public native int refreshDevice();
    public native int sysDestroy();
    public native int getDevCount();
    public native int getDevInfo(int index, KeyDevInfo devInfo);
    public native int setCurrentDevIndex(int index);
    private native int getCurrentDevIndex();
    public native int getKeyInstance(int index, long[] handle);
    public native int keyOpen(long handle);
    public native int keyClose(long handle);
    private native int keyVerifyPin(long handle, String pin);
    private native int keyReadSignCert(long handle, byte[] certBuf, int[] certLen);
    private native int keyGetSN(long handle, byte[] sn, int[] length);
    private native int keyRSAPrikeyCalc(long handle, int bits, byte[] data, int len, byte[] out, int[] outLen);
    private native int keySM2Sign(long handle, int isHashed, byte[] data, int len, byte[] out, int[] outLen);
    private native int keyGetSM2Param(long handle, XDJA_SM2_PARAM sm2Param);
    private native int keyGetSM2Id(long handle, byte[] id, int[] idLen);
    private native String keyGetError(int code);

    // xdja key interface called by other app
    private native int xdjaKeyReadCert(long handle, byte[] fid, byte[] cert, int[] len);
    private native int xdjaKeyVerifyPin(long handle, String pin, int role);
    private native int xdjaKeyRSAPrikeyCalc(long handle, byte[] prifid, byte[] data, int len, byte[] out, int[] outLen);
    private native int xdjaKeySM2Sign(long handle, byte[] pubfid, byte[] prifid,
                                      int datatype, byte[] data, int len, byte[] sign, int[] signLen);
    private native int xdjaKeyReadFile(long handle, byte[] fid, int pos, int len, byte[] out);

    private abstract class KeyRunnable{
        protected abstract int doRun(Object... args);

        public int run(Object... args) {

            if (state == State.NULL) {
                return StatusCode.NOT_INITIALIZED;
            }

            if (state == State.INIT_KEY_FAILED) {
                return StatusCode.INIT_FAILED;
            }
            if (state == State.READY) {
                return doRun(args);
            }

            if (state == State.INITIALIZING && Function.isMainThread()) { // 主线程不能阻塞
                return StatusCode.INIT_IN_PROGRESS;
            }

            while (state == State.INITIALIZING) { // 非主线程可以睡眠等待
                Log.d(THIS_FILE, "Not main thread. Key module is not ready. Wait! Currnet state " + state);

                try {
                    Thread.sleep(200);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }

                if (state == State.READY) {
                    Log.d(THIS_FILE, "Key module is ready. Call doRun!");
                    return doRun(args);
                }

                if (state == State.NO_KEY) {
                    return StatusCode.NO_VALID_KEY;
                }

            }

            return StatusCode.NO_VALID_KEY;

        }
    }

    private class XdjaReadCertRunnable extends KeyRunnable {

        @Override
        protected int doRun(Object... args) {
            return xdjaKeyReadCert(currentKey, (byte[])args[0], (byte[])args[1], (int[])args[2]);
        }
    }

    private class XdjaVerifyPINRunnable extends KeyRunnable{

        @Override
        protected int doRun(Object... args) {
            return xdjaKeyVerifyPin(currentKey, (String) args[0], (int) args[1]);
        }
    }

    private class VerifyPINRunnable extends KeyRunnable{

        @Override
        protected int doRun(Object... args) {
            return keyVerifyPin(currentKey, (String) args[0]);
        }
    };

    private class GetSNRunnable extends KeyRunnable{

        @Override
        protected int doRun(Object... args) {
            return keyGetSN(currentKey, (byte[])args[0], (int[])args[1]);
        }
    }

    private class ReadSignCertRunnable extends KeyRunnable{

        @Override
        protected int doRun(Object... args) {
            return keyReadSignCert(currentKey, (byte[])args[0], (int[])args[1]);
        }
    }

    private class RSAPrikeyCalcRunnable extends KeyRunnable {

        @Override
        protected int doRun(Object... args) {
            return keyRSAPrikeyCalc(currentKey, (int)args[0], (byte[])args[1], (int)args[2], (byte[])args[3], (int[])args[4]);
        }
    }

    // xdja rsa prikey calc
    private class XdjaRSAPrikeyCalcRunnable extends KeyRunnable {

        @Override
        protected int doRun(Object... args) {
            return xdjaKeyRSAPrikeyCalc(currentKey, (byte[]) args[0], (byte[])args[1], (int)args[2], (byte[])args[3], (int[])args[4]);
        }
    }

    private class SM2SignRunnable extends KeyRunnable {

        @Override
        protected int doRun(Object... args) {
            return keySM2Sign(currentKey, (int)args[0], (byte[])args[1], (int)args[2], (byte[])args[3], (int[])args[4]);
        }
    }

    // xdja sm2 sign
    private class XdjaSM2SignRunnable extends KeyRunnable {

        @Override
        protected int doRun(Object... args) {
            return xdjaKeySM2Sign(currentKey, (byte[])args[0], (byte[])args[1], (int)args[2],
                    (byte[])args[3], (int)args[4], (byte[])args[5], (int[]) args[6]);
        }
    }

    // xdja read file
    private class XdjaReadFileRunnable extends KeyRunnable {

        @Override
        protected int doRun(Object... args) {
            return xdjaKeyReadFile(currentKey, (byte[])args[0], (int)args[1], (int)args[2], (byte[])args[3]);
        }
    }

    private class GetSM2ParamRunnable extends KeyRunnable {

        @Override
        protected int doRun(Object... args) {
            return keyGetSM2Param(currentKey, (XDJA_SM2_PARAM)args[0]);
        }
    }

    private class GetSM2IDRunnable extends KeyRunnable {

        @Override
        protected int doRun(Object... args) {
            return keyGetSM2Id(currentKey, (byte[])args[0], (int[])args[1]);
        }
    }

    public int verifyPIN(String pin) {
        return new VerifyPINRunnable().run(pin);
    }

    // xdja key read cert
    public int xdjaKeyReadCert(byte[] fid, byte[] cert, int[] len) {
        return new XdjaReadCertRunnable().run(fid, cert, len);
    }

    // xdja verify pin with role
    public int xdjaKeyVerifyPIN(String pin, int role) {
        return new XdjaVerifyPINRunnable().run(pin, role);
    }

    public int getSN(byte[] sn, int[] length) {
        return new GetSNRunnable().run(sn, length);
    }

    public int readSignCert(byte[] certBuf, int[] certLen) {
        return new ReadSignCertRunnable().run(certBuf, certLen);
    }

    // xdja key read file
    public int xdjaKeyReadFile(byte[] fid, int pos, int len, byte[] out) {
        return new XdjaReadFileRunnable().run(fid, pos, len, out);
    }

    public int RSAPrikeyCalc(int bits, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        return new RSAPrikeyCalcRunnable().run(bits, dataIn, inLen, dataOut, outLen);
    }

    // xdja rsa prikey calc
    public int xdjaRSAPrikeyCalc(byte[] prifid, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        return new XdjaRSAPrikeyCalcRunnable().run(prifid, dataIn, inLen, dataOut, outLen);
    }

    public int SM2Sign(int isHashed, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        return new SM2SignRunnable().run(isHashed, dataIn, inLen, dataOut, outLen);
    }

    // xdja sm2 sign
    public int xdjaSM2Sign(byte[] pubfid, byte[] prifid, int isHashed,  byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        return new XdjaSM2SignRunnable().run(pubfid, prifid, isHashed, dataIn, inLen, dataOut, outLen);
    }

    public int getSM2Param(XDJA_SM2_PARAM sm2Param) {
        return new GetSM2ParamRunnable().run(sm2Param);
    }

    public int getSM2ID(byte[] id, int[] len) {
        return new GetSM2IDRunnable().run(id, len);
    }

    public KeySelector getKeySelector() {
        return keySelector;
    }

    public void setKeySelector(KeySelector keySelector) {
        this.keySelector = keySelector;
    }


    private static final int INIT_DEVICE_AUTO = 1001;
    private static final int NOTIFY_NO_CARD = 1002;

    HandlerThread handlerThread = new HandlerThread("key_thread");
    MyHandler myHandler = null;

    /**
     * 根据定义的规则，自动选择加密设备
     * @param keyFlag
     * @param keySelector
     * @return 选择的设备index
     */
    public void initDviceAuto(int keyFlag, KeySelector keySelector) {
        int status;

        currentKeyFlag = keyFlag;
        this.keySelector = keySelector;

//        if ((keyFlag&COVER_KEY) == COVER_KEY) {
//            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP_MR1) {
//                Log.e(THIS_FILE, "Start to call cover card instance init=====");
//                CoverCardWrapper.getInstance().init();
//            }
//        }

        if ((keyFlag&ZHONGFU_KEY) == ZHONGFU_KEY) {
            BluetoothAdapter bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
            if (!bluetoothAdapter.isEnabled()) {
                bluetoothAdapter.enable();
            }
        }


        Message message = myHandler.obtainMessage();
        message.what = INIT_DEVICE_AUTO;
        myHandler.sendEmptyMessage(INIT_DEVICE_AUTO);

//        initKeySystem();
//        EventManager.getInstance().addSDCardStateChangedListener(instance);
//        CoverCardWrapper.getInstance().addStateListener(instance);
    }

}
