package com.xdja.safehelper.update;

import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.support.annotation.NonNull;
import android.support.annotation.StringRes;
import android.support.v4.content.FileProvider;
import android.text.TextUtils;
import android.view.View;
import android.widget.Toast;


import com.xdja.safehelper.ActivityStackManager;
import com.xdja.safehelper.view.ProgressBarDialog;
import com.xdja.safehelper.view.TitleAndContentDialog;
import com.xdja.safehelper.view.ToastUtils;
import com.xdja.update.CheckCallback;
import com.xdja.update.DownloadCallback;
import com.xdja.update.UpdateManager;
import com.xdja.update.bean.CheckResult;
import com.xdja.update.enums.DownLoadFileError;
import com.xdja.safehelper.R;
import com.xdja.vhsm.utils.ContextCache;
import com.xdja.vhsm.utils.NetUtils;
import com.xdja.safehelper.view.MyCustomDialog;

import java.io.File;

public class Update {

    private Update() {
    }

    private static Update update = new Update();

    public static Update getInstance() {
        return update;
    }


    public void init() {
        UpdateManager.getInstance().install(ContextCache.getContext());
    }

    /**
     * 检测更新
     */
    public void checkUpdate(final Activity activity, final boolean isShowToast) {
        UpdateManager.getInstance().checkUpdate(new CheckCallback() {
            @Override
            public void onChecking() {

            }

            @Override
            public void onCheckResult(@NonNull CheckResult checkResult) {
                //强制更新；
                if (checkResult.getResult().equals(CheckResult.RESULT_UPDATE_POSITIVE)) {
                    SHUpdateManager.getInstance().hasNewVersion = true;
                    showUpdateDialog(activity, isShowToast, checkResult);
                } else if (checkResult.getResult().equals(CheckResult.RESULT_UPDATE_NEGATIVE)) {
                    SHUpdateManager.getInstance().hasNewVersion = false;
                    if (isShowToast) {
                        ToastUtils.makeText(ContextCache.getContext(), R.string.sh_update_newest, Toast.LENGTH_SHORT).show();
                    }
                } else {
                    if (isShowToast) {
                        ToastUtils.makeText(ContextCache.getContext(), R.string.sh_update_fail, Toast.LENGTH_SHORT).show();
                    }
                }
            }
        });
    }


    private void showUpdateDialog(final Activity activity, final boolean isShowToast, final CheckResult checkResult) {
        String ver = checkResult.getUpdates().get(0).getVersion();
        TitleAndContentDialog.Builder builder = new TitleAndContentDialog.Builder(activity)
                .setTitle(R.string.sh_update_title)
                .setMessage(activity.getString(R.string.sh_update_content, ver))
                .setPositiveButton(R.string.sh_update_ok, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        dialog.cancel();
                        download(activity, isShowToast, checkResult);
                    }
                });
        final int tag = checkResult.getUpdates().get(0).getUpdateTag();  //tag ==0，检测更新；tag ==1，强制更新
        builder.setNegativeButton(R.string.sh_update_cancel, new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
                UpdateManager.getInstance().stopDownload();
                if (tag == 1) { //强制更新
                    ActivityStackManager.getInstance().removeAll();

                }
            }
        });

        TitleAndContentDialog titleAndContentDialog = builder.create();
        titleAndContentDialog.setCancelable(false);
        titleAndContentDialog.setCanceledOnTouchOutside(false);
        titleAndContentDialog.show();
    }


    private void download(final Activity activity, final boolean isShowToast, CheckResult checkResult) {
        if (!NetUtils.isNetworkConnected(activity)) {
            ToastUtils.makeText(ContextCache.getContext(), R.string.sh_net_error, Toast.LENGTH_SHORT).show();
            return;
        }

        if (!NetUtils.isWifiConnected(activity)) {
            showNetDialog(activity, isShowToast, checkResult);
            return;
        }
        downloadFile(activity, isShowToast, checkResult);
    }

    private void downloadFile(final Activity activity, final boolean isShowToast, CheckResult checkResult) {
        final int tag = checkResult.getUpdates().get(0).getUpdateTag();  //tag ==0，检测更新；tag ==1，强制更新
        final ProgressBarDialog progressBarDialog = new ProgressBarDialog(activity);
        progressBarDialog.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                UpdateManager
                        .getInstance().stopDownload();
                if (tag == 1) {
                    if (!isShowToast) {
                        activity.finish();
                    }
                }
            }
        });
        progressBarDialog.setCancelable(false);
        progressBarDialog.setCanceledOnTouchOutside(false);

        UpdateManager
                .getInstance().downloadFile(checkResult
                .getUpdates()
                .get(0)
                .getFiles()
                .get(0), new DownloadCallback() {
            @Override
            public void onStart(@NonNull String fileId) {
                progressBarDialog.show();
            }

            @Override
            public void onProgress(@NonNull String fileId, long totalSize, long cSize) {
                int progress = Float.valueOf((((float) cSize / (float) totalSize) * 100)).intValue();
                progressBarDialog.updateProgress(progress);
            }

            @Override
            public void onStop(@NonNull String fileId) {

            }

            @Override
            public void onError(@NonNull String fileId, @NonNull DownLoadFileError error, Exception ex) {
                UpdateManager
                        .getInstance().stopDownload();
                progressBarDialog.dismiss();
                if (isShowToast) {
                    if (error.getKey() == 2) {
                        ToastUtils.makeText(ContextCache.getContext(), R.string.sh_net_error, Toast.LENGTH_SHORT).show();
                    } else {
                        ToastUtils.makeText(ContextCache.getContext(), R.string.sh_download_fail, Toast.LENGTH_SHORT).show();
                    }
                } else {
                    if (error.getKey() == 2) {
                        showDownLoadFailDialog(activity, R.string.sh_net_error);
                    } else {
                        showDownLoadFailDialog(activity, R.string.sh_download_fail);
                    }
                }
            }

            @Override
            public void onComplete(@NonNull String fileId, @NonNull String filePath) {
                progressBarDialog.updateProgress(100);
                progressBarDialog.dismiss();
                install(filePath);
            }

            @Override
            public int unit() {
                return 1024 * 1024;
            }

            @NonNull
            @Override
            public String downloadDir() {
                return ContextCache.getContext().getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS).getPath();
            }
        });
    }


    private void showNetDialog(final Activity activity, final boolean isShowToast, final CheckResult checkResult) {
        MyCustomDialog myCustomDialog = new MyCustomDialog.Builder(activity)
                .setMessage(R.string.sh_update_net_status)
                .setNegativeButton(R.string.sh_cancel, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        dialog.dismiss();
                    }
                })
                .setPositiveButton(R.string.sh_update_continue, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        dialog.dismiss();
                        downloadFile(activity, isShowToast, checkResult);
                    }
                })
                .create();
        myCustomDialog.setCancelable(false);
        myCustomDialog.setCanceledOnTouchOutside(false);
        myCustomDialog.show();
    }

    /**
     * 网络异常弹框
     */
    public void showDownLoadFailDialog(final Activity activity, @StringRes int strRes) {
        MyCustomDialog netErrorDialog = new MyCustomDialog.Builder(activity)
                .setMessage(strRes)
                .setPositiveButton(R.string.sh_confirm, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialogInterface, int i) {
                        dialogInterface.dismiss();
                        activity.finish();
                    }
                })
                .create();
        netErrorDialog.setCancelable(false);
        netErrorDialog.setCanceledOnTouchOutside(false);
        netErrorDialog.show();
    }


    /**
     * 安装apk
     *
     * @param filePath
     */
    private boolean install(String filePath) {
        try {
            if (TextUtils.isEmpty(filePath))
                return false;
            File file = new File(filePath);
            if (!file.exists()) {
                return false;
            }
            Intent intent = new Intent(Intent.ACTION_VIEW);
            intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);//增加读写权限
            }
            intent.setDataAndType(getPathUri(ContextCache.getContext(), filePath), "application/vnd.android.package-archive");
            ContextCache.getContext().startActivity(intent);
        } catch (Exception e) {
            e.printStackTrace();
            ToastUtils.makeText(ContextCache.getContext(), "安装失败，请重新下载", Toast.LENGTH_SHORT).show();
            return false;
        } catch (Error error) {
            error.printStackTrace();
            ToastUtils.makeText(ContextCache.getContext(), "安装失败，请重新下载", Toast.LENGTH_SHORT).show();
            return false;
        }
        return true;
    }


    private Uri getPathUri(Context context, String filePath) {
        Uri uri;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            String packageName = context.getPackageName();
            uri = FileProvider.getUriForFile(context, packageName + ".fileProvider", new File(filePath));
        } else {
            uri = Uri.fromFile(new File(filePath));
        }
        return uri;
    }
}
