package com.xdja.multichip.process;

import android.os.Bundle;
import android.util.Log;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.XDJA_SM2_PRIKEY;
import com.xdja.SafeKey.XDJA_SM2_PUBKEY;
import com.xdja.multichip.param.ImportCertBean;
import com.xdja.multichip.utils.ConvertUtil;
import com.xdja.skfapi.EccCipherBlob;
import com.xdja.skfapi.EccPublicKeyBlob;
import com.xdja.skfapi.EnvelopedKeyBlob;
import com.xdja.skfapi.SkfApiCode;

import java.util.Arrays;

import static com.xdja.skfapi.SkfApiCode.ECC_MAX_XCOORDINATE_BITS_LEN;
import static com.xdja.skfapi.SkfApiCode.ECC_MAX_YCOORDINATE_BITS_LEN;

/**
 * @author: zhangxiaolong@xdja.com <br/>
 * date:   2018/4/19 <br/>
 */

public class SuppertImportCert {
    /**
     * 导入证书
     *
     * @param bundle
     * @return
     */
    protected Bundle importCert(SupperJniApiBinder binder, Bundle bundle) {
        Bundle result = new Bundle();
        bundle.setClassLoader(ImportCertBean.class.getClassLoader());
        try {
            // 从bundle中获取bean
            ImportCertBean bean = bundle.getParcelable(ImportCertBean.class.getSimpleName());

            EnvelopedKeyBlob envelopedKey = getBean(bean.envelopedKey);
            int ret;
            //先验证PIN码，如果失败直接返回
            ret = binder.VerifyPIN(binder.mHandle, bean.role, bean.pin.getBytes(), bean.pin.length());
            if (ret != 0) {
                result.putInt("ret", ret);
                Log.e("ImportCertLog", "VerifyPin ret = " + ret);
                return result;
            }

            // 关于fid: 根据签名公钥的fid(发起者应该知道签名公钥fid),
            // 计算签名证书id(fid+2)， 交换证书（fid-1）, 交换私钥（fid-2）, 交换公钥（fid-3）

            //签名私钥fid
            byte[] signPriFid = {bean.signPubFid[0], (byte) (bean.signPubFid[1] + 1)};

            // 加过密的对称秘钥
            byte[] eccCipherBytes = getEccCipherBytes(envelopedKey.eccCipherBlob);
            byte[] out = new byte[eccCipherBytes.length];
            int[] outLen = new int[1];
            // 解密得到 对称秘钥
            ret = binder.SM2DecryptGM(binder.mHandle, signPriFid, eccCipherBytes, eccCipherBytes.length, out, outLen);
            if (ret != 0) {
                result.putInt("ret", ret);
                Log.e("ImportCertLog", "SM2DecryptGM ret = " + ret);
                return result;
            }

            // 解密过的对称秘钥
            byte[] cipherKey = Arrays.copyOf(out, outLen[0]);
            byte[] encPriKey = new byte[envelopedKey.cbEncryptedPriKey.length];
            //使用对称算法 解密 交换私钥
            if (envelopedKey.ulSymmAlgID == SkfApiCode.SGD_SM1_ECB) {
                ret = binder.SM1KEY(binder.mHandle, cipherKey, envelopedKey.cbEncryptedPriKey, envelopedKey.cbEncryptedPriKey.length, JNIAPI.ECB_DECRYPT, encPriKey, null);
                if (ret != 0) {
                    result.putInt("ret", ret);
                    Log.w("ImportCertLog", "SM1KEY ret = " + ret);
                    return result;
                }
            } else if (envelopedKey.ulSymmAlgID == SkfApiCode.SGD_SM4_ECB) {
                ret = binder.SM4KEY(binder.mHandle, cipherKey, envelopedKey.cbEncryptedPriKey, envelopedKey.cbEncryptedPriKey.length, JNIAPI.ECB_DECRYPT, encPriKey, null);
                if (ret != 0) {
                    result.putInt("ret", ret);
                    Log.w("ImportCertLog", "SM4KEY ret = " + ret);
                    return result;
                }
            } else {
                result.putInt("ret", -2);
                return result;
            }

            //导入签名证书
            byte[] signCertFid = {bean.signPubFid[0], (byte) (bean.signPubFid[1] + 2)};
            ret = binder.WriteCert(binder.mHandle, signCertFid, bean.signCert, bean.signCert.length);
            if (ret != 0) {
                result.putInt("ret", ret);
                Log.w("ImportCertLog", "WriteCert signCert ret = " + ret);
                return result;
            }

            byte[] exchangePubFid = {bean.signPubFid[0], (byte) (bean.signPubFid[1] - 3)};
            XDJA_SM2_PUBKEY pubKey = new XDJA_SM2_PUBKEY();
            System.arraycopy(envelopedKey.pubKey.xCoordinate, 0, pubKey.x, 0, pubKey.x.length);
            System.arraycopy(envelopedKey.pubKey.yCoordinate, 0, pubKey.y, 0, pubKey.y.length);
            //写入交换公钥
            ret = binder.WriteSm2PubKey(binder.mHandle, exchangePubFid, pubKey);
            if (ret != 0) {
                result.putInt("ret", ret);
                Log.w("ImportCertLog", "WriteSm2PubKey ret = " + ret);
                return result;
            }

            byte[] exchangePriFid = {bean.signPubFid[0], (byte) (bean.signPubFid[1] - 2)};
            XDJA_SM2_PRIKEY priKey = new XDJA_SM2_PRIKEY();
            System.arraycopy(encPriKey, 0, priKey.d, 0, encPriKey.length);
            //写入交换私钥
            ret = binder.WriteSm2PriKey(binder.mHandle, exchangePriFid, priKey);
            if (ret != 0) {
                result.putInt("ret", ret);
                Log.w("ImportCertLog", "WriteSm2PriKey ret = " + ret);
                return result;
            }

            byte[] exchangeCertFid = {bean.signPubFid[0], (byte) (bean.signPubFid[1] - 1)};
            //导入交换证书
            ret = binder.WriteCert(binder.mHandle, exchangeCertFid, bean.encCert, bean.encCert.length);
            if (ret != 0) {
                result.putInt("ret", ret);
                Log.w("ImportCertLog", "WriteCert encCert ret = " + ret);
                return result;
            }
            result.putInt("ret", 0);
        } catch (Exception e) {
            e.printStackTrace();
            result.putInt("ret", -1);
        }
        return result;
    }

    /**
     * 得到EnvelopedKeyBlob结构数据
     *
     * @param bytes
     * @return
     */
    private EnvelopedKeyBlob getBean(byte[] bytes) {
        if (bytes == null) {
            return null;
        }
        EnvelopedKeyBlob envelopedKey = new EnvelopedKeyBlob();

        int start;
        int len;

        start = 0;
        len = 4;
        envelopedKey.version = ConvertUtil.bytesToInt(Arrays.copyOfRange(bytes, start, start + len));

        start = start + len;
        len = 4;
        envelopedKey.ulSymmAlgID = ConvertUtil.bytesToInt(Arrays.copyOfRange(bytes, start, start + len));

        start = start + len;
        len = 4;
        envelopedKey.ulBits = ConvertUtil.bytesToInt(Arrays.copyOfRange(bytes, start, start + len));


        start = start + len;
        len = 64;
        envelopedKey.cbEncryptedPriKey = Arrays.copyOfRange(bytes, start, start + len);

        start = start + len;
        len = 4 + (ECC_MAX_XCOORDINATE_BITS_LEN / 8) + (ECC_MAX_YCOORDINATE_BITS_LEN / 8);
        envelopedKey.pubKey = getEccPublicKeyBlob(Arrays.copyOfRange(bytes, start, start + len));

        int tmpStart = (ECC_MAX_XCOORDINATE_BITS_LEN / 8) + (ECC_MAX_YCOORDINATE_BITS_LEN / 8) + 32;
        int tmpLen = 4;
        int eccCipherLen = ConvertUtil.bytesToInt(Arrays.copyOfRange(bytes, tmpStart, tmpStart + tmpLen));

        start = start + len;
        len = (ECC_MAX_XCOORDINATE_BITS_LEN / 8) + (ECC_MAX_YCOORDINATE_BITS_LEN / 8) + 32 + 4 + eccCipherLen;
        envelopedKey.eccCipherBlob = getEccCipherBlob(Arrays.copyOfRange(bytes, start, start + len));

        return envelopedKey;
    }

    private EccPublicKeyBlob getEccPublicKeyBlob(byte[] bytes) {
        if (bytes == null) {
            return null;
        }
        EccPublicKeyBlob eccPublicKeyBlob = new EccPublicKeyBlob();
        int start;
        int len;

        start = 0;
        len = 4;
        eccPublicKeyBlob.bitLen = ConvertUtil.bytesToInt(Arrays.copyOfRange(bytes, start, start + len));

        start = start + len;
        len = ECC_MAX_XCOORDINATE_BITS_LEN / 8;
        eccPublicKeyBlob.xCoordinate = Arrays.copyOfRange(bytes, start, start + len);

        start = start + len;
        len = ECC_MAX_YCOORDINATE_BITS_LEN / 8;
        eccPublicKeyBlob.yCoordinate = Arrays.copyOfRange(bytes, start, start + len);

        return eccPublicKeyBlob;
    }

    private EccCipherBlob getEccCipherBlob(byte[] bytes) {
        if (bytes == null) {
            return null;
        }
        EccCipherBlob eccCipher = new EccCipherBlob();

        int start;
        int len;

        start = 0;
        len = ECC_MAX_XCOORDINATE_BITS_LEN / 8;
        eccCipher.xXCoordinate = Arrays.copyOfRange(bytes, start, start + len);

        start = start + len;
        len = ECC_MAX_YCOORDINATE_BITS_LEN / 8;
        eccCipher.yCoordinate = Arrays.copyOfRange(bytes, start, start + len);

        start = start + len;
        len = 32;
        eccCipher.hash = Arrays.copyOfRange(bytes, start, start + len);

        start = start + len;
        len = 4;
        eccCipher.cipherLen = ConvertUtil.bytesToInt(Arrays.copyOfRange(bytes, start, start + len));

        start = start + len;
        len = eccCipher.cipherLen;
        eccCipher.cipher = Arrays.copyOfRange(bytes, start, start + len);

        return eccCipher;
    }

    /**
     * 得到EccCipherBlob的byte数组。
     * 注：不包括里面的cipherLen
     *
     * @param eccCipher
     * @return
     */
    private byte[] getEccCipherBytes(EccCipherBlob eccCipher) {
        if (eccCipher == null) {
            return null;
        }
        int len = eccCipher.xXCoordinate.length +
                eccCipher.yCoordinate.length +
                eccCipher.hash.length +
                4 +
                eccCipher.cipherLen;
        byte[] result = new byte[len];

        int start = 0;
        len = eccCipher.xXCoordinate.length;
        System.arraycopy(eccCipher.xXCoordinate, 0, result, start, start + len);

        start = start + len;
        len = eccCipher.yCoordinate.length;
        System.arraycopy(eccCipher.yCoordinate, 0, result, start, start + len);

        start = start + len;
        len = eccCipher.hash.length;
        System.arraycopy(eccCipher.hash, 0, result, start, start + len);

        start = start + len;
        len = 4;
        System.arraycopy(ConvertUtil.intToBytes(eccCipher.cipherLen), 0, result, start, start + len);

        start = start + len;
        len = eccCipher.cipher.length;
        System.arraycopy(eccCipher.cipher, 0, result, start, start + len);

        return result;
    }
}
