package com.xdja.multichip.process.bluetooth;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothManager;
import android.bluetooth.le.BluetoothLeScanner;
import android.bluetooth.le.ScanCallback;
import android.bluetooth.le.ScanResult;
import android.content.BroadcastReceiver;
import android.content.ContentProvider;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.Process;
import android.os.RemoteException;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import com.xdja.multichip.param.JniApiParam;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/7/21 <br/>
 */

public class BluetoothKeyProcessProvider extends ContentProvider {

    public static final String KEY_RET = "ret";
    public static final String KEY_BINDER = "Binder";
    public static final String KEY_JNIAPIPARAM = "JniApiParam";
    public static final String KEY_RESULT = "result";
    public static final String KEY_DEVADDRESS = "devAddress";
    public static final String KEY_DEVNAME = "devName";
    public static final String KEY_MESSENGER = "Messenger";

    /**
     * 参数为空
     */
    public final static int RET_PARAM_NULL = -1;

    /**
     * 不能连接蓝牙KEY
     */
    public final static int RET_CANNOT_CONNECT = -2;

    private ExecutorService cachedThreadPool = Executors.newCachedThreadPool();

    /**
     * 蓝牙key的binder对象 map
     * 其中key，参看{@link #convertToKey(String, String)}
     */
    private HashMap<String, JniApiBluetoothKeyBinder> binderMap = new HashMap<>();

    /**
     * 已连接的蓝牙地址和名称；
     */
    private HashMap<String, String> connectedAddressNameMap = null;

    private SelfScanCallback selfScanCallback = null;

    @Override
    public boolean onCreate() {
        initData();

        //add 2017年8月24日16:41:17 weizg 启动服务，提升进程的回收级别
        if (getContext() != null) {
            getContext().startService(new Intent(getContext(), BluetoothKeyProcessService.class));
        }
        return false;
    }

    int initFlag = 0;

    private void initData() {
        // 引入initFlag的目的是，当在蓝牙关闭时，界面又调用删除接口，这时候会创建蓝牙卡操作进程，
        // 但是得到的bleLeScanner为空，会引起异常崩溃。
        // 所以引入initFlag，并在call方法中每次都调用initData，
        // 这样能够保证再打开蓝牙时，能够正常的startScan
        if (initFlag == 1) {
            return;
        }
        BluetoothManager bleManager = (BluetoothManager) getContext().getSystemService(Context.BLUETOOTH_SERVICE);
        BluetoothAdapter bleAdapter = bleManager.getAdapter();
        if (bleAdapter.getState() != BluetoothAdapter.STATE_ON) {
            return;
        }
        IntentFilter filter = new IntentFilter();
        filter.addAction(BluetoothAdapter.ACTION_STATE_CHANGED);
        getContext().registerReceiver(mReceiver, filter);

        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
            BluetoothLeScanner bleLeScanner = bleAdapter.getBluetoothLeScanner();
            bleLeScanner.startScan(getSelfScanCallback().scanCallback);
        } else {
            bleAdapter.startLeScan(getSelfScanCallback().leScanCallback);
        }
        initFlag = 1;
    }

    private void unInit() {
        if (initFlag == 0) {
            return;
        }
        BluetoothManager bleManager = (BluetoothManager) getContext().getSystemService(Context.BLUETOOTH_SERVICE);
        BluetoothAdapter bleAdapter = bleManager.getAdapter();
        if (bleAdapter.getState() != BluetoothAdapter.STATE_ON) {
            return;
        }

        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
            BluetoothLeScanner bleLeScanner = bleAdapter.getBluetoothLeScanner();
            bleLeScanner.stopScan(getSelfScanCallback().scanCallback);
        } else {
            bleAdapter.stopLeScan(getSelfScanCallback().leScanCallback);
        }
    }

    @Nullable
    @Override
    public Cursor query(@NonNull Uri uri, @Nullable String[] projection, @Nullable String selection, @Nullable String[] selectionArgs, @Nullable String sortOrder) {
        return null;
    }

    @Nullable
    @Override
    public String getType(@NonNull Uri uri) {
        return null;
    }

    @Nullable
    @Override
    public Uri insert(@NonNull Uri uri, @Nullable ContentValues values) {
        return null;
    }

    @Override
    public int delete(@NonNull Uri uri, @Nullable String selection, @Nullable String[] selectionArgs) {
        return 0;
    }

    @Override
    public int update(@NonNull Uri uri, @Nullable ContentValues values, @Nullable String selection, @Nullable String[] selectionArgs) {
        return 0;
    }


    @Nullable
    @Override
    public Bundle call(@NonNull String method, @Nullable String arg, @Nullable Bundle extras) {
        initData();
        if ("GetJniApiBinder".equals(method)) {
            return handleGetJniApiBinder(method, arg, extras);
        } else if ("ConnectBluetoothKey".equals(method)) {
            return handleConnectBluetoothKey(method, arg, extras);
        } else if ("ConnectBluetoothKeyCallBack".equals(method)) {
            return handleConnectBluetoothKeyCallBack(method, arg, extras);
        } else if ("DisconnectBluetoothKey".equals(method)) {
            return handleDisconnectBluetoothKey(method, arg, extras);
        }
        return super.call(method, arg, extras);
    }

    /**
     * 处理 获取Binder
     *
     * @param method
     * @param arg
     * @param extras
     * @return
     */
    private Bundle handleGetJniApiBinder(@NonNull String method, @Nullable String arg, @Nullable Bundle extras) {
        ArrayList<Bundle> list = new ArrayList<>();
        Bundle bundle = new Bundle();

        if (judgeBluetoothCanUse()) {
            for (Map.Entry<String, JniApiBluetoothKeyBinder> entry : binderMap.entrySet()) {
                JniApiBluetoothKeyBinder binder = entry.getValue();
                JniApiParam param = binder.getJniApiParam();
                if (param != null) {
                    Bundle singleBundle = new Bundle();
                    singleBundle.putBinder(KEY_BINDER, binder);
                    singleBundle.putParcelable(KEY_JNIAPIPARAM, binder.getJniApiParam());
                    list.add(singleBundle);
                }
            }
        }
        bundle.putInt(KEY_RET, 0);
        bundle.putParcelableArrayList(KEY_RESULT, list);
        return bundle;
    }

    /**
     * 处理 蓝牙连接
     *
     * @param method
     * @param arg
     * @param extras
     * @return
     */
    private Bundle handleConnectBluetoothKey(@NonNull String method, @Nullable String arg, @Nullable Bundle extras) {
        Bundle bundle = new Bundle();
        String devAddress = extras.getString(KEY_DEVADDRESS);
        String devName = extras.getString(KEY_DEVNAME);
        if (TextUtils.isEmpty(devAddress) || TextUtils.isEmpty(devName)) {
            bundle.putInt(KEY_RET, RET_PARAM_NULL);
            return bundle;
        }
        return getConnectBluetoothKeyBundle(devAddress, devName);
    }

    /**
     * 内部使用方法
     *
     * @param devAddress
     * @param devName
     * @return
     */
    private Bundle getConnectBluetoothKeyBundle(String devAddress, String devName) {
        synchronized (BluetoothKeyProcessProvider.class) {
            Bundle bundle = new Bundle();
            String binderKey = convertToKey(devAddress, devName);
            JniApiBluetoothKeyBinder binder = null;
            if (binderMap.containsKey(binderKey)) {
                binder = binderMap.get(binderKey);
            } else {
                Pair<Integer, JniApiBluetoothKeyBinder> pair = JniApiBluetoothKeyBinder.make(getContext(), devAddress, devName);
                if (pair.first == 0) {
                    binder = pair.second;
                    binderMap.put(binderKey, binder);
                    //TODO: zhangxiaolong
                    // 因为目前只暂时只支持一个蓝牙key的连接，因此当蓝牙key连接成功一个之后，就不再接受扫描了
                    unInit();
                }
            }
            if (binder != null) {
                bundle.putInt(KEY_RET, 0);
                bundle.putBinder(KEY_BINDER, binder);
                bundle.putString(KEY_DEVADDRESS, devAddress);
                bundle.putString(KEY_DEVNAME, devName);
                JniApiParam param = binder.getJniApiParam();
                bundle.putParcelable(KEY_JNIAPIPARAM, param);
            } else {
                bundle.putInt(KEY_RET, RET_CANNOT_CONNECT);
            }
            return bundle;
        }
    }

    private Bundle handleConnectBluetoothKeyCallBack(@NonNull String method, @Nullable String arg, @Nullable Bundle extras) {
        Bundle bundle = new Bundle();
        final String devAddress = extras.getString(KEY_DEVADDRESS);
        final String devName = extras.getString(KEY_DEVNAME);
        if (TextUtils.isEmpty(devAddress) || TextUtils.isEmpty(devName)) {
            bundle.putInt(KEY_RET, RET_PARAM_NULL);
            return bundle;
        }
        final IBinder binder = extras.getBinder(KEY_MESSENGER);
        cachedThreadPool.submit(new Runnable() {
            @Override
            public void run() {
                Bundle bundle = getConnectBluetoothKeyBundle(devAddress, devName);
                if (binder != null) {
                    Messenger messenger = new Messenger(binder);
                    Message msg = new Message();
                    msg.what = 0;
                    msg.obj = bundle;
                    try {
                        messenger.send(msg);
                    } catch (RemoteException e) {
                        e.printStackTrace();
                    }
                }

            }
        });

        bundle.putInt(KEY_RET, 0);
        return bundle;
    }

    /**
     * 处理断开蓝牙key
     *
     * @param method
     * @param arg
     * @param extras
     * @return
     */
    private Bundle handleDisconnectBluetoothKey(@NonNull String method, @Nullable String arg, @Nullable Bundle extras) {
        Bundle bundle = new Bundle();
        String devAddress = extras.getString(KEY_DEVADDRESS);
        String devName = extras.getString(KEY_DEVNAME);
        if (TextUtils.isEmpty(devAddress) || TextUtils.isEmpty(devName)) {
            bundle.putInt(KEY_RET, RET_PARAM_NULL);
            return bundle;
        }
        delConnectedBluetoothKey(devAddress, devName);
        if (connectedAddressNameMap == null) {
            connectedAddressNameMap = getAllConnectedBluetoothKey();
        }
        connectedAddressNameMap.remove(devAddress);
        String key = convertToKey(devAddress, devName);

        JniApiBluetoothKeyBinder binder = binderMap.remove(key);
        if (binder != null) {
            binder.handleDisconnect();
        }
        bundle.putInt(KEY_RET, 0);
        return bundle;
    }

    /**
     * 判断蓝牙是否可用
     *
     * @return
     */
    private boolean judgeBluetoothCanUse() {
        BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();
        if (adapter == null) {
            return false;
        }

        int state = adapter.getState();
        if (state == BluetoothAdapter.STATE_ON || state == BluetoothAdapter.STATE_CONNECTED) {
            return true;
        }
        return false;
    }

    /**
     * 将devAddress和DevName转换成key
     *
     * @param devAddress
     * @param devName
     * @return
     */
    private String convertToKey(String devAddress, String devName) {
        return devAddress + "#" + devName;
    }

    /**
     * 存储已连接的蓝牙key
     *
     * @param devAddress
     * @param devName
     */
    private void saveConnectedBluetoothKey(String devAddress, String devName) {
        SharedPreferences preferences = getContext().getSharedPreferences("bluetooth_key_list.xml", Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = preferences.edit();
        editor.putString(devAddress, devName);
        editor.commit();
    }

    /**
     * 得到所有连接过的蓝牙key信息
     *
     * @return
     */
    private HashMap<String, String> getAllConnectedBluetoothKey() {
        SharedPreferences preferences = getContext().getSharedPreferences("bluetooth_key_list.xml", Context.MODE_PRIVATE);
        HashMap<String, String> map = (HashMap<String, String>) preferences.getAll();
        return map;
    }

    /**
     * 删除存储的已连接的蓝牙key
     *
     * @param devAddress
     * @param devName
     */
    private void delConnectedBluetoothKey(String devAddress, String devName) {
        SharedPreferences preferences = getContext().getSharedPreferences("bluetooth_key_list.xml", Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = preferences.edit();
        editor.remove(devAddress);
        editor.commit();
    }

    private BroadcastReceiver mReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            String action = intent.getAction();
            if (BluetoothAdapter.ACTION_STATE_CHANGED.equals(action)) {
                int state = intent.getIntExtra(BluetoothAdapter.EXTRA_STATE, 0);
                switch (state) {
                    case BluetoothAdapter.STATE_OFF: {
                        Process.killProcess(Process.myPid());
                        break;
                    }
                }
            }
        }
    };

    private SelfScanCallback getSelfScanCallback() {
        if (selfScanCallback == null) {
            selfScanCallback = new SelfScanCallback();
        }
        return selfScanCallback;
    }

    /**
     * 监听类
     */
    class SelfScanCallback {

        //modify 2017年10月10日16:39:25 weizg
        //在API21下防止ScanCallBack初始化
        ScanCallback scanCallback;
        BluetoothAdapter.LeScanCallback leScanCallback;

        SelfScanCallback(){
            if(android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP){
                Log.d("ble_callback","ScanCallback");

                scanCallback = new ScanCallback() {
                    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
                    @Override
                    public void onScanResult(int callbackType, ScanResult result) {
                        BluetoothDevice device = result.getDevice();
                        if (device == null || device.getName() == null) {
                            return;
                        }
                        String devAddress = device.getAddress();
                        String devName = device.getName();
                        forScanResult(devAddress, devName);
                    }
                };
            }else {

                Log.d("ble_callback","leScanCallback");

                leScanCallback = new BluetoothAdapter.LeScanCallback() {
                    @Override
                    public void onLeScan(BluetoothDevice device, int rssi, byte[] scanRecord) {
                        if (device == null || device.getName() == null) {
                            return;
                        }
                        String devAddress = device.getAddress();
                        String devName = device.getName();
                        forScanResult(devAddress, devName);
                    }
                };
            }
        }

/*        ScanCallback scanCallback = new ScanCallback() {
            @TargetApi(Build.VERSION_CODES.LOLLIPOP)
            @Override
            public void onScanResult(int callbackType, ScanResult result) {
                BluetoothDevice device = result.getDevice();
                if (device == null || device.getName() == null) {
                    return;
                }
                String devAddress = device.getAddress();
                String devName = device.getName();
                forScanResult(devAddress, devName);
            }
        };

        BluetoothAdapter.LeScanCallback leScanCallback = new BluetoothAdapter.LeScanCallback() {
            @Override
            public void onLeScan(BluetoothDevice device, int rssi, byte[] scanRecord) {
                if (device == null || device.getName() == null) {
                    return;
                }
                String devAddress = device.getAddress();
                String devName = device.getName();
                forScanResult(devAddress, devName);
            }
        };*/

        void forScanResult(final String devAddress, final String devName) {
            cachedThreadPool.submit(new Runnable() {
                @Override
                public void run() {
                    String key = convertToKey(devAddress, devName);
                    if (binderMap.containsKey(key)) {
                        return;
                    }
                    if (connectedAddressNameMap == null) {
                        connectedAddressNameMap = getAllConnectedBluetoothKey();
                    }
                    if (connectedAddressNameMap.containsKey(devAddress)) {
                        getConnectBluetoothKeyBundle(devAddress, devName);
                    }
                }
            });
        }
    }
}
