package com.xdja.multichip.process.tfcard;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Bundle;
import android.util.Log;

import com.xdja.multichip.GetMultiJniApiBinder;
import com.xdja.multichip.MultiChip;
import com.xdja.multichip.param.ChipParam;
import com.xdja.multichip.param.JniApiParam;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;


class TFJniApiHandle {
    public static final String action_tf_mounted = "com_xdja_safekeyservice_tf_mounted";
    public static final String action_tf_removed = "com_xdja_safekeyservice_tf_removed";
    public static final String tag_currrent_tf = "tag_current_safe_tf";

    private TFJniApiBinder chipBinder = null;
    private final String tag = "XdjaTFProcessService";
    ArrayList<Bundle> tfInfoBundleList = new ArrayList<>();

    private HashMap<String, TFJniApiBinder> pathBinderMap = new HashMap<>();

    private static TFJniApiHandle instance;

    private TFJniApiHandle() {
    }

    public static TFJniApiHandle getInstance() {
        if (instance == null) {
            synchronized (TFJniApiHandle.class) {
                if (instance == null) {
                    instance = new TFJniApiHandle();
                }
            }
        }
        return instance;
    }


    /**
     * @param context
     * @return
     */
    public ArrayList<Bundle> getCurrentTFCardInfo(Context context) {
        if (context != null) {
            registerTFEventReceiver(context);
        }
        //检查获取当前设备上所有外置TF卡的路径列表
        //根据列表信息获取对应的卡信息
        ArrayList<Bundle> bundleList = new ArrayList<>();

        List<String> allExternalSdCardPath = TFCardUtils.getSuspicionSDCardPath(context);
        if (allExternalSdCardPath.size() > 0) {

            for (String sdcardPath : allExternalSdCardPath) {
                Log.i(tag, "sdcardPath: " + sdcardPath);

                TFJniApiBinder tfJniApiBinder = handleJniApiBinder(context, sdcardPath);
                JniApiParam chipParam = tfJniApiBinder.getJniApiParam();

                //modify 2017年7月10日14:25:42 weizg
                //新增判断TF卡是否已经获取到；原因是，在魅族手机上，发现一个TF卡会遍历到三个路径，导致获取到三个TF信息
                //如果打开卡成功
                if (chipParam != null && !isChipGetted(bundleList, chipParam)) {
                    Bundle bundle = new Bundle();
                    bundle.putBinder(GetMultiJniApiBinder.KEY_BINDER_BINDER, tfJniApiBinder.asBinder());
                    bundle.putParcelable(GetMultiJniApiBinder.KEY_PARCELABLE_CHIPPARAM, chipParam);

                    bundleList.add(bundle);

                    //add 2017年7月18日15:06:54 weizg 外置TF在手机会有多个挂载路径，如果全部遍历打开的话，可能
                    //接口时间过长，引起界面卡顿，考虑到现在设备一般只有一个TF卡，故如果成功打开过一个则不再继续
                    break;
                }

            }
        }

        return bundleList;
    }


    private boolean isChipGetted(ArrayList<Bundle> bundleList, JniApiParam chipParam) {
        boolean isInList = false;

        for (Bundle bundle : bundleList) {
            JniApiParam chipParamInList = bundle.getParcelable(GetMultiJniApiBinder.KEY_PARCELABLE_CHIPPARAM);

            if (chipParamInList != null && chipParam.cardId.
                    equalsIgnoreCase(chipParamInList.cardId)) {
                isInList = true;
                break;
            }
        }

        return isInList;
    }

    /**
     * 处理chipParam
     */
    private TFJniApiBinder handleJniApiBinder(Context context, String tfcardPath) {
        //创建具体的操作芯片的Binder
        if (chipBinder == null) {
            chipBinder = createJniApiBinder(context, tfcardPath);
        }
        return chipBinder;
    }

    private synchronized TFJniApiBinder createJniApiBinder(Context context, String tfcardPath) {
        // 保证一个tfcardPath下只有一个binder
        if (pathBinderMap.containsKey(tfcardPath)) {
            return pathBinderMap.get(tfcardPath);
        }
        TFJniApiBinder binder = new TFJniApiBinder(context, tfcardPath);
        pathBinderMap.put(tfcardPath, binder);
        return binder;
    }


    private void registerTFEventReceiver(Context context) {
        IntentFilter intentFilter = new IntentFilter();
        intentFilter.addAction(Intent.ACTION_MEDIA_MOUNTED);
        intentFilter.addAction(Intent.ACTION_MEDIA_REMOVED);
        intentFilter.addAction(Intent.ACTION_MEDIA_EJECT);
        intentFilter.addDataScheme("file");

        context.registerReceiver(tfCardEventReceiver, intentFilter);
    }

    //动态注册TF卡插拔事件广播监听；
    // 如果监听到TF卡插入事件，则重新获取可用列表，更新本地缓存；
    // 如果监听到TF卡移除事件，则遍历缓存的列表信息，确定那个tf卡别移除
    //如果卡不可用，则发送卡当前不可用广播
    private BroadcastReceiver tfCardEventReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {

            Log.v(tag, "xdjatfprocessService BroadcastReceiver");

            //注意：当TF卡卸载或移除（从卡槽拔出），某系统上（魅族）会杀死打开过TF的进程；
            //新增了Intent.ACTION_MEDIA_REMOVED广播类型，因为在TF快速插入，再拔出时，系统只发出了该类型广播
            //tf卡被弹出（正常卸载或者强制移除卡）
            if (Intent.ACTION_MEDIA_EJECT.equals(intent.getAction()) ||
                    Intent.ACTION_MEDIA_REMOVED.equals(intent.getAction())) {
                Log.v(tag, "tf卡移除");
                //确定那个卡被移除，然后广播通知；信息列表
                handleMediaRemoved(context);

            } else if (Intent.ACTION_MEDIA_MOUNTED.equals(intent.getAction())) {
                //tf卡被挂载
                Log.v(tag, "tf卡被挂载");
                handleMediaMounted(context);
            }
        }
    };

    private void handleMediaRemoved(Context context) {
        //如果接收到设备移除广播
        //确定那个卡被移除，然后广播通知；信息列表
        if (tfInfoBundleList.size() > 0) {
            //如果有缓存可用tf信息

            //重新获取当前可用的tf信息，然后更新缓存
            List<Bundle> currentTFCardInfo = getCurrentTFCardInfo(context);
            //如果当前有缓存信息，则更新缓存，新增可用的，删除当前不可用的
            updateTFInfoInCache(currentTFCardInfo);

            Log.i(tag, "handleMediaRemoved, sendTFInfoChangeBroadcast");

            sendTFInfoChangeBroadcast(context, action_tf_removed, tfInfoBundleList);

        }
    }

    private void handleMediaMounted(Context context) {
        //重新获取当前可用TF卡信息
        List<Bundle> currentTFCardInfo = getCurrentTFCardInfo(context);

        //如果遍历到TF卡信息
        if (!currentTFCardInfo.isEmpty()) {
            //检查出新增的TF卡信息
            if (tfInfoBundleList.isEmpty()) {
                //如果当前尚未缓存任何信息
                tfInfoBundleList.addAll(currentTFCardInfo);

                //发送当前可用设备信息广播
                sendTFInfoChangeBroadcast(context, action_tf_mounted, tfInfoBundleList);

            } else {
                //如果当前有缓存信息，则更新缓存，新增可用的，删除当前不可用的
                updateTFInfoInCache(currentTFCardInfo);

                sendTFInfoChangeBroadcast(context, action_tf_mounted, tfInfoBundleList);
            }


        } else {
            //如果未遍历到可用，清空本地缓存，然后发送当前无可用card通知
            tfInfoBundleList.clear();

            sendTFInfoChangeBroadcast(context, action_tf_mounted, new ArrayList<Bundle>(0));
        }
    }

    private void sendTFInfoChangeBroadcast(Context context, String action, ArrayList<Bundle> bundles) {
        Intent tfAddBroadcastIntent = new Intent();
        tfAddBroadcastIntent.setAction(action);
        tfAddBroadcastIntent.putParcelableArrayListExtra(tag_currrent_tf, bundles);
        context.sendBroadcast(tfAddBroadcastIntent);
    }

    private void updateTFInfoInCache(List<Bundle> currentTFCardInfo) {
        HashMap<String, Bundle> currentInfoMap = new HashMap<>();
        for (Bundle bundle : currentTFCardInfo) {
            JniApiParam chipParam = bundle.getParcelable(GetMultiJniApiBinder.KEY_PARCELABLE_CHIPPARAM);

            if(chipParam != null){
                currentInfoMap.put(chipParam.cardId, bundle);
            }

        }

        //如果缓存的TF卡信息没有在最新查询到的列表中，则移除
        int cacheListSize = tfInfoBundleList.size();
        for (int i = cacheListSize - 1; i >= 0; i--) {

            Bundle bundle = tfInfoBundleList.get(i);

            JniApiParam chipParam = bundle.getParcelable(GetMultiJniApiBinder.KEY_PARCELABLE_CHIPPARAM);
            if (chipParam == null){
                continue;
            }
            //如果没有在最新查找的列表中，则移除
            if (currentInfoMap.get(chipParam.cardId) == null) {
                tfInfoBundleList.remove(i);
            }
        }

        //新查询到的列表中，是否有当前缓存中没有包含的，如果有，则添加
        HashMap<String, Bundle> tfInfoInCacheMap = new HashMap<>();
        for (Bundle bundle : tfInfoBundleList) {
            JniApiParam chipParam = bundle.getParcelable(GetMultiJniApiBinder.KEY_PARCELABLE_CHIPPARAM);
            if(chipParam != null){
                tfInfoInCacheMap.put(chipParam.cardId, bundle);
            }
        }

        for (Bundle bundle : currentTFCardInfo) {
            JniApiParam chipParam = bundle.getParcelable(GetMultiJniApiBinder.KEY_PARCELABLE_CHIPPARAM);
            //如果当前tf卡信息没有在缓存list中
            if (chipParam != null && tfInfoInCacheMap.get(chipParam.cardId) == null) {
                tfInfoBundleList.add(bundle);
            }
        }
    }


}

