package com.xdja.multichip.process.tfcard;

import android.content.Context;
import android.content.pm.PackageManager;
import android.os.RemoteException;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.XDJA_DEVINFO;
import com.xdja.multichip.AccessControlManager;
import com.xdja.multichip.JNIPinManager;
import com.xdja.multichip.InsideMultiChipUnitePinManager;
import com.xdja.multichip.param.ChipParam;
import com.xdja.multichip.param.ErrorCode;
import com.xdja.multichip.process.IChipParamChangeListener;
import com.xdja.multichip.process.SupperChipBinder;
import com.xdja.skfapi.BlockCipherParam;
import com.xdja.skfapi.DevInfo;
import com.xdja.skfapi.EccCipherBlob;
import com.xdja.skfapi.EccPrivateKeyBlob;
import com.xdja.skfapi.EccPublicKeyBlob;
import com.xdja.skfapi.EccSignatureBlob;
import com.xdja.skfapi.EnvelopedKeyBlob;
import com.xdja.skfapi.FileAttribute;
import com.xdja.skfapi.RsaPrivateKeyBlob;
import com.xdja.skfapi.RsaPublicKeyBlob;
import com.xdja.skfapi.SkfApi;
import com.xdja.skfapi.SkfApiCode;

import static com.xdja.skfapi.SkfApiCode.SGD_SM2_1;

/**
 * 信大捷安TF芯片
 *
 * author: wzg@xdja.com <br/>
 * date:   2017/6/6 <br/>
 * <p>
 */

public class XdjaTFBinder extends SupperChipBinder {
    private static final String tag = "XdjaTFBinder";
    private long tfDevHandle = -1;

    //原有卡库接口，tf设备的操作句柄
    private long xdjaTFDevHandle = -1;

//    private static XdjaTFBinder instance;

    private final String tf_oper_path_part1 = "/Android/data/";
    private final String tf_oper_path_part2 = "/files/";

    //tf卡路径
    private String tfcardOperFullPath;
    //add 2017年7月18日13:35:10 weizg 新增tf根目录变量
    //原因是，在HTC m8上，tf卡手机，已经root，可以通过打开根目录的cmd.bin打开tf卡；
    //另外，据介绍，如果TF卡可以通过根目录，私有目录两种方式都打开设备，但一种方式打开后，另外
    //一种方式就无法打开。
    private String tfcardPath;

    private Context context;
    private static JNIAPI jniapi = new JNIAPI();
    private static SkfApi skfApi = new SkfApi();

    XdjaTFBinder(Context context, String tfcardPath) {
        this.context = context;
        this.tfcardOperFullPath = getTFOperFullPath(tfcardPath);
        this.tfcardPath = tfcardPath;
    }

    String getTfcardPath(){
        return tfcardPath;
    }

    private String getTFOperFullPath(String tfcardPath){
        //例如：/storage/sdcard1/Android/data/com.xdja.safekeyservice/files
        return tfcardPath + tf_oper_path_part1 + context.getPackageName() + tf_oper_path_part2;
    }

    /*public static XdjaTFBinder getInstance(Context context) {
        if (instance == null) {
            synchronized (XdjaTFBinder.class) {
                if (instance == null) {
                    instance = new XdjaTFBinder(context);
                }
            }
        }
        return instance;
    }*/

    //加载国密接口的so库
    static {
        try {
            System.loadLibrary("skfproxy");
        } catch (Throwable t) {
            Log.i(tag,"load skf lib exception");
            t.printStackTrace();
        }
    }


    @Override
    public ChipParam getChipParam() throws RemoteException {
        //枚举打开TF卡
        //因现有封装的国密接口中没有打开TF卡类型，需要用原有卡库打开卡
       //打开tf卡的使原始JNI api
        //如果打开失败或者获取卡信息设备，则返回null

        /*final long[] devHandle = new long[1];
        //打开TF卡类型的安全芯片
        int ret = jniapi.OpenDevByNameEx(context.getPackageName().getBytes(), devHandle);*/

        int ret = openTFDev();

        Log.i(tag,"openRet: " + ret);

        //打开成功，则获取设备信息
        if(ret == JNIAPI.XKR_OK){

            //获取设备信息
            String cardId = getTFCardId();

            if(!TextUtils.isEmpty(cardId)){
                ChipParam chipParam = new ChipParam();
                chipParam.chipType = ChipParam.TYPE_TF;
                chipParam.uniqueID = cardId;
                chipParam.manufacturer = ChipParam.MANUFACTURER_XDJA;

                return chipParam;
            }

        }

        return null;
    }

    private String getTFCardId(){
        Log.i(tag,"getTFCardId: ");

        XDJA_DEVINFO devinfo = new XDJA_DEVINFO();
        int ret = jniapi.GetDevInfo(xdjaTFDevHandle, devinfo);
        if(ret == 0){
            return new String(devinfo.cardid, 0, 32 );
        }else if(ret == -1){
            //句柄无效，则重新打开一次
            final long[] devHandle = new long[1];
            ret = jniOpenTFDev(devHandle);

            if(ret == 0){
                xdjaTFDevHandle = devHandle[0];
                if(jniapi.GetDevInfo(devHandle[0], devinfo) == 0){
                    return new String(devinfo.cardid, 0, 32 );
                }
            }
        }

        return "";

        /*final long[] devHandle = new long[1];
        //打开TF卡类型的安全芯片
        int ret = jniOpenTFDev(devHandle);

        XDJA_DEVINFO devinfo = new XDJA_DEVINFO();
        if(ret == 0
                && jniapi.GetDevInfo(devHandle[0], devinfo) == 0){
           return new String(devinfo.cardid, 0, 32 );
        }

        return "";*/
    }

    /**
     * 打开TF卡设备，如果打开成功则保存打开的handle
     * @return 打开结果
     */
    private int openTFDev(){
        final long[] devHandle = new long[1];
        //打开TF卡类型的安全芯片
//        int ret = jniapi.OpenDevByNameEx(context.getPackageName().getBytes(), devHandle);
        int ret = jniOpenTFDev(devHandle);

        //使用自定路径打开TF卡
//        int ret = jniapi.OpenDevByName(tfcardOperFullPath.getBytes(), devHandle);

        Log.i(tag,"openTFDev: " + ret);

        if(ret == 0){

//            testEnumDev();

            //modify 2017年7月18日15:16:24 weizg 为了后面获取卡号时的接口执行时间，
            // 本处缓存原始操作句柄
            xdjaTFDevHandle = devHandle[0];

            XDJA_DEVINFO devinfo = new XDJA_DEVINFO();
            ret = jniapi.GetDevInfo(devHandle[0], devinfo);

            if(ret == 0){

                String tfCardId = new String(devinfo.cardid, 0, 32 );
                Pair<Integer, Long> skfDevHandlePair = getSkfDevHandle(tfCardId);

                if(skfDevHandlePair.first == 0){
                    tfDevHandle = skfDevHandlePair.second;
                }else {
                    //国密接口未找到对应的设备句柄
                    ret = skfDevHandlePair.first;
                }
            }

        }

        return ret;
    }

    private void testEnumDev() {
        int[] devNum = new int[1];
        int enumDevRet = jniapi.EnumDev(JNIAPI.CT_TF_XDJA, devNum);
        Log.i(tag, "EnumDev ret: " + enumDevRet);
        if (enumDevRet == 0) {
            Log.i(tag, "EnumDev num: " + devNum[0]);
        }
    }

    private int jniOpenTFDev(long[] devHandle){
        //因为存在TF卡可以通过根目录/cmd.bin和私有目录/cmd.bin都能打开的情况，同时统一一种方式打开后，
        //另外一种方式就无法打开卡，但同一种方式可以两个进程同时打开，所以此处尝试两种方式打开卡
        int ret = jniapi.OpenDevByNameEx(context.getPackageName().getBytes(), devHandle);
        Log.i(tag,"jniOpenTFDev OpenDevByNameEx: " + ret);

        if(ret != 0){

            //如果打开不成功，则尝试另外一种方式
            String tfPathTmp = tfcardPath;
            if(!tfPathTmp.endsWith("/")){
                tfPathTmp = tfcardPath + "/";
            }

            Log.i(tag,"tfPathTmp : " + tfPathTmp);

            ret = jniapi.OpenDevByName(tfPathTmp.getBytes(),devHandle);
            Log.i(tag,"OpenDevByName: " + ret);
        }

        return ret;
    }


    /**
     * 根据cardID获取国密接口的设备句柄；<br/><br/>
     *
     * 之所以这样做，是因为通过国密接口操作TF卡时，需要首先通过原来卡库接口打开设备后，才能通过国密接口
     * 枚举连接设备获取句柄；同时如果当前手机上既有芯片也有TF卡，通过国密接口枚举设备会找到多个设备，为了
     * 找到相应设备句柄，通过传入cardId方式过了国密接口句柄；
     *
     * <br/>
     * 正确方式应该是国密接口应该提供芯片的信息来区分TF卡，板载芯片类型。
     * @param cardID 芯片卡号
     * @return cardId对应的国密接口设备句柄；如果失败返回-1；
     */
    private Pair<Integer, Long> getSkfDevHandle(String cardID){
        Pair<Integer, Long> getSfkDevHandleResPair = Pair.create(JNIAPI.XKR_NO_KEY,
                Long.valueOf(-1));

        long devHandle = -1;

        byte szDevNameList[] = new byte[260];
        int[] ulSize = new int[1];

        //枚举设备
        int ret = skfApi.EnumDev(1, szDevNameList, ulSize);
        Log.i(tag,"skfApi.EnumDev ret: " + ret);

        if (ret==0){
            String namesStr = new String(szDevNameList);
            if (namesStr.contains("\0")){
                String[] devNames = namesStr.split("\0");

                Log.i(tag,"skfApi.EnumDev: " + namesStr);

                //遍历打开每个设备，通过芯片id确定设备句柄
                for (String devName : devNames){
                    //连接设备
                    //获取设备信息
                    //比对是否为TF卡

                    long[] phDdev = new long[1];
                    ret = skfApi.ConnectDev(devName.getBytes(),phDdev);

                    Log.i(tag,"skfApi.ConnectDev name: " + devName);
                    Log.i(tag,"skfApi.ConnectDev ret: " + ret);

                    //连接设备成功
                    if(ret == 0){
                        DevInfo devInfo = new DevInfo();
                        ret = skfApi.GetDevInfo(phDdev[0], devInfo);
                        Log.i(tag,"skfApi.GetDevInfo ret: " + ret);

                        Log.i(tag,"xdja safektf id: " + cardID);

                        /*StringBuilder stringBuilder = new StringBuilder();
                        stringBuilder.append("[");
                        for (int i = 0; i < devInfo.serialNumber.length; i++) {
                            stringBuilder.append(devInfo.serialNumber[i]);
                            stringBuilder.append(",");
                        }
                        stringBuilder.append("]");
                        Log.i(tag,"skfApi.serialNumber data: " + stringBuilder.toString());*/

                        Log.i(tag,"skfApi.GetDevInfo id: " + new String(devInfo.serialNumber));

                        String cardIdBySkf = new String(devInfo.reserved, 0, 32);
                        Log.i(tag,"skfApi.reserved : " + cardIdBySkf);

                        //获取设备信息成功
                        if(ret == 0 &&
                                cardID.equalsIgnoreCase(cardIdBySkf)){

                            getSfkDevHandleResPair = Pair.create(0,phDdev[0]);

                            break;
                        }
                    }

                }
            }

        }else{
            getSfkDevHandleResPair = Pair.create(ret,Long.valueOf(-1));
        }


        return getSfkDevHandleResPair;
    }

    /**
     * 检查当前TF卡的操作句柄；
     * 如果本地没有缓存则尝试打开TF卡并返回句柄；如果尝试打开或检查失败则返回非零
     * @return 检查成功返回0；否则返回非零
     */
    private int checkTfDevHandle(){
        if(tfDevHandle != -1){
            return SkfApiCode.SAR_OK;
        }

        //尝试打开设备
        return reOpenTFDev();
    }

    private int reOpenTFDev() {
        int openRet = openTFDev();

        //未找到设备
        if(openRet != JNIAPI.XKR_OK){
           int ret = SkfApiCode.SAR_FAIL;

            if(openRet == JNIAPI.XKR_NO_KEY){
                ret = SkfApiCode.SAR_DEV_NOT_FOUND;
            }

            return ret;
        }

        return SkfApiCode.SAR_OK;
    }


    private boolean isByteArrayEmpty(byte [] bytes){
        return bytes == null || bytes.length < 0;
    }

    @Override
    public int WaitForDevEvent(byte[] szDevName, int[] pulDevNameLen, int[] pulEvent) throws RemoteException {
        //本次不实现该接口
        return -1;
    }

    @Override
    public int CancelWaitForDevEvent() throws RemoteException {
        //本次不实现该接口
        return -1;
    }

    @Override
    public int EnumDev(int bPresent, byte[] szNameList, int[] pulSize) throws RemoteException {
        //TF卡类不支持枚举，原因是Android 4.4以上卡库不支持枚举TF卡；使用时直接open设备
        return SkfApiCode.SAR_NOSUPPORTYETERR;
    }

    @Override
    public int ConnectDev(byte[] szName, long[] phDdev) throws RemoteException {
        //TF类型的芯片，连接（打开）设备时不用考虑第一个参数
        if(phDdev == null || phDdev.length < 1){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        int connectRet = SkfApiCode.SAR_FAIL;

        if(tfDevHandle == -1){
            //open tf dev
            int openTFDevRet = openTFDev();
            //open success
            if(openTFDevRet == JNIAPI.XKR_OK){
                phDdev[0] = tfDevHandle;
                connectRet = SkfApiCode.SAR_OK;
            }
        }else {
            phDdev[0] = tfDevHandle;

            connectRet = SkfApiCode.SAR_OK;
        }


        return connectRet;
    }

    @Override
    public int DisConnectDev(long hDev) throws RemoteException {
        //因为open接口没有对外开放，所以现在关闭接口也暂时不实现
        /*if(hDev == 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        int closeRet = jniapi.CloseDev(hDev);
        Log.i(tag,"closeDev ret: " + closeRet);

        if(closeRet != JNIAPI.XKR_OK){
            return SkfApiCode.SAR_FAIL;
        }*/

        return 0;
    }

    @Override
    public int GetDevState(byte[] szDevName, int[] pulDevState) throws RemoteException {
        return skfApi.GetDevState(szDevName, pulDevState);
    }

    @Override
    public int SetLabel(long hDev, byte[] szLabel) throws RemoteException {
        int ret;

        if(szLabel == null || szLabel.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.SetLabel(tfDevHandle,szLabel);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.SetLabel(tfDevHandle,szLabel);
            }
        }

        return ret;
    }

    @Override
    public int GetDevInfo(long hDev, DevInfo pDevInfo) throws RemoteException {
        int ret;

        if(pDevInfo == null ){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.GetDevInfo(tfDevHandle,pDevInfo);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.GetDevInfo(tfDevHandle,pDevInfo);
            }
        }

        return ret;
    }

    @Override
    public int LockDev(long hDev, int ulTimeOut) throws RemoteException {
        int ret;

        if(ulTimeOut < 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.LockDev(tfDevHandle,ulTimeOut);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.LockDev(tfDevHandle,ulTimeOut);
            }
        }

        return ret;
    }

    @Override
    public int UnlockDev(long hDev) throws RemoteException {
        int ret;

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.UnlockDev(tfDevHandle);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.UnlockDev(tfDevHandle);
            }
        }

        return ret;
    }

    @Override
    public int Transmit(long hDev, byte[] pbCommand, int ulCommandLen, byte[] pbData, int[] pulDataLen) throws RemoteException {
        int ret;

        if(pbCommand == null || ulCommandLen <= 0 || pbData == null || pbData.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.Transmit(tfDevHandle,pbCommand,ulCommandLen,pbData,pulDataLen);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.Transmit(tfDevHandle,pbCommand,ulCommandLen,pbData,pulDataLen);
            }
        }

        return ret;
    }

    @Override
    public int ChangeDevAuthKey(long hDev, byte[] pbKeyValue, int ulKeyLen) throws RemoteException {
        int ret;

        if(pbKeyValue == null || pbKeyValue.length <= 0 || ulKeyLen <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.ChangeDevAuthKey(tfDevHandle,pbKeyValue,ulKeyLen);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.ChangeDevAuthKey(tfDevHandle,pbKeyValue,ulKeyLen);
            }
        }

        return ret;
    }

    @Override
    public int DevAuth(long hDev, byte[] pbAuthData, int ulLen) throws RemoteException {
        int ret;

        if(pbAuthData == null || pbAuthData.length <= 0 || ulLen <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.DevAuth(tfDevHandle,pbAuthData,ulLen);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.DevAuth(tfDevHandle,pbAuthData,ulLen);
            }
        }

        return ret;
    }

    @Override
    public int ChangePIN(long hApplication, int ulPINType, byte[] szOldPin, byte[] szNewPin, int[] pulRetryCount) throws RemoteException {
//        int ret;

        AccessControlManager accessControlManager = AccessControlManager.getInstance(context);
        int checkRes = accessControlManager.checkByUidPid(getCallingUid(), getCallingPid(), "ChangePIN");
        if(checkRes != 0){
            Log.i(tag,"checkByUidPid res: " + checkRes);
            return ErrorCode.RET_NO_POWER;
        }

        if(hApplication <=0
                || szOldPin == null || szOldPin.length <= 0
                || szNewPin == null || szNewPin.length <= 0
                || pulRetryCount == null || pulRetryCount.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        int ret = skfApi.ChangePIN(hApplication,ulPINType,szOldPin,szNewPin,pulRetryCount);
        callUnitePin(ret,ulPINType,szNewPin,"ChangePIN");
        return ret;
    }

    @Override
    public int GetPINInfo(long hApplication, int ulPINType, int[] pulMaxRetryCount, int[] pulRemainRetryCount, int[] pbDefaultPin) throws RemoteException {
//        int ret;

        if(hApplication <=0
                || pulMaxRetryCount == null || pulMaxRetryCount.length <= 0
                || pulRemainRetryCount == null || pulRemainRetryCount.length <= 0
                || pbDefaultPin == null || pbDefaultPin.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.GetPINInfo(hApplication,ulPINType,pulMaxRetryCount,pulRemainRetryCount,pbDefaultPin);
    }

    @Override
    public int VerifyPIN(long hApplication, int ulPINType, byte[] szPIN, int[] pulRetryCount) throws RemoteException {
//        int ret;

        AccessControlManager accessControlManager = AccessControlManager.getInstance(context);
        int checkRes = accessControlManager.checkByUidPid(getCallingUid(), getCallingPid(), "VerifyPIN");
        if(checkRes != 0){
            Log.i(tag,"checkByUidPid res: " + checkRes);
            return ErrorCode.RET_NO_POWER;
        }

        if(hApplication <=0
                || szPIN == null || szPIN.length <= 0
                || pulRetryCount == null || pulRetryCount.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        int ret = skfApi.VerifyPIN(hApplication,ulPINType,szPIN,pulRetryCount);
        callUnitePin(ret,ulPINType,szPIN,"VerifyPIN");
        return ret;
    }

    @Override
    public int UnblockPIN(long hApplication, byte[] szAdminPIN, byte[] szNewUserPIN, int[] pulRetryCount) throws RemoteException {
//        int ret;

        AccessControlManager accessControlManager = AccessControlManager.getInstance(context);
        int checkRes = accessControlManager.checkByUidPid(getCallingUid(), getCallingPid(), "UnblockPIN");
        if(checkRes != 0){
            Log.i(tag,"checkByUidPid res: " + checkRes);
            return ErrorCode.RET_NO_POWER;
        }

        if(hApplication <=0
                || szAdminPIN == null || szAdminPIN.length <= 0
                || szNewUserPIN == null || szNewUserPIN.length <= 0
                || pulRetryCount == null || pulRetryCount.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        int ret = skfApi.UnblockPIN(hApplication, szAdminPIN, szNewUserPIN, pulRetryCount);
        callUnitePin(ret,0x01,szNewUserPIN,"UnblockPIN");
        return ret;
    }

    @Override
    public int ClearSecureState(long hApplication) throws RemoteException {
        return skfApi.ClearSecureState(hApplication);
    }

    @Override
    public int CreateApplication(long hDev, byte[] szAppName, byte[] szAdminPin, int dwAdminPinRetryCount, byte[] szUserPin, int dwUserPinRetryCount, int dwCreateFileRights, long[] phApplication) throws RemoteException {
        int ret;

        if(szAppName == null || szAppName.length <= 0
                || szAdminPin == null || szAdminPin.length <= 0
                || szUserPin == null || szUserPin.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        return skfApi.CreateApplication(tfDevHandle, szAppName, szAdminPin, dwAdminPinRetryCount, szUserPin, dwUserPinRetryCount, dwCreateFileRights, phApplication);
    }

    @Override
    public int EnumApplication(long hDev, byte[] szAppName, int[] pulSize) throws RemoteException {
        int ret;

        if(szAppName == null
                || pulSize == null || pulSize.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //add 2017年7月6日14:07:34 weizg 如果szAppName长度为0，则赋值为null，只获取长度
        /*if(szAppName.length == 0){
            szAppName = null;
        }*/

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        return skfApi.EnumApplication(tfDevHandle, szAppName, pulSize);
    }

    @Override
    public int DeleteApplication(long hDev, byte[] szAppName) throws RemoteException {
        int ret;

        if(szAppName == null || szAppName.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        return skfApi.DeleteApplication(tfDevHandle, szAppName);
    }

    @Override
    public int OpenApplication(long hDev, byte[] szAppName, long[] phApplication) throws RemoteException {
        int ret;

        if(szAppName == null || szAppName.length <= 0
                || phApplication == null || phApplication.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        return skfApi.OpenApplication(tfDevHandle, szAppName, phApplication);
    }

    @Override
    public int CloseApplication(long hApplication) throws RemoteException {
//        int ret;

        if(hApplication <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.CloseApplication(hApplication);
    }

    @Override
    public int CreateFile(long hApplication, byte[] szFileName, int ulFileSize, int ulReadRights, int ulWriteRights) throws RemoteException {
        //        int ret;

        if(hApplication <= 0
                || szFileName == null || szFileName.length <= 0 || szFileName.length > 32){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.CreateFile(hApplication, szFileName, ulFileSize, ulReadRights, ulWriteRights);
    }

    @Override
    public int DeleteFile(long hApplication, byte[] szFileName) throws RemoteException {
        //        int ret;

        if(hApplication <= 0
                || szFileName == null || szFileName.length <= 0 || szFileName.length > 32){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.DeleteFile(hApplication, szFileName);
    }

    @Override
    public int EnumFiles(long hApplication, byte[] szFileList, int[] pulSize) throws RemoteException {
        //        int ret;

        if(hApplication <= 0
                || szFileList == null
                || pulSize == null || pulSize.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        /*if (szFileList.length == 0) {
            szFileList = null;
        }*/

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.EnumFiles(hApplication, szFileList, pulSize);
    }

    @Override
    public int GetFileInfo(long hApplication, byte[] szFileName, FileAttribute pFileInfo) throws RemoteException {
        //        int ret;

        if(hApplication <= 0
                || szFileName == null || szFileName.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.GetFileInfo(hApplication, szFileName, pFileInfo);
    }

    @Override
    public int ReadFile(long hApplication, byte[] szFileName, int ulOfffset, int ulSize, byte[] pbOutData, int[] pulOutLen) throws RemoteException {
        //        int ret;

        if(hApplication <= 0
                || szFileName == null || szFileName.length <= 0
                || pbOutData == null || pbOutData.length <= 0
                || pulOutLen == null || pulOutLen.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.ReadFile(hApplication, szFileName, ulOfffset, ulSize, pbOutData, pulOutLen);
    }

    @Override
    public int WriteFile(long hApplication, byte[] szFileName, int ulOfffset, byte[] pbData, int ulSize) throws RemoteException {
        //        int ret;

        if(hApplication <= 0
                || szFileName == null || szFileName.length <= 0
                || pbData == null || pbData.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.WriteFile(hApplication, szFileName, ulOfffset, pbData, ulSize);
    }

    @Override
    public int CreateContainer(long hApplication, byte[] szContainerName, long[] pbContainer) throws RemoteException {
        //        int ret;

        if(hApplication <= 0
                || szContainerName == null || szContainerName.length <= 0 || szContainerName.length > 64
                || pbContainer == null || pbContainer.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.CreateContainer(hApplication, szContainerName, pbContainer);
    }

    @Override
    public int DeleteContainer(long hApplication, byte[] szContainerName) throws RemoteException {
        //        int ret;

        if(hApplication <= 0
                || szContainerName == null || szContainerName.length <= 0 || szContainerName.length > 64){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.DeleteContainer(hApplication, szContainerName);
    }

    @Override
    public int EnumContainer(long hApplication, byte[] szContainerName, int[] pulSize) throws RemoteException {

        if(hApplication <= 0
                || szContainerName == null){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        /*if (szContainerName.length == 0) {
            szContainerName = null;
        }*/
        return skfApi.EnumContainer(hApplication, szContainerName, pulSize);
    }

    @Override
    public int OpenContainer(long hApplication, byte[] szContainerName, long[] pbContainer) throws RemoteException {
        //        int ret;

        if(hApplication <= 0
                || szContainerName == null || szContainerName.length <= 0 || szContainerName.length > 64
                || pbContainer == null || pbContainer.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.OpenContainer(hApplication, szContainerName, pbContainer);
    }

    @Override
    public int CloseContainer(long hContainer) throws RemoteException {

        return skfApi.CloseContainer(hContainer);
    }

    @Override
    public int GetContainerType(long hContainer, int[] pulContainerType) throws RemoteException {
        // int ret;

        if(pulContainerType == null || pulContainerType.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        /*ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.GetContainerType(hContainer, pulContainerType);
    }

    @Override
    public int ImportCertificate(long hContainer, int bSignFlag, byte[] pbCert, int ulCertLen) throws RemoteException {
        if(pbCert == null || pbCert.length <= 0
                || ulCertLen <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.ImportCertificate(hContainer, bSignFlag, pbCert, ulCertLen);
    }

    @Override
    public int ExportCertificate(long hContainer, int bSignFlag, byte[] pbCert, int[] pulCertLen) throws RemoteException {
        /*if (pbCert.length == 0) {
            pbCert = null;
        }*/
        return skfApi.ExportCertificate(hContainer, bSignFlag, pbCert, pulCertLen);
    }

    @Override
    public int GenRandom(long hDev, byte[] pbRandom, int ulRandomLen) throws RemoteException {
        int ret;

        if(pbRandom == null || pbRandom.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.GenRandom(tfDevHandle, pbRandom, ulRandomLen);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.GenRandom(tfDevHandle, pbRandom, ulRandomLen);
            }
        }

        return ret;
    }

    @Override
    public int GenExtRSAKey(long hDev, int ulBitsLen, RsaPrivateKeyBlob pBlob) throws RemoteException {
        int ret;

        /*if(pbRandom == null || pbRandom.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }*/

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.GenExtRSAKey(tfDevHandle, ulBitsLen, pBlob);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.GenExtRSAKey(tfDevHandle, ulBitsLen, pBlob);
            }
        }

        return ret;
    }

    @Override
    public int GenRSAKeyPair(long hContainer, int ulBitsLen, RsaPublicKeyBlob pBlob) throws RemoteException {
        /*int ret;

        if(pbRandom == null || pbRandom.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }*/

        return skfApi.GenRSAKeyPair(hContainer, ulBitsLen, pBlob);
    }

    @Override
    public int ImportRSAKeyPair(long hContainer, int ulSymAlgId, byte[] pbWrappedKey, int ulWrappedKeyLen, byte[] pbEncryptedData, int ulEncryptedDataLen) throws RemoteException {
        if(pbWrappedKey == null || pbWrappedKey.length <= 0
                || ulWrappedKeyLen <= 0
                || pbEncryptedData == null || pbEncryptedData.length <= 0
                || ulEncryptedDataLen <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.ImportRSAKeyPair(hContainer, ulSymAlgId, pbWrappedKey, ulWrappedKeyLen, pbEncryptedData, ulEncryptedDataLen);
    }

    @Override
    public int RSASignData(long hContainer, byte[] pbData, int ulDataLen, byte[] pbSignature, int[] pulSignLen) throws RemoteException {
        if(hContainer <= 0
                || pbData == null || pbData.length <= 0
                || ulDataLen <= 0
                || pbSignature == null || pbSignature.length <= 0
                || pulSignLen == null || pulSignLen.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.RSASignData(hContainer, pbData, ulDataLen, pbSignature, pulSignLen);
    }

    @Override
    public int RSAVerify(long hDev, RsaPublicKeyBlob pRSAPubKeyBlob, byte[] pbData, int ulDataLen, byte[] pbSignature, int ulSignLen) throws RemoteException {

        if(pbData == null || pbData.length <= 0
                || ulDataLen <= 0
                || pbSignature == null || pbSignature.length <= 0
                || ulSignLen <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        int ret;
        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.RSAVerify(tfDevHandle, pRSAPubKeyBlob, pbData, ulDataLen, pbSignature, ulSignLen);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.RSAVerify(tfDevHandle, pRSAPubKeyBlob, pbData, ulDataLen, pbSignature, ulSignLen);
            }
        }

        return ret;
    }

    @Override
    public int RSAExportSessionKey(long hContainer, int ulAlgId, RsaPublicKeyBlob pPubKey, byte[] pbData, int[] pulDataLen, long[] phSessionKey) throws RemoteException {
        if(pbData == null || pbData.length <= 0
                || pulDataLen == null || pulDataLen.length <= 0
                || phSessionKey == null || phSessionKey.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.RSAExportSessionKey(hContainer, ulAlgId, pPubKey, pbData, pulDataLen, phSessionKey);
    }

    @Override
    public int ExtRSAPubKeyOperation(long hDev, RsaPublicKeyBlob pRSAPubKeyBlob, byte[] pbInput, int ulInputLen, byte[] pbOutput, int[] pulOutputLen) throws RemoteException {
        if(pRSAPubKeyBlob == null
                || pbInput == null || pbInput.length <= 0
                || ulInputLen <= 0
                || pbOutput == null || pbOutput.length <= 0
                || pulOutputLen == null || pulOutputLen.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        int ret;
        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.ExtRSAPubKeyOperation(tfDevHandle, pRSAPubKeyBlob, pbInput, ulInputLen, pbOutput, pulOutputLen);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.ExtRSAPubKeyOperation(tfDevHandle, pRSAPubKeyBlob, pbInput, ulInputLen, pbOutput, pulOutputLen);
            }
        }

        return ret;
    }

    @Override
    public int ExtRSAPriKeyOperation(long hDev, RsaPrivateKeyBlob pRSAPriKeyBlob, byte[] pbInput, int ulInputLen, byte[] pbOutput, int[] pulOutputLen) throws RemoteException {
        if(pRSAPriKeyBlob == null
                || pbInput == null || pbInput.length <= 0
                || ulInputLen <= 0
                || pbOutput == null || pbOutput.length <= 0
                || pulOutputLen == null || pulOutputLen.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        int ret;
        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.ExtRSAPriKeyOperation(tfDevHandle, pRSAPriKeyBlob, pbInput, ulInputLen, pbOutput, pulOutputLen);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.ExtRSAPriKeyOperation(tfDevHandle, pRSAPriKeyBlob, pbInput, ulInputLen, pbOutput, pulOutputLen);
            }
        }

        return ret;
    }

    @Override
    public int GenECCKeyPair(long hContainer, int ulAlgId, EccPublicKeyBlob pBlob) throws RemoteException {
        if(pBlob == null
                || ulAlgId != SGD_SM2_1){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.GenECCKeyPair(hContainer, ulAlgId, pBlob);
    }

    @Override
    public int ImportECCKeyPair(long hContainer, EnvelopedKeyBlob pEenvlopedKeyBlob) throws RemoteException {
        if(pEenvlopedKeyBlob == null){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.ImportECCKeyPair(hContainer, pEenvlopedKeyBlob);
    }

    @Override
    public int ECCSignData(long hContainer, byte[] pbData, int ulDataLen, EccSignatureBlob pSignature) throws RemoteException {
        if(pbData == null || pbData.length <= 0
                || ulDataLen <= 0
                || pSignature == null){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.ECCSignData(hContainer, pbData, ulDataLen, pSignature);
    }

    @Override
    public int ECCVerify(long hDev, EccPublicKeyBlob pECCPubKeyBlob, byte[] pbData, int ulDataLen, EccSignatureBlob pSignature) throws RemoteException {
        if(pECCPubKeyBlob == null
                || pbData == null || pbData.length <= 0
                || ulDataLen <= 0
                || pSignature == null ){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        int ret;
        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.ECCVerify(tfDevHandle, pECCPubKeyBlob, pbData, ulDataLen, pSignature);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.ECCVerify(tfDevHandle, pECCPubKeyBlob, pbData, ulDataLen, pSignature);
            }
        }

        return ret;
    }

    @Override
    public int ECCExportSessionKey(long hContainer, int ulAlgId, EccPublicKeyBlob pPubKey, EccCipherBlob pData, long[] phSessionKey) throws RemoteException {
        if(hContainer <= 0
                || pPubKey == null
                || phSessionKey == null || phSessionKey.length <= 0
                || pData == null ){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.ECCExportSessionKey(hContainer, ulAlgId, pPubKey, pData, phSessionKey);
    }

    @Override
    public int ExtECCEncrypt(long hDev, EccPublicKeyBlob pECCPubkeyBlob, byte[] pbPlainText, int ulPlainTextLen, EccCipherBlob pCipherText) throws RemoteException {
        if(pECCPubkeyBlob == null
                || pbPlainText == null || pbPlainText.length <= 0
                || ulPlainTextLen <= 0
                || pCipherText == null ){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        int ret;
        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.ExtECCEncrypt(tfDevHandle, pECCPubkeyBlob, pbPlainText, ulPlainTextLen, pCipherText);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.ExtECCEncrypt(tfDevHandle, pECCPubkeyBlob, pbPlainText, ulPlainTextLen, pCipherText);
            }
        }

        return ret;
    }

    @Override
    public int ExtECCDecrypt(long hDev, EccPrivateKeyBlob pECCPriKeyBlob, EccCipherBlob pCipherText, byte[] pbPlainText, int[] pulPlainTextLen) throws RemoteException {
        if(pECCPriKeyBlob == null){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        int ret;
        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        /*if (pbPlainText.length == 0) {
            pbPlainText = null;
        }*/

        ret = skfApi.ExtECCDecrypt(tfDevHandle, pECCPriKeyBlob, pCipherText, pbPlainText, pulPlainTextLen);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.ExtECCDecrypt(tfDevHandle, pECCPriKeyBlob, pCipherText, pbPlainText, pulPlainTextLen);
            }
        }

        return ret;
    }

    @Override
    public int ExtECCSign(long hDev, EccPrivateKeyBlob pECCPriKeyBlob, byte[] pbData, int ulDataLen, EccSignatureBlob pSignature) throws RemoteException {
        if(pECCPriKeyBlob == null
                || pbData == null || pbData.length <= 0
                || ulDataLen <= 0
                || pSignature == null ){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        int ret;
        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.ExtECCSign(tfDevHandle, pECCPriKeyBlob, pbData, ulDataLen, pSignature);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.ExtECCSign(tfDevHandle, pECCPriKeyBlob, pbData, ulDataLen, pSignature);
            }
        }

        return ret;
    }

    @Override
    public int ExtECCVerify(long hDev, EccPublicKeyBlob pECCPubKeyBlob, byte[] pbData, int ulDataLen, EccSignatureBlob pSignature) throws RemoteException {
        if(pECCPubKeyBlob == null
                || pbData == null || pbData.length <= 0
                || ulDataLen <= 0
                || pSignature == null ){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        int ret;
        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.ExtECCVerify(tfDevHandle, pECCPubKeyBlob, pbData, ulDataLen, pSignature);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.ExtECCVerify(tfDevHandle, pECCPubKeyBlob, pbData, ulDataLen, pSignature);
            }
        }

        return ret;
    }

    @Override
    public int GenerateAgreementDataWithECC(long hContainer, int ulAlgId, EccPublicKeyBlob pTempECCPubKeyBlob, byte[] pbID, int ulIDLen, long[] phAgreementHandle) throws RemoteException {
        if(hContainer <= 0
                ||pTempECCPubKeyBlob == null
                || pbID == null || pbID.length <= 0
                || ulIDLen <= 0
                || phAgreementHandle == null || phAgreementHandle.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.GenerateAgreementDataWithECC(hContainer, ulAlgId, pTempECCPubKeyBlob, pbID, ulIDLen, phAgreementHandle);
    }

    @Override
    public int GenerateAgreementDataAndKeyWithECC(long hContainer, int ulAlgId, EccPublicKeyBlob pSponsorECCPubKeyBlob, EccPublicKeyBlob pSponsorTempECCPubKeyBlob, EccPublicKeyBlob pTempECCPubKeyBlob, byte[] pbID, int ulIDLen, byte[] pbSponsorID, int ulSponsorIDLen, long[] phKeyHandle) throws RemoteException {
        if(hContainer <= 0
                ||pTempECCPubKeyBlob == null
                ||pSponsorTempECCPubKeyBlob == null
                || pbID == null || pbID.length <= 0
                || ulIDLen <= 0
                || ulIDLen > 32
                || pbSponsorID == null || pbSponsorID.length <= 0
                || ulSponsorIDLen <= 0
                || ulSponsorIDLen > 32
                || phKeyHandle == null || phKeyHandle.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.GenerateAgreementDataAndKeyWithECC(hContainer, ulAlgId, pSponsorECCPubKeyBlob, pSponsorTempECCPubKeyBlob, pTempECCPubKeyBlob, pbID, ulIDLen, pbSponsorID, ulSponsorIDLen, phKeyHandle);
    }

    @Override
    public int GenerateKeyWithECC(long hAgreementHandle, EccPublicKeyBlob pECCPubKeyBlob, EccPublicKeyBlob pTempECCPubKeyBlob, byte[] pbID, int ulIDLen, long[] phKeyHandle) throws RemoteException {
        if(hAgreementHandle <= 0
                || pbID == null || pbID.length <= 0
                || ulIDLen <= 0
                || ulIDLen > 32
                || phKeyHandle == null || phKeyHandle.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.GenerateKeyWithECC(hAgreementHandle, pECCPubKeyBlob, pTempECCPubKeyBlob, pbID, ulIDLen, phKeyHandle);
    }

    @Override
    public int ExportPublicKey(long hContainer, int bSignFlag, byte[] pbBlob, int[] pulBlobLen) throws RemoteException {
        if(hContainer <= 0 || pulBlobLen == null){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        /*if (pbBlob.length == 0) {
            pbBlob = null;
        }*/
        return skfApi.ExportPublicKey(hContainer, bSignFlag, pbBlob, pulBlobLen);
    }

    @Override
    public int ImportSessionKey(long hContainer, int ulAlgId, byte[] pbWrapedData, int ulWrapedLen, long[] phKey) throws RemoteException {
        if(hContainer <= 0
                || pbWrapedData == null || pbWrapedData.length <= 0
                || ulWrapedLen <= 0
                || phKey == null || phKey.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.ImportSessionKey(hContainer, ulAlgId, pbWrapedData, ulWrapedLen, phKey);
    }

    @Override
    public int SetSymmKey(long hDev, byte[] pbKey, int ulAlgID, long[] phKey) throws RemoteException {
        // TODO: 2017/6/7 接口意义
        if(pbKey == null || pbKey.length <= 0
                || phKey == null || phKey.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        int ret;
        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.SetSymmKey(tfDevHandle, pbKey, ulAlgID, phKey);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.SetSymmKey(tfDevHandle, pbKey, ulAlgID, phKey);
            }
        }

        return ret;
    }

    @Override
    public int EncryptInit(long hKey, BlockCipherParam EncryptParam) throws RemoteException {
        if(EncryptParam == null){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.EncryptInit(hKey, EncryptParam);
    }

    @Override
    public int Encrypt(long hKey, byte[] pbData, int ulDataLen, byte[] pbEncryptedData, int[] pulEncryptedLen) throws RemoteException {
        if(pbData == null || pbData.length <= 0
                || pulEncryptedLen == null ){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        /*if (pbEncryptedData.length == 0) {
            pbEncryptedData = null;
        }*/
        return skfApi.Encrypt(hKey, pbData, ulDataLen, pbEncryptedData, pulEncryptedLen);
    }

    @Override
    public int EncryptUpdate(long hKey, byte[] pbData, int ulDataLen, byte[] pbEncryptedData, int[] pulEncryptedLen) throws RemoteException {
        if(pbData == null || pbData.length <= 0
                || pulEncryptedLen == null ){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.EncryptUpdate(hKey, pbData, ulDataLen, pbEncryptedData, pulEncryptedLen);
    }

    @Override
    public int EncryptFinal(long hKey, byte[] pbEncryptedData, int[] pulEncryptedLen) throws RemoteException {
        if(pulEncryptedLen == null ){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.EncryptFinal(hKey, pbEncryptedData, pulEncryptedLen);
    }

    @Override
    public int DecryptInit(long hKey, BlockCipherParam EncryptParam) throws RemoteException {
        if(EncryptParam == null){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.DecryptInit(hKey, EncryptParam);
    }

    @Override
    public int Decrypt(long hKey, byte[] pbEncryptedData, int ulEncryptedLen, byte[] pbData, int[] pulDataLen) throws RemoteException {
        if(pbEncryptedData == null || pbEncryptedData.length <= 0
                || ulEncryptedLen <= 0 ){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        /*if (pbData.length == 0) {
            pbData = null;
        }*/
        return skfApi.Decrypt(hKey, pbEncryptedData, ulEncryptedLen, pbData, pulDataLen);
    }

    @Override
    public int DecryptUpdate(long hKey, byte[] pbEncryptedData, int ulEncryptedLen, byte[] pbData, int[] pulDataLen) throws RemoteException {
        if(pbEncryptedData == null || pbEncryptedData.length <= 0
                || ulEncryptedLen <= 0
                || pbData == null || pbData.length <= 0
                || pulDataLen == null || pulDataLen.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.DecryptUpdate(hKey, pbEncryptedData, ulEncryptedLen, pbData, pulDataLen);
    }

    @Override
    public int DecryptFinal(long hKey, byte[] pbDecryptedData, int[] pulDecryptedDataLen) throws RemoteException {
        /*if (pbDecryptedData.length == 0){
            pbDecryptedData = null;
        }*/

        return skfApi.DecryptFinal(hKey, pbDecryptedData, pulDecryptedDataLen);
    }

    @Override
    public int DigestInit(long hDev, int ulAlgID, EccPublicKeyBlob pPubKey, byte[] pucID, int ulIDLen, long[] phHash) throws RemoteException {
        if(ulAlgID == SkfApiCode.SGD_SM3 &&
                (pPubKey == null
                || pucID == null || pucID.length <= 0)){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        if(phHash == null || phHash.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        int ret;
        //如果当前未打开设备，则首先打开设备，然后执行业务
        ret = checkTfDevHandle();
        if(ret != 0){
            return ret;
        }

        ret = skfApi.DigestInit(tfDevHandle, ulAlgID, pPubKey, pucID, ulIDLen, phHash);

        if(ret == -1){
            //如果返回失败未找到，则重新打开设备后，再次调用接口
            ret = reOpenTFDev();
            if(ret == 0){
                ret = skfApi.DigestInit(tfDevHandle, ulAlgID, pPubKey, pucID, ulIDLen, phHash);
            }
        }

        return ret;
    }

    @Override
    public int Digest(long hHash, byte[] pbData, int ulDataLen, byte[] phHashData, int[] pulHashLen) throws RemoteException {
        if(pbData == null || pbData.length <= 0
                        || pulHashLen == null || pulHashLen.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        /*if (phHashData.length == 0) {
            phHashData = null;
        }*/
        return skfApi.Digest(hHash, pbData, ulDataLen, phHashData, pulHashLen);
    }

    @Override
    public int DigestUpdate(long hHash, byte[] pbData, int ulDataLen) throws RemoteException {
        if(pbData == null){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.DigestUpdate(hHash, pbData, ulDataLen);
    }

    @Override
    public int DigestFinal(long hHash, byte[] pHashData, int[] pulHashLen) throws RemoteException {

        /*if (pHashData.length == 0) {
            pHashData = null;
        }*/
        return skfApi.DigestFinal(hHash, pHashData, pulHashLen);
    }

    @Override
    public int MacInit(long hKey, BlockCipherParam pMacParam, long[] phMac) throws RemoteException {
        if(phMac == null || phMac.length <= 0
                || pMacParam == null){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }


        return skfApi.MacInit(hKey, pMacParam, phMac);
    }

    @Override
    public int Mac(long hMac, byte[] pbData, int ulDataLen, byte[] pbMacData, int[] pulMacLen) throws RemoteException {
        /*if (pbMacData.length == 0) {
            pbMacData = null;
        }*/
        return skfApi.Mac(hMac, pbData, ulDataLen, pbMacData, pulMacLen);
    }

    @Override
    public int MacUpdate(long hMac, byte[] pbData, int ulDataLen) throws RemoteException {
        if(pbData == null || pbData.length <= 0
                || ulDataLen <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        return skfApi.MacUpdate(hMac, pbData, ulDataLen);
    }

    @Override
    public int MacFinal(long hMac, byte[] pbMacData, int[] pulMacDataLen) throws RemoteException {
        /*if (pbMacData.length == 0) {
            pbMacData = null;
        }*/
        return skfApi.MacFinal(hMac, pbMacData, pulMacDataLen);
    }

    @Override
    public int CloseHandle(long hHandle) throws RemoteException {
        return skfApi.CloseHandle(hHandle);
    }

    /**
     * 注册监听
     *
     * @param listener 监听者
     */
    @Override
    public void regist(IChipParamChangeListener listener) {
        // TODO: 2017/6/7  未实现
    }

    /**
     * 取消注册监听
     *
     * @param listener 监听者
     */
    @Override
    public void unRegist(IChipParamChangeListener listener) {
        // TODO: 2017/6/7  未实现
    }

    @Override
    public int XdjaUnlockPin(int role, byte[] key, byte[] newPin) throws RemoteException {

        if(key == null || key.length <= 0
                || newPin == null || newPin.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //接口调用者权限校验
        AccessControlManager accessControlManager = AccessControlManager.getInstance(context);
        int checkRes = accessControlManager.checkByUidPid(getCallingUid(), getCallingPid(), "XdjaUnlockPin");
        if(checkRes != 0){
            Log.i(tag,"checkByUidPid res: " + checkRes);
            return ErrorCode.RET_NO_POWER;
        }

        //modify 2017年7月28日15:11:10 weizg
        //该接口非国密接口，句柄需要使用原始卡句柄
        final long[] devHandle = new long[1];
        int ret = jniapi.UnlockPIN(xdjaTFDevHandle, role, key, key.length, newPin, newPin.length);
        if(ret == -1){
            //如果未打开，则重新打开一次
            ret = jniOpenTFDev(devHandle);
            if(ret == 0){
                xdjaTFDevHandle = devHandle[0];
                ret = jniapi.UnlockPIN(xdjaTFDevHandle, role, key, key.length, newPin, newPin.length);
            }
        }

        //调用统一PIN码接口，保存更新
        callUnitePin(ret,role,newPin,"XdjaUnlockPin");

        return ret;
    }

    @Override
    public int XdjaUnlockPinUseUsn(int role, byte[] usn, byte[] usnlockCode, byte[] newPin) throws RemoteException {

        if(usn == null || usn.length <= 0
                || newPin == null || newPin.length <= 0
                || usnlockCode == null || usnlockCode.length <= 0){
            return SkfApiCode.SAR_INVALIDPARAMERR;
        }

        //接口调用者权限校验
        AccessControlManager accessControlManager = AccessControlManager.getInstance(context);
        int checkRes = accessControlManager.checkByUidPid(getCallingUid(), getCallingPid(), "XdjaUnlockPinUseUsn");
        if(checkRes != 0){
            Log.i(tag,"checkByUidPid res: " + checkRes);
            return ErrorCode.RET_NO_POWER;
        }
        JNIAPI jniapi = new JNIAPI();

        //modify 2017年7月28日15:17:33 weizg
        //该接口非国密接口，句柄需要使用原始卡句柄
        final long[] devHandle = new long[1];
        int ret = JNIPinManager.getInstance().XdjaUnlockPinUseUsn(jniapi,xdjaTFDevHandle,role, usn, usnlockCode, newPin);
        if(ret == -1){
            //如果句柄无效，未打开，则重新打开一次
            ret = jniOpenTFDev(devHandle);
            if(ret == 0){
                xdjaTFDevHandle = devHandle[0];
                ret = JNIPinManager.getInstance().XdjaUnlockPinUseUsn(jniapi,xdjaTFDevHandle,role, usn, usnlockCode, newPin);
            }
        }

        //调用统一PIN码接口，保存更新
        callUnitePin(ret,role,newPin,"XdjaUnlockPinUseUsn");

        return ret;

        //打开TF卡类型的安全芯片
        /*int ret = jniapi.OpenDevByNameEx(context.getPackageName().getBytes(), devHandle);

        if(ret == 0){
            ret = JNIPinManager.getInstance().XdjaUnlockPinUseUsn(devHandle[0],role, usn, usnlockCode, newPin);
        }

        return ret;*/
    }

    @Override
    public String XdjaGetCosInfo() throws RemoteException {
        //modify 2017年7月11日11:30:41 weizg
        //该接口非国密接口，句柄需要使用原始卡句柄;
        //同时，首先使用缓存句柄尝试打开，如果失败再次打开卡
        final long[] devHandle = new long[1];
        XDJA_DEVINFO devinfo = new XDJA_DEVINFO();
        int ret = jniapi.GetDevInfo(xdjaTFDevHandle, devinfo);
        if(ret == -1){
            ret = jniOpenTFDev(devHandle);
            if(ret == 0){

                xdjaTFDevHandle = devHandle[0];

                ret = jniapi.GetDevInfo(devHandle[0], devinfo);
            }
        }

        if(ret == 0){
            Log.i(tag,"XdjaGetCosInfo getDevInof Ret: " + ret);
            return JNIPinManager.getInstance().getCosVer(devinfo);
        }

        return "";

        //打开TF卡类型的安全芯片
        /*int ret = jniapi.OpenDevByNameEx(context.getPackageName().getBytes(), devHandle);
        if (ret ==0){
            XDJA_DEVINFO devinfo = new XDJA_DEVINFO();
            int getDevInfoRet = jniapi.GetDevInfo(devHandle[0], devinfo);
            Log.i(tag,"XdjaGetCosInfo getDevInof Ret: " + getDevInfoRet);
            return JNIPinManager.getInstance().getCosVer(devinfo);
        }
        return "";*/
    }

    /**
     * 得到卡号
     * @return 卡号
     */
    private String getCardId(){
        DevInfo devInfo = new DevInfo();
        try {
            int ret = GetDevInfo(0,devInfo);
            if (ret == 0){
                return new String(devInfo.serialNumber);
            }
        } catch (RemoteException e) {
            e.printStackTrace();
        }
        return "";
    }

    /**
     * 获取包名
     * @param callingUid uid
     * @param callingPid pid
     * @return uid,pid对应的包名
     */
    private String getPkgName(int callingUid,int callingPid){
        PackageManager pm = context.getPackageManager();
        String[] pkgList = pm.getPackagesForUid(callingUid);
        if(pkgList != null && pkgList.length > 0){
            return pkgList[0];
        }else {
            Log.i(tag,"TFBinder getPkgName null, uid: " + callingUid);
            return "";
        }
    }

    /**
     * 调用统一PIN码，将结果告之
     * @param ret 结果
     * @param role 角色
     * @param pin PIN码
     * @param method 方法名
     */
    private void callUnitePin(int ret,int role,byte[] pin,String method) {
        String cardId = getCardId();
        InsideMultiChipUnitePinManager.getInstance().handlePinResult(context,
                cardId,
                role,
                pin,
                ret,
                method,
                getCallingUid(),
                getCallingPid(),
                "SkfApi");
    }
}
