package com.xdja.multichip;

import android.Manifest;
import android.bluetooth.BluetoothAdapter;
import android.content.BroadcastReceiver;
import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.content.pm.ProviderInfo;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.RemoteException;
import android.os.SystemClock;
import android.text.TextUtils;
import android.util.Log;

import com.xdja.jar.forcallprovider.ForCallProvider;
import com.xdja.multichip.param.JniApiParam;
import com.xdja.multichip.process.bluetooth.BluetoothKeyProcessProvider;
import com.xdja.multichip.process.bluetooth.BluetoothKeyProcessService;
import com.xdja.multichip.process.board.OnboardProcessProvider;
import com.xdja.multichip.process.coveredcard.CCProcessProvider;
import com.xdja.multichip.process.tfcard.TFProcessProvider;
import com.xdja.multichip.utils.PermissionUtil;
import com.xdja.multichip.utils.SharedPreferencesUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/5/26 <br/>
 * <p>
 * 此类是一个Binder，用于跟各个卡操作进程交互，并向Jar包提供服务。
 */

public class GetMultiJniApiBinder extends IGetMultiJniApi.Stub {

    public static final String KEY_BINDER_BINDER = "Binder";
    public static final String KEY_PARCELABLE_CHIPPARAM = "JniApiParam";

    private static final String NAME_PROCESS_ONBOARD = "onboard";
    private static final String NAME_PROCESS_TF = "TF";
    private static final String NAME_PROCESS_CC = "CC";
    private static final String NAME_PROCESS_BLUETOOTH = "bluetooth";

    private static HashMap<String, String> providerNameMap = new HashMap<>();

    /**
     * 等待超时时间
     */
    private static final int WAIT_TIMEOUT = 10;

    /**
     * 设备上有芯片的列表（MODEL）
     */
    private static ArrayList<String> hasChipDevList = new ArrayList<>();
    /**
     * 没有TF卡的设备列表
     */
    private static ArrayList<String> hasNotTFDevList = new ArrayList<>();

    static {
        hasChipDevList.add("ACTOMA ACE");
        hasChipDevList.add("HUAWEI eH880");

        hasNotTFDevList.add("ACTOMA ACE");

        providerNameMap.put(NAME_PROCESS_ONBOARD, OnboardProcessProvider.class.getName());
        providerNameMap.put(NAME_PROCESS_TF, TFProcessProvider.class.getName());
        providerNameMap.put(NAME_PROCESS_BLUETOOTH, BluetoothKeyProcessProvider.class.getName());
        providerNameMap.put(NAME_PROCESS_CC, CCProcessProvider.class.getName());
    }

    private static GetMultiJniApiBinder instance;
    private Context context;

    /**
     * 线程池
     */
    private ExecutorService cachedThreadPool = Executors.newCachedThreadPool();

    /**
     * 上次调用方法{@link #getMultiJniApi()}的时间
     */
    private long lastCall = 0;

    private GetMultiJniApiBinder(Context context) {
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }
    }

    /**
     * 得到实例;
     * 单例。
     *
     * @param context
     * @return
     */
    public static GetMultiJniApiBinder getInstance(Context context) {
        if (instance == null) {
            synchronized (GetMultiJniApiBinder.class) {
                if (instance == null) {
                    instance = new GetMultiJniApiBinder(context);
                }
            }
        }
        return instance;
    }

    List<Bundle> lastRusult = new ArrayList<>();

    /**
     * 得到卡信息；
     *
     * @return 一个Bundle列表；其中单个Bundle中包含两个数据，
     * 关键字分别为：{@link #KEY_BINDER_BINDER},{@link #KEY_PARCELABLE_CHIPPARAM}
     * @throws RemoteException
     */
    @Override
    public synchronized List<Bundle> getMultiJniApi() throws RemoteException {
        Log.w("GetAll", " getMultiJniApi start: ");

        long curCall = SystemClock.elapsedRealtime();
        if ((curCall - lastCall) < 1000) {
            return lastRusult;
        }
        lastRusult.clear();
        List<String> chipProcessList = getChipProcessList();
        final CountDownLatch latch = new CountDownLatch(chipProcessList.size());

        for (final String chipProcess : chipProcessList) {
            final String name = convertToProviderName(chipProcess);
            if (TextUtils.isEmpty(name)) {
                latch.countDown();
                continue;
            }
            cachedThreadPool.execute(new Runnable() {
                @Override
                public void run() {
                    Uri uri = Uri.parse("content://" + name);
                    ContentResolver resolver = context.getContentResolver();
                    Bundle bundleResult = resolver.call(uri, "GetJniApiBinder", "", null);
                    if (bundleResult != null) {
                        int ret = bundleResult.getInt("ret");
                        if (ret == 0) {
                            ArrayList<Bundle> bundle = bundleResult.getParcelableArrayList("result");

                            if (bundle != null) {
                                for (Bundle singleBundle : bundle) {
                                    singleBundle.setClassLoader(JniApiParam.class.getClassLoader());
                                    if (singleBundle.getParcelable(KEY_PARCELABLE_CHIPPARAM) != null) {
                                        lastRusult.add(singleBundle);
                                    }
                                }
                            }
                        }
                    }
                    latch.countDown();
                }
            });
        }
        try {
            latch.await(WAIT_TIMEOUT, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        lastCall = SystemClock.elapsedRealtime();

        Log.w("GetAll", " getMultiJniApi end: ");

        saveJniApiParam(lastRusult);
        return lastRusult;
    }


    /**
     * 得到 操作芯片进程列表的list
     *
     * @return
     */
    private List<String> getChipProcessList() {
        ArrayList<String> result = new ArrayList<>();
        result.addAll(getOnboardChipProcessList());
        result.addAll(getTFChipProcessList());
        result.addAll(getBluekeyChipProcessList());

        //add by xwy on 2017-9-14 15:10:01 添加贴膜卡的支持
        result.addAll(getCoveredCardProcessList());
        return result;
    }

    /**
     * 存在
     */
    private final static int FLAG_ONBOARD_CHIP_EXIST = 1;
    /**
     * 不存在
     */
    private final static int FLAG_ONBOARD_CHIP_NOT_EXIST = -1;
    /**
     * 未知
     */
    private final static int FLAG_ONBOARD_CHIP_UNKNOW = 0;
    /**
     * 板载芯片是否存在标志位
     */
    private int onboardChipIsExist = FLAG_ONBOARD_CHIP_UNKNOW;

    /**
     * 返回操作板载芯片进程的进程名,如果手机上没有芯片，则返回size为0的进程名
     * <p>应该从多种维度来实现这个方法：
     * <p>1.如果ACE、MATE8、海信等手机，应该需要有
     * <p>2.如果不是以上的手机，则应该在{@link #onboardChipIsExist}为{@link #FLAG_ONBOARD_CHIP_UNKNOW}时，
     * 再次进行判断
     *
     * @return
     */
    private synchronized List<String> getOnboardChipProcessList() {
        ArrayList<String> list = new ArrayList<>();
        //标志位是不存在，则直接返回
        if (onboardChipIsExist == FLAG_ONBOARD_CHIP_NOT_EXIST) {
            return list;
        }
        //标志位是存在，标明有芯片
        else if (onboardChipIsExist == FLAG_ONBOARD_CHIP_EXIST) {
            list.add(NAME_PROCESS_ONBOARD);
        }
        //标志位是未知，需要进行判断，如果当前设备在已知列表中，则将标志位置为FLAG_ONBOARD_CHIP_EXIST
        else if (onboardChipIsExist == FLAG_ONBOARD_CHIP_UNKNOW) {
            boolean flag = checkCurDevIsKownHasChip();
            if (flag) {
                onboardChipIsExist = FLAG_ONBOARD_CHIP_EXIST;
            }
            list.add(NAME_PROCESS_ONBOARD);
        }
        return list;
    }

    /**
     * 只有在hasNotTFDevList中设备，才不返回TF卡操作进程名
     *
     * @return
     */
    private List<String> getTFChipProcessList() {
        String model = Build.MODEL;
        ArrayList<String> result = new ArrayList<>();
        if (hasNotTFDevList.contains(model)) {
            return result;
        }
        result.add(NAME_PROCESS_TF);
        return result;
    }

    /**
     * 得到贴膜卡的进程名
     *
     * @return
     */
    private List<String> getCoveredCardProcessList() {
        ArrayList<String> result = new ArrayList<>();

        List<String> permissions = new ArrayList<>();
        permissions.add(Manifest.permission.READ_SMS);
        permissions.add(Manifest.permission.SEND_SMS);
        permissions.add(Manifest.permission.RECEIVE_SMS);
        //permissions.add(Manifest.permission.BROADCAST_SMS);

        permissions.add(Manifest.permission.READ_CONTACTS);
        permissions.add(Manifest.permission.WRITE_CONTACTS);

        List<String> deniedPermissions = PermissionUtil.getDeniedPermissions(context, permissions);
        /*Map<String,Boolean> map = PermissionUtil.getPermissionState(context,permissions);
        Set<String> i = map.keySet();
        for (String s:i) {
            Log.e("xxxxxxxxx",s+"  "+map.get(s));

        }
        Log.e("xxxxxxxxx",deniedPermissions.size()+"");*/
        if (deniedPermissions.size() == 0) {
            result.add(NAME_PROCESS_CC);
        }
        //result.add(NAME_PROCESS_CC);

        return result;
    }

    /**
     * 根据蓝牙状态来判断是否返回；
     * 如果蓝牙没有打开，则返回的size为0
     *
     * @return
     */
    private List<String> getBluekeyChipProcessList() {
        regeditBluetoothListener();
        ArrayList<String> result = new ArrayList<>();
        //start : update by xwy on 2017-9-12 14:21:56
        //解决中兴有一个手机上 java.lang.SecurityException: Need BLUETOOTH permission:的问题
        //如果出现异常直接返回一个空的结果。
        try {
            BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();
            int state = adapter.getState();
            if (state == BluetoothAdapter.STATE_OFF || state == BluetoothAdapter.STATE_TURNING_OFF) {
                return result;
            }
            result.add(NAME_PROCESS_BLUETOOTH);

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            return result;
        }
        //end : update by xwy on 2017-9-12 14:21:56
    }

    /**
     * 判断当前设备是否含有芯片
     *
     * @return 如果含有芯片，则返回true，没有则返回false
     */
    private boolean checkCurDevIsKownHasChip() {
        boolean flag = false;
        String model = Build.MODEL;
        if (hasChipDevList.contains(model)) {
            flag = true;
        }
        return flag;
    }

    /**
     * 蓝牙状态监听标志位;
     * 0：表示还没有开启监听；
     * 其他：已开启
     */
    private int bluetoothFlag = 0;

    private synchronized void regeditBluetoothListener() {
        if (bluetoothFlag != 0) {
            return;
        }
        bluetoothFlag = 1;
        IntentFilter filter = new IntentFilter();
        filter.addAction(BluetoothAdapter.ACTION_STATE_CHANGED);
        context.registerReceiver(bluetoothReceiver, filter);
    }

    /**
     * 蓝牙监听广播
     */
    private BroadcastReceiver bluetoothReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            int blueState = intent.getIntExtra(BluetoothAdapter.EXTRA_STATE, 0);
            if (BluetoothAdapter.STATE_OFF == blueState) {
                Intent stopIntent = new Intent();
                stopIntent.setClassName(context.getPackageName(), BluetoothKeyProcessService.class.getName());
                context.stopService(stopIntent);
            }
        }
    };

    private String convertToProviderName(String name) {
        if (TextUtils.isEmpty(name)) {
            return "";
        }
        String pkgName = ForCallProvider.getCallPkg(context);
        if (TextUtils.isEmpty(pkgName)) {
            return providerNameMap.get(name);
        }
        PackageManager pm = context.getPackageManager();
        ComponentName cn = new ComponentName(pkgName, providerNameMap.get(name));
        String authority = "";
        try {
            ProviderInfo pi = pm.getProviderInfo(cn, PackageManager.GET_META_DATA);
            authority = pi.authority;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        if (TextUtils.isEmpty(authority)) {
            return providerNameMap.get(name);
        }
        return authority;
    }

    private void saveJniApiParam(List<Bundle> list) {
        final List<Bundle> saveList = new ArrayList<>();
        saveList.addAll(list);
        cachedThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                for (Bundle bundle : saveList) {
                    JniApiParam param = bundle.getParcelable(KEY_PARCELABLE_CHIPPARAM);
                    if (param != null) {
                        SharedPreferencesUtil.saveCardIdAndType(context, param.cardId, param.chipType);
                    }
                }
            }
        });
    }
}


