package com.xdja.multichip;

import android.content.ComponentName;
import android.content.Context;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.ProviderInfo;
import android.content.pm.ServiceInfo;
import android.os.Bundle;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.Parcel;
import android.os.Process;
import android.os.RemoteException;
import android.os.SystemClock;
import android.text.TextUtils;
import android.util.Log;

import com.xdja.multichip.anno.AutoWriteList;
import com.xdja.multichip.jniapi.JarMultiJniApiErrorCode;
import com.xdja.multichip.param.IResultCallback;
import com.xdja.multichip.param.JniApiParam;
import com.xdja.multichip.param.ParamKeywords;
import com.xdja.multichip.process.IChipForCall;
import com.xdja.multichip.process.board.OnboardChipForCall;
import com.xdja.multichip.process.coveredcard.CCChipForCall;
import com.xdja.multichip.process.tfcard.TFChipForCall;
import com.xdja.multichip.process.vhsm.net.NetVhsmChipForCall;
import com.xdja.multichip.utils.SharedPreferencesUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/5/26 <br/>
 * <p>
 * 此类是一个Binder，用于跟各个卡操作进程交互，并向Jar包提供服务。
 */

public class GetMultiJniApiBinder extends IGetMultiJniApi.Stub {

    private static final String TAG = "GetMultiJniApiBinder";

    /**
     * 等待超时时间(秒)
     */
    private static final int WAIT_TIMEOUT = 10;

    private static GetMultiJniApiBinder instance;
    private Context context;

    /**
     * 线程池
     */
    private ExecutorService cachedThreadPool = Executors.newCachedThreadPool();

    /**
     * 上次调用方法{@link #getMultiJniApi()}的时间
     */
    private long lastCall = 0;

    private GetMultiJniApiBinder(Context context) {
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }
        init(this.context);
    }

    @AutoWriteList
    private ArrayList<IChipForCall> chipForCallList = new ArrayList<>();

    private void init(Context context) {
        chipForCallList.clear();
        //modify zhangxiaolong 2020-10-22 调整顺序，让慢的在前面；去掉无用代码；去掉蓝牙key(因为一直没有用)；
        chipForCallList.add(new CCChipForCall());
        chipForCallList.add(new TFChipForCall());
        chipForCallList.add(new OnboardChipForCall());
        chipForCallList.add(new NetVhsmChipForCall());
        for (IChipForCall chipForCall : chipForCallList) {
            chipForCall.setContext(context);
        }
    }

    /**
     * 得到实例;
     * 单例。
     *
     * @param context
     * @return
     */
    public static GetMultiJniApiBinder getInstance(Context context) {
        if (instance == null) {
            synchronized (GetMultiJniApiBinder.class) {
                if (instance == null) {
                    instance = new GetMultiJniApiBinder(context);
                }
            }
        }
        return instance;
    }

    List<Bundle> lastResult = new ArrayList<>();

    @Override
    public boolean onTransact(int code, Parcel data, Parcel reply, int flags) throws RemoteException {
        //modify zhangxiaolong 2020-10-22 去掉原来的限制
        return super.onTransact(code, data, reply, flags);
    }

    /**
     * 得到卡信息；
     *
     * @return 一个Bundle列表；其中单个Bundle中包含两个数据，
     * 关键字分别为：{@link ParamKeywords#KEY_Binder_Binder},{@link ParamKeywords#KEY_Parcelable_JniApiParam}
     * @throws RemoteException
     */
    @Override
    public synchronized List<Bundle> getMultiJniApi() throws RemoteException {
        Log.w(TAG, "getMultiJniApi start: ");

        long curCall = SystemClock.elapsedRealtime();
        if ((curCall - lastCall) < 1000) {
            return lastResult;
        }
        lastResult.clear();

        //modify zhangxiaolong 2020-10-22 去掉了原来的代码，因为已经从chipForCallList中去掉了不需要的类型

        lastResult.addAll(getJniApiBinder(chipForCallList));
        lastCall = SystemClock.elapsedRealtime();

        Log.w(TAG, " getMultiJniApi end: ");

        saveJniApiParam(lastResult);
        return lastResult;
    }

    private static final String TAG_CHIPFORCALL = "chipForCall: ";

    /**
     * 调用卡操作进程
     *
     * @param chipForCallList
     * @return
     */
    private ArrayList<Bundle> getJniApiBinder(ArrayList<IChipForCall> chipForCallList) {
        final ArrayList<Bundle> resultList = new ArrayList<>();
        final CountDownLatch latch = new CountDownLatch(chipForCallList.size());
        for (final IChipForCall chipForCall : chipForCallList) {
            cachedThreadPool.execute(new Runnable() {
                @Override
                public void run() {
                    try {
                        if (!judgeChipCanUse(chipForCall)) {
                            Log.e(TAG, TAG_CHIPFORCALL + chipForCall.getChipName() + " false");
                            return;
                        }
                        Log.e(TAG, TAG_CHIPFORCALL + chipForCall.getChipName() + " true");
                        List<Bundle> bundles = chipForCall.getJniApiBinder();
                        if (bundles == null) {
                            Log.e(TAG, TAG_CHIPFORCALL + chipForCall.getChipName() + " bundles = null");
                            return;
                        }
                        Log.e(TAG, TAG_CHIPFORCALL + chipForCall.getChipName() + " bundles size:" + bundles.size());
                        for (Bundle singleBundle : bundles) {
                            singleBundle.setClassLoader(JniApiParam.class.getClassLoader());
                            if (singleBundle.getParcelable(ParamKeywords.KEY_Parcelable_JniApiParam) != null) {
                                resultList.add(singleBundle);
                            }
                        }
                    } finally {
                        latch.countDown();
                    }
                }
            });
        }
        try {
            latch.await(WAIT_TIMEOUT, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        return resultList;
    }


    @Override
    public Bundle callMethod(Bundle bundle) throws RemoteException {
        if (bundle == null) {
            return null;
        }
        if (bundle.containsKey(ParamKeywords.KEY_String_method)) {
            String method = bundle.getString(ParamKeywords.KEY_String_method);
            if (ParamKeywords.KEY_METHOD_makeByType.equals(method)) {
                return makeByType(bundle);
            } else if (ParamKeywords.KEY_METHOD_setSingleChipCanUseFlag.equals(method)) {
                return setSingleChipCanUse(bundle);
            } else if (ParamKeywords.KEY_METHOD_getSingleChipCanUseFlag.equals(method)) {
                return getSingleChipCanUseFlag(bundle);
            } else if (ParamKeywords.KEY_METHOD_SPICONFIG.equals(method)) {
                return setChipSpiConfig(bundle);
            } else if (ParamKeywords.KEY_METHOD_GetAllAsync.equals(method)) {
                return getAllAsync(bundle);
            } else if (ParamKeywords.KEY_METHOD_CcSetLogFlag.equals(method)) {
                return handleCcSetLogFlag(bundle);
            }
            //modify zhangxiaolong 2020-10-23 去掉支持此方法 KEY_METHOD_CreateDefaultVhsm(单机版VHSM)
        }
        Bundle result = new Bundle();
        result.putInt(ParamKeywords.KEY_int_ret, JarMultiJniApiErrorCode.NO_SUCH_METHOD);
        return result;
    }

    /**
     * 根据类型获取卡代理<br>
     * 返回值中的Bundle，有两个关键字：{@link ParamKeywords#KEY_int_ret},{@link ParamKeywords#KEY_ParcelableArrayList_Bundle}
     * 当关键字{@link ParamKeywords#KEY_int_ret}为0时，表示正常调用，但另一个关键字(值是一个list)的size仍有可能为0。
     * 而{@link ParamKeywords#KEY_ParcelableArrayList_Bundle}的值类型是List(Bundle),
     * 而这里面的Bundle也有两个关键字：{@link ParamKeywords#KEY_Binder_Binder},{@link ParamKeywords#KEY_Parcelable_JniApiParam}.
     *
     * @param callBundle
     * @return
     */
    private Bundle makeByType(Bundle callBundle) {
        Bundle result = new Bundle();

        int cardType = callBundle.getInt(ParamKeywords.KEY_int_cardType);
        IChipForCall chipForCall = findChipForCallByCardType(cardType);
        ArrayList<Bundle> resultList = new ArrayList<>();
        if (chipForCall != null) {
            ArrayList<IChipForCall> list = new ArrayList<>();
            list.add(chipForCall);
            resultList = getJniApiBinder(list);
            saveJniApiParam(resultList);
        }
        result.putInt(ParamKeywords.KEY_int_ret, 0);
        result.putParcelableArrayList(ParamKeywords.KEY_ParcelableArrayList_Bundle, resultList);

        return result;
    }

    /**
     * 目前只支持板载芯片
     *
     * @param bundle
     * @return
     */
    private Bundle setChipSpiConfig(Bundle bundle) {
        int chipType = bundle.getInt(ParamKeywords.SPICONFIG_SUPPORT_CHIPTYPE, JniApiParam.TYPE_ONBOARD);
        IChipForCall iChipForCall = findChipForCallByCardType(chipType);
        if (iChipForCall == null) {
            Log.e(TAG, "spiConfig find Chip ForCall fail By CardType " + chipType);
            return null;
        }
        return iChipForCall.callMethod(ParamKeywords.KEY_METHOD_SPICONFIG, "", bundle);
    }

    /**
     * 根据卡类型查找
     *
     * @param cardType
     * @return
     */
    private IChipForCall findChipForCallByCardType(int cardType) {
        if (chipForCallList == null) {
            return null;
        }
        for (IChipForCall chipForCall : chipForCallList) {
            if (chipForCall.getChipType() == cardType) {
                return chipForCall;
            }
        }
        return null;
    }


    /**
     * 设置是否提供卡操作标志位
     *
     * @param bundle
     * @return
     */
    private synchronized Bundle setSingleChipCanUse(Bundle bundle) {
        Bundle result = new Bundle();
        int uid = getCallingUid();
        if (uid != Process.myUid()) {
            result.putInt(ParamKeywords.KEY_int_ret, JarMultiJniApiErrorCode.RET_NO_POWER);
            return result;
        }
        String chipName = bundle.getString(ParamKeywords.KEY_String_chipName, "");
        String value = bundle.getString(ParamKeywords.KEY_String_value, "");
        if (TextUtils.isEmpty(chipName) || TextUtils.isEmpty(value)) {
            result.putInt(ParamKeywords.KEY_int_ret, JarMultiJniApiErrorCode.RET_PARAM_ERROR);
            return result;
        }
        MultiJniApiConfig.CanUseFlag flag;
        if (MultiJniApiConfig.CanUseFlag.ENABLE.name().equals(value)) {
            flag = MultiJniApiConfig.CanUseFlag.ENABLE;
        } else if (MultiJniApiConfig.CanUseFlag.UNABLE.name().equals(value)) {
            flag = MultiJniApiConfig.CanUseFlag.UNABLE;
        } else {
            flag = MultiJniApiConfig.CanUseFlag.DEFAULTED;
        }
        MultiJniApiConfig.getInstance(context).setChipCanUseFlag(chipName, flag);
        setChipStatus(chipName, flag);
        result.putInt(ParamKeywords.KEY_int_ret, 0);
        return result;
    }

    /**
     * 得到芯片是否可用
     *
     * @param bundle
     * @return
     */
    private Bundle getSingleChipCanUseFlag(Bundle bundle) {
        Bundle result = new Bundle();
        int uid = getCallingUid();
        if (uid != Process.myUid()) {
            result.putInt(ParamKeywords.KEY_int_ret, JarMultiJniApiErrorCode.RET_NO_POWER);
            return result;
        }
        String chipName = bundle.getString(ParamKeywords.KEY_String_chipName, "");
        boolean flag = MultiJniApiConfig.getInstance(context).judgeChipCanUse(chipName);
        result.putInt(ParamKeywords.KEY_int_ret, 0);
        result.putBoolean(ParamKeywords.KEY_boolean_flag, flag);
        result.putString(ParamKeywords.KEY_String_chipName, chipName);
        return result;
    }

    /**
     * 异步获取所有
     *
     * @param bundle
     * @return
     */
    private Bundle getAllAsync(final Bundle bundle) {
        Log.w(TAG, "getAllAsync start: ");

        IBinder binder = bundle.getBinder(ParamKeywords.KEY_Binder_Binder);
        final Messenger messenger = new Messenger(binder);
        final CountDownLatch latch = new CountDownLatch(chipForCallList.size());
        for (final IChipForCall chipForCall : chipForCallList) {
            cachedThreadPool.execute(new Runnable() {
                @Override
                public void run() {
                    Bundle result = null;
                    try {
                        if (!judgeChipCanUse(chipForCall)) {
                            Log.e(TAG, TAG_CHIPFORCALL + chipForCall.getChipName() + " false");
                            return;
                        }
                        Log.e(TAG, TAG_CHIPFORCALL + chipForCall.getChipName() + " true");
                        List<Bundle> bundles = chipForCall.getJniApiBinder();
                        if (bundles == null || bundles.size() == 0) {
                            Log.e(TAG, TAG_CHIPFORCALL + chipForCall.getChipName() + " bundles = null");
                            return;
                        }
                        Log.e(TAG, TAG_CHIPFORCALL + chipForCall.getChipName() + " bundles size:" + bundles.size());
                        result = bundles.get(0);
                    } finally {
                        if (result != null) {
                            Message msg = Message.obtain();
                            msg.what = IResultCallback.RET_ON_RESULT;
                            msg.obj = result;
                            try {
                                messenger.send(msg);
                            } catch (RemoteException e) {
                                e.printStackTrace();
                            }
                        }
                        latch.countDown();
                    }
                }
            });
        }
        cachedThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    latch.await();
                    Message msg = Message.obtain();
                    msg.what = IResultCallback.RET_ON_END;
                    messenger.send(msg);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
        Log.w(TAG, " getAllAsync end: ");
        Bundle result = new Bundle();
        result.putInt(ParamKeywords.KEY_int_ret, 0);
        return result;
    }

    private void saveJniApiParam(List<Bundle> list) {
        final List<Bundle> saveList = new ArrayList<>();
        saveList.addAll(list);
        cachedThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                for (Bundle bundle : saveList) {
                    JniApiParam param = bundle.getParcelable(ParamKeywords.KEY_Parcelable_JniApiParam);
                    if (param != null) {
                        SharedPreferencesUtil.saveCardIdAndType(context, param.cardId, param.chipType);
                    }
                }
            }
        });
    }


    //add 2018年12月13日15:24:14 weizg
    private ActivityInfo[] activities;
    private ProviderInfo[] providers;
    private ActivityInfo[] receivers;
    private ServiceInfo[] services;

    //add 2018年12月17日10:27:28 weizg
    private HashMap<String, List<String>> processClassMap = new HashMap<>(5);

    private List<String> getClassInProcessList(String processName) {
        List<String> processClassList = processClassMap.get(processName);

        if (processClassList != null) {
            return processClassList;
        } else {
            List<String> classInProcessListBySearch = getClassInProcessListBySearch(processName);
            processClassMap.put(processName, classInProcessListBySearch);

            return classInProcessListBySearch;
        }
    }

    private List<String> getClassInProcessListBySearch(String processName) {
        PackageManager pm = context.getPackageManager();
        ArrayList<String> result = new ArrayList<>();
        try {
            String pkgName = context.getPackageName();
            processName = pkgName + ":" + processName;

            searchForActivity(processName, pm, result);

            searchForProvider(processName, pm, result);

            searchForReceiver(processName, pm, result);

            searchForService(processName, pm, result);
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        return result;
    }

    private void searchForService(String processName, PackageManager pm, ArrayList<String> result)
            throws PackageManager.NameNotFoundException {
        PackageInfo pi;
        if (services == null) {
            pi = pm.getPackageInfo(context.getPackageName(), PackageManager.GET_SERVICES | PackageManager.GET_DISABLED_COMPONENTS);
            services = pi.services;
            for (ServiceInfo info : services) {
                if (info.processName.equals(processName)) {
                    result.add(info.name);
                }
            }
        }
    }

    private void searchForReceiver(String processName, PackageManager pm, ArrayList<String> result)
            throws PackageManager.NameNotFoundException {
        PackageInfo pi;
        if (receivers == null) {
            pi = pm.getPackageInfo(context.getPackageName(), PackageManager.GET_RECEIVERS | PackageManager.GET_DISABLED_COMPONENTS);
            receivers = pi.receivers;
            for (ActivityInfo info : receivers) {
                if (info.processName.equals(processName)) {
                    result.add(info.name);
                }
            }
        }
    }

    private void searchForProvider(String processName, PackageManager pm, ArrayList<String> result)
            throws PackageManager.NameNotFoundException {
        PackageInfo pi;
        if (providers == null) {
            pi = pm.getPackageInfo(context.getPackageName(), PackageManager.GET_PROVIDERS | PackageManager.GET_DISABLED_COMPONENTS);
            providers = pi.providers;
            for (ProviderInfo info : providers) {
                if (info.processName.equals(processName)) {
                    result.add(info.name);
                }
            }
        }
    }

    private void searchForActivity(String processName, PackageManager pm, ArrayList<String> result)
            throws PackageManager.NameNotFoundException {
        PackageInfo pi;
        if (activities == null) {
            pi = pm.getPackageInfo(context.getPackageName(), PackageManager.GET_ACTIVITIES | PackageManager.GET_DISABLED_COMPONENTS);
            activities = pi.activities;
            for (ActivityInfo info : activities) {
                if (info.processName.equals(processName)) {
                    result.add(info.name);
                }
            }
        }
    }

    /**
     * 设置进程状态
     *
     * @param chipName
     * @param flag
     */
    private void setChipStatus(String chipName, MultiJniApiConfig.CanUseFlag flag) {
        List<String> classInProcessList = getClassInProcessList(chipName);
        PackageManager pm = context.getPackageManager();
        int setFlag = PackageManager.COMPONENT_ENABLED_STATE_ENABLED;
        if (flag == MultiJniApiConfig.CanUseFlag.ENABLE) {
            setFlag = PackageManager.COMPONENT_ENABLED_STATE_ENABLED;
        } else if (flag == MultiJniApiConfig.CanUseFlag.UNABLE) {
            setFlag = PackageManager.COMPONENT_ENABLED_STATE_DISABLED;
        } else if (flag == MultiJniApiConfig.CanUseFlag.DEFAULTED) {
            boolean b = MultiJniApiConfig.getInstance(context).judgeStateFromProperties(chipName);
            if (b) {
                setFlag = PackageManager.COMPONENT_ENABLED_STATE_ENABLED;
            } else {
                setFlag = PackageManager.COMPONENT_ENABLED_STATE_DISABLED;
            }
        }
        for (String process : classInProcessList) {
            ComponentName cn = new ComponentName(context, process);
            pm.setComponentEnabledSetting(cn, setFlag, PackageManager.DONT_KILL_APP);
        }
    }


    //===========================================================================================
    //
    //===========================================================================================

    /**
     * 判断是否调用芯片
     *
     * @return
     */
    private boolean judgeChipCanUse(IChipForCall chipForCall) {
        //1.首先从配置文件、设置的属性等判断是否允许
        boolean flag = MultiJniApiConfig.getInstance(context).judgeChipCanUse(chipForCall.getChipName());
        //如果不可用直接返回
        if (!flag) {
            return false;
        }
        return true;
    }

    private ArrayList<IChipForCall> getChipForCallList() {
        return chipForCallList;
    }


    private Bundle handleCcSetLogFlag(Bundle bundle) {
        CCChipForCall ccChipForCall = new CCChipForCall();
        ccChipForCall.setContext(context);
        String method = bundle.getString(ParamKeywords.KEY_String_method);
        return ccChipForCall.callMethod(method, "", bundle);
    }
}

