package com.xdja.multichip;

import android.bluetooth.BluetoothAdapter;
import android.content.BroadcastReceiver;
import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.RemoteException;
import android.os.SystemClock;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import com.xdja.multichip.param.ChipParam;
import com.xdja.multichip.process.bluetooth.BluetoothKeyProcessService;
import com.xdja.multichip.process.board.OnboardProcessProvider;
import com.xdja.multichip.process.tfcard.TFProcessProvider;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/5/26 <br/>
 * <p>
 * 此类是一个Binder，用于跟各个卡操作进程交互，并向Jar包提供服务。
 */

public class MultiChip extends IMultiChip.Stub {

    public static final String KEY_BINDER_BINDER = "Binder";
    public static final String KEY_PARCELABLE_CHIPPARAM = "ChipParam";

    private static final String NAME_PROCESS_ONBOARD = "onboard";
    private static final String NAME_PROCESS_TF = "TF";
    private static final String NAME_PROCESS_BLUETOOTH = "bluetooth";

    private static HashMap<String, String> providerNameMap = new HashMap<>();

    /**
     * 等待超时时间
     */
    private static final int WAIT_TIMEOUT = 3;

    /**
     * 设备上有芯片的列表（MODEL）
     */
    private static ArrayList<String> hasChipDevList = new ArrayList<>();
    /**
     * 没有TF卡的设备列表
     */
    private static ArrayList<String> hasNotTFDevList = new ArrayList<>();

    static {
        hasChipDevList.add("ACTOMA ACE");
        hasChipDevList.add("HUAWEI eH880");

        hasNotTFDevList.add("ACTOMA ACE");

        providerNameMap.put(NAME_PROCESS_ONBOARD, OnboardProcessProvider.class.getName());
        providerNameMap.put(NAME_PROCESS_TF, TFProcessProvider.class.getName());
    }

    private static MultiChip instance;
    private Context context;
    /**
     * 用于临时保存CountDownLatch的map
     */
    private HashMap<String, CountDownLatch> latchMap = new HashMap<>(0);

    /**
     * 用于保存子进程返回结果
     */
    private HashMap<String, Pair<String, List<Bundle>>> tmpLatchTagMap = new HashMap<>(0);

    /**
     * 线程池
     */
    private ExecutorService cachedThreadPool = Executors.newCachedThreadPool();

    /**
     * 上次调用方法{@link #getChipInfo()}的时间
     */
    private long lastCall = 0;

    private MultiChip(Context context) {
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }
    }

    /**
     * 得到实例;
     * 单例。
     *
     * @param context
     * @return
     */
    public static MultiChip getInstance(Context context) {
        if (instance == null) {
            synchronized (MultiChip.class) {
                if (instance == null) {
                    instance = new MultiChip(context);
                }
            }
        }
        return instance;
    }

    List<Bundle> lastRusult = new ArrayList<>();

    /**
     * 得到卡信息；
     *
     * @return 一个列表；其中Bundle中，包含两个数据，关键字分别为：{@link #KEY_BINDER_BINDER},{@link #KEY_PARCELABLE_CHIPPARAM}
     * @throws RemoteException
     */
    @Override
    public synchronized List<Bundle> getChipInfo() throws RemoteException {
        return getChipInfoUseProvider();
    }

    /**
     * 使用ContentProvider的方式
     *
     * @return
     * @throws RemoteException
     */
    private List<Bundle> getChipInfoUseProvider() throws RemoteException {
        long curCall = SystemClock.elapsedRealtime();
        if ((curCall - lastCall) < 1000) {
            return lastRusult;
        }
        lastRusult.clear();
        List<String> chipProcessList = getChipProcessList();
        final CountDownLatch latch = new CountDownLatch(chipProcessList.size());

        Log.d("MultiChip", "chipProcessList size: " + chipProcessList.size());

        for (final String chipProcess : chipProcessList) {
            final String name = convertToProviderName(chipProcess);
            if (TextUtils.isEmpty(name)) {
                latch.countDown();
                continue;
            }
            cachedThreadPool.execute(new Runnable() {
                @Override
                public void run() {
                    Log.d("MultiChip", chipProcess + " start");

                    Uri uri = Uri.parse("content://" + name);
                    ContentResolver resolver = context.getContentResolver();
                    Log.v("testTag",uri.toString());
                    Bundle bundleResult = resolver.call(uri, "GetChipBinder", "", null);
                    if (bundleResult != null) {
                        int ret = bundleResult.getInt("ret");
                        if (ret == 0) {
                            ArrayList<Bundle> bundle = bundleResult.getParcelableArrayList("result");
                            if (bundle != null) {
                                for (Bundle singleBundle : bundle) {
                                    singleBundle.setClassLoader(ChipParam.class.getClassLoader());
                                    if (singleBundle.getParcelable(KEY_PARCELABLE_CHIPPARAM) != null) {
                                        lastRusult.add(singleBundle);
                                    }
                                }
                            }
                        }
                    }
                    latch.countDown();

                    Log.d("MultiChip", chipProcess + " end");
                }
            });
        }
        try {
            latch.await(WAIT_TIMEOUT, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        lastCall = SystemClock.elapsedRealtime();
        return lastRusult;
    }

    /**
     * 得到 操作芯片进程列表的list
     *
     * @return
     */
    private List<String> getChipProcessList() {
        ArrayList<String> result = new ArrayList<>();
        result.addAll(getOnboardChipProcessList());
        result.addAll(getTFChipProcessList());
        result.addAll(getBluekeyChipProcessList());
        return result;
    }

    /**
     * 存在
     */
    private final static int FLAG_ONBOARD_CHIP_EXIST = 1;
    /**
     * 不存在
     */
    private final static int FLAG_ONBOARD_CHIP_NOT_EXIST = -1;
    /**
     * 未知
     */
    private final static int FLAG_ONBOARD_CHIP_UNKNOW = 0;
    /**
     * 板载芯片是否存在标志位
     */
    private int onboardChipIsExist = FLAG_ONBOARD_CHIP_UNKNOW;

    /**
     * 返回操作板载芯片进程的进程名,如果手机上没有芯片，则返回size为0的进程名
     * <p>应该从多种维度来实现这个方法：
     * <p>1.如果ACE、MATE8、海信等手机，应该需要有
     * <p>2.如果不是以上的手机，则应该在{@link #onboardChipIsExist}为{@link #FLAG_ONBOARD_CHIP_UNKNOW}时，
     * 再次进行判断
     *
     * @return
     */
    private synchronized List<String> getOnboardChipProcessList() {
        ArrayList<String> list = new ArrayList<>();
        //标志位是不存在，则直接返回
        if (onboardChipIsExist == FLAG_ONBOARD_CHIP_NOT_EXIST) {
            return list;
        }
        //标志位是存在，标明有芯片
        else if (onboardChipIsExist == FLAG_ONBOARD_CHIP_EXIST) {
            list.add(NAME_PROCESS_ONBOARD);
        }
        //标志位是未知，需要进行判断，如果当前设备在已知列表中，则将标志位置为FLAG_ONBOARD_CHIP_EXIST
        else if (onboardChipIsExist == FLAG_ONBOARD_CHIP_UNKNOW) {
            boolean flag = checkCurDevIsKownHasChip();
            if (flag) {
                onboardChipIsExist = FLAG_ONBOARD_CHIP_EXIST;
            }
            list.add(NAME_PROCESS_ONBOARD);
        }
        return list;
    }

    /**
     * 只有在hasNotTFDevList中设备，才不返回TF卡操作进程名
     *
     * @return
     */
    private List<String> getTFChipProcessList() {
        String model = Build.MODEL;
        ArrayList<String> result = new ArrayList<>();
        if (hasNotTFDevList.contains(model)) {
            return result;
        }
        result.add(NAME_PROCESS_TF);
        return result;
    }

    /**
     * 根据蓝牙状态来判断是否返回；
     * 如果蓝牙没有打开，则返回的size为0
     *
     * @return
     */
    private List<String> getBluekeyChipProcessList() {
        regeditBluetoothListener();
        ArrayList<String> result = new ArrayList<>();
        BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();
        int state = adapter.getState();
        if (state == BluetoothAdapter.STATE_OFF || state == BluetoothAdapter.STATE_TURNING_OFF) {
            return result;
        }
        result.add(NAME_PROCESS_BLUETOOTH);
        return result;
    }

    /**
     * 判断当前设备是否含有芯片
     *
     * @return 如果含有芯片，则返回true，没有则返回false
     */
    private boolean checkCurDevIsKownHasChip() {
        boolean flag = false;
        String model = Build.MODEL;
        if (hasChipDevList.contains(model)) {
            flag = true;
        }
        return flag;
    }

    /**
     * 蓝牙状态监听标志位;
     * 0：表示还没有开启监听；
     * 其他：已开启
     */
    private int bluetoothFlag = 0;

    private synchronized void regeditBluetoothListener() {
        if (bluetoothFlag != 0) {
            return;
        }
        bluetoothFlag = 1;
        IntentFilter filter = new IntentFilter();
        filter.addAction(BluetoothAdapter.ACTION_STATE_CHANGED);
        context.registerReceiver(bluetoothReceiver, filter);
    }

    /**
     * 蓝牙监听广播
     */
    private BroadcastReceiver bluetoothReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            int blueState = intent.getIntExtra(BluetoothAdapter.EXTRA_STATE, 0);
            if (BluetoothAdapter.STATE_OFF == blueState) {
                Intent stopIntent = new Intent();
                stopIntent.setClassName(context.getPackageName(), BluetoothKeyProcessService.class.getName());
                context.stopService(stopIntent);
            }
        }
    };

    private String convertToProviderName(String name) {
        if (TextUtils.isEmpty(name)) {
            return "";
        }
        return providerNameMap.get(name);
    }
}


