package com.xdja.uniteauth;

import android.content.Context;
import android.content.SharedPreferences;
import android.text.TextUtils;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

/**
 * @author zhangxiaolong@xdja.com <br/>
 * @date 2019/1/2 <br/>
 */
public class UniteAuthConfig {

    public static final String ASSET_FILE_NAME = "UniteAuth.properties";

    public static final String PREFRENCE_uac = "uac";

    public static final String KEY_role = "role";
    public static final String KEY_contain = "contain";
    public static final String KEY_certType = "certType";
    /**
     * 是否显示认证界面
     */
    public static final String KEY_showAuthActivity = "showAuthActivity";
    public static final String KEY_needFirstInputPin = "needFirstInputPin";
    public static final String KEY_inputedFirstPin = "inputedFirstPin";
    /**
     * 是否显示“忘记PIN码”入口
     */
    public static final String KEY_isShowForgetPin = "isShowForgetPin";
    /**
     * 服务器配置地址
     */
    public static final String KEY_APP_ENDPOINT_UAC = "serverAppEndPoint_uac";

    /**
     * 模式。
     * 目前有默认模式和pin模式
     * 默认模式：(default)或者没有属性中没有module
     * 先判断 showAuthActivity决定是否显示输入PIN码界面，
     * 如果 showAuthActivity为true,弹出AuthLoginActivity界面，验证PIN码正确后获取code。
     * 如果 showAuthActivity为false，走withoutActivity流程获取code。
     * pin模式：(pin)
     * 在这种模式下，调用统一认证获取code码时，先检查PIN码是否存在。
     * 如果PIN码存在，走withoutActivity流程获取code。
     * 如果PIN码不存在，则弹出AuthLoginActivity界面，验证PIN码正确后获取code。
     */
    public static final String KEY_module = "module";

    /**
     * 默认模式：default
     */
    public static final String KEY_module_defalut = "default";

    /**
     * pin模式：pin
     */
    public static final String KEY_module_pin = "pin";

    /**
     * 默认使用0x11角色
     */
    public static final int default_role = 0x11;

    /**
     * 默认使用6号容器
     */
    public static final int default_contain = 6;
    /**
     * 默认使用签名证书
     */
    public static final int default_certType = 2;

    /**
     * 默认使用界面进行统一认证
     */
    public static final boolean default_showAuthActivity = true;


    /**
     * 默认是否需要在特定界面上输入过一次PIN。
     */
    public static final boolean default_needFirstInputPin = false;

    /**
     * 默认 是否已经输入过要求的第一次PIN码
     */
    public static final boolean default_inputedFirstPin = false;

    /**
     * 默认 是否显示忘记PIN码入口，默认值为显示
     */
    public static final boolean default_isShowForgetPin = true;

    /**
     * 默认pinModule
     */
    public static final boolean default_pinModule = false;

    private static UniteAuthConfig instance;
    private Context context;
    private Properties properties;

    private UniteAuthConfig(Context context) {
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }
        if (this.context == null) {
            throw new NullPointerException("Context is null.");
        }

    }

    public static UniteAuthConfig getInstance(Context context) {
        if (instance == null) {
            synchronized (UniteAuthConfig.class) {
                if (instance == null) {
                    instance = new UniteAuthConfig(context);
                }
            }
        }
        return instance;
    }

    /**
     * 得到使用的角色
     *
     * @return
     */
    public int getRole() {
        int role = default_role;
        String roleStr = getProperties().getProperty(KEY_role, "" + default_role);
        if (!TextUtils.isEmpty(roleStr)) {
            try {
                if (roleStr.startsWith("0x") || roleStr.startsWith("0X")) {
                    role = Integer.valueOf(roleStr.substring(2, roleStr.length()), 16);
                } else {
                    role = Integer.valueOf(roleStr);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return role;
    }

    /**
     * 得到使用的容器号
     *
     * @return
     */
    public int getContain() {
        int contain = default_contain;
        String containStr = getProperties().getProperty(KEY_contain);
        if (!TextUtils.isEmpty(containStr)) {
            try {
                contain = Integer.valueOf(containStr);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return contain;
    }

    /**
     * 得到证书类型
     *
     * @return
     */
    public int getCertType() {
        int certType = default_certType;
        String certTypeStr = getProperties().getProperty(KEY_certType);
        if (!TextUtils.isEmpty(certTypeStr)) {
            try {
                certType = Integer.valueOf(certTypeStr);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return certType;
    }

    /**
     * 是否是pinModule
     *
     * @return
     */
    public boolean isPinModule() {
        boolean flag = false;
        String property = getProperties().getProperty(KEY_module);
        if (!TextUtils.isEmpty(property)) {
            if (property.equals(KEY_module_pin)) {
                flag = true;
            }
        }
        return flag;
    }

    /**
     * 是否显示认证界面
     *
     * @return
     */
    public boolean isShowAuthActivity() {
        boolean flag = default_showAuthActivity;
        String property = getProperties().getProperty(KEY_showAuthActivity);
        if (!TextUtils.isEmpty(property)) {
            if (property.equals("false")) {
                flag = false;
            }
        }
        return flag;
    }


    /**
     * 在properties中 needFirstInputPin 的值
     *
     * @return
     */
    public boolean getNeedFirstInputPinFromProperties() {
        boolean flag = default_needFirstInputPin;
        String property = getProperties().getProperty(KEY_needFirstInputPin);
        if (!TextUtils.isEmpty(property)) {
            if (property.equals("true")) {
                flag = true;
            }
        }
        return flag;
    }

    /**
     * 设置是否需要在特定界面上输入过一次PIN 在SharedPreferences中。
     *
     * @param flag
     */
    public void setInputedFirstPin(boolean flag) {
        SharedPreferences preferences = context.getSharedPreferences(PREFRENCE_uac, Context.MODE_PRIVATE);
        SharedPreferences.Editor edit = preferences.edit();
        edit.putBoolean(KEY_inputedFirstPin, flag);
        edit.apply();
    }

    /**
     * 从SharedPreferences中读取
     * 默认为false
     *
     * @return
     */
    public boolean getInputedFirstPinFromPreferences() {
        SharedPreferences preferences = context.getSharedPreferences(PREFRENCE_uac, Context.MODE_PRIVATE);
        boolean flag = preferences.getBoolean(KEY_inputedFirstPin, default_inputedFirstPin);
        return flag;
    }

    /**
     * 获取统一认真要连接的后台地址
     *
     * @return
     */
    public String getUacServerAddress() {
        String property = getProperties().getProperty(KEY_APP_ENDPOINT_UAC);
        return property;
    }

    /**
     * 判断needFirstInputPin
     *
     * @return true有两种情况：1.properties规定的不需要输入PIN码；2.properties规定的让输入PIN码，且已经输入过了。
     * false：表示properties规定的让输入PIN码，但还没有输入
     */
    public boolean judgeInputPinFlag() {
        boolean flag;
        //如果properties中是不需要判断的，直接返回true
        flag = UniteAuthConfig.getInstance(context).getNeedFirstInputPinFromProperties();
        if (!flag) {
            return true;
        }

        //如果properties中需要让用户在特定界面输入过一次PIN，接着判断SharedPreferences中
        flag = UniteAuthConfig.getInstance(context).getInputedFirstPinFromPreferences();
        //如果已经输入过了，则是在SharedPreferences中已经置为true
        if (flag) {
            return true;
        } else {
            return flag;
        }
    }


    /**
     * 是否显示忘记PIN码入口
     *
     * @return
     */
    public boolean isShowForgetPinView() {
        boolean flag = default_isShowForgetPin;
        String property = getProperties().getProperty(KEY_isShowForgetPin);
        if (!TextUtils.isEmpty(property)) {
            if (property.equals("false")) {
                flag = false;
            }
        }
        return flag;
    }

    /**
     * 获取property
     *
     * @return
     */
    private Properties getProperties() {
        if (properties == null) {
            synchronized (UniteAuthConfig.class) {
                if (properties == null) {
                    try {
                        properties = new Properties();
                        InputStream is = context.getAssets().open(ASSET_FILE_NAME);
                        properties.load(is);
                        is.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        return properties;
    }
}
