package com.xdja.uniteauth.data.local;

import android.content.Context;
import android.content.SharedPreferences;
import android.text.TextUtils;
import android.util.Log;

import com.xdja.uniteauth.utils.SafeCardUtil;
import com.xdja.uniteauth.utils.UacPreferencesUtil;

/**
 * Author:kongguoguang
 * Date:2019/8/9
 * Time:15:44
 * Summary:SharedPreferences工具类，存储设置信息
 */
public class UacPreference {

    private final String TAG = "UacPreferenceTag";

    private static final String DEFAULT_PREFERENCE = "uac_default";

    private Context context;

    //用户名标识
    private static final String KEY_USER_NAME = "key_user_name";

    //用户昵称标识
    private static final String KEY_USER_NICKNAME = "key_user_nickname";

    //用户id标识
    private static final String KEY_USER_ID = "key_user_id";

    //用户密码标识
    private static final String KEY_USER_PASSWORD = "key_user_password";

    //用户idp id
    private static final String KEY_USER_IDP_ID = "key_user_idp_id";

    //用户SN
    private static final String KEY_USER_SN = "key_user_SN";

    //指纹认证可用标识
    private static final String KEY_FINGERPRINT_AUTH_AVAILABLE = "key_fingerprint_auth_available";

    //指纹ID标识
    private static final String KEY_FINGERPRINT_ID = "key_fingerprint_id";

    //指纹传感器锁定时间戳
    private static final String KEY_FINGERPRINT_LOCKED_TIMESTAMP = "key_fingerprint_locked_timestamp";

    //手势认证可用标识
    private static final String KEY_GESTURE_AUTH_AVAILABLE = "key_gesture_auth_available";

    //手势密码标识
    private static final String KEY_GESTURE_PASSWORD = "key_gesture_password";

    //手势认证锁定时间戳
    private static final String KEY_GESTURE_LOCKED_TIMESTAMP = "key_gesture_locked_timestamp";

    //隐私保护时间戳标识
    private static final String KEY_PRIVACY_PROTECT_TIMESTAMP = "key_privacy_protect_timestamp";

    //隐私保护间隔标识
    private static final String KEY_PRIVACY_PROTECT_INTERVAL = "key_privacy_protect_interval";

    /**
     * 推荐开启指纹、手势标识
     */
    private static final String KEY_RECOMMEND_FINGERPRINT_GESTURE = "key_recommend_fingerprint_gesture";

    private static final UacPreference ourInstance = new UacPreference();

    public static UacPreference getInstance() {
        return ourInstance;
    }

    private UacPreference() {
    }

    private SharedPreferences defaultPreference;

    private SharedPreferences userPreference;

    private SharedPreferences certPreference;

    public void init(Context context) {
        Log.d(TAG, "init()");
        this.context = context;
        defaultPreference = context.getSharedPreferences(DEFAULT_PREFERENCE, Context.MODE_PRIVATE);
        //如果数据被篡改则直接清空全部数据
        if (!UacPreferencesUtil.verifySign(defaultPreference)) {
            UacPreferencesUtil.clear(defaultPreference);
        }

        initUserPreference();
        initCertPreference(SafeCardUtil.getInstance().getSn());
    }

    private void initUserPreference() {
        String userId = getUserId();
        if (!TextUtils.isEmpty(userId)) {
            userPreference = context.getSharedPreferences(userId, Context.MODE_PRIVATE);
            //如果数据被篡改则直接清空全部数据
            if (!UacPreferencesUtil.verifySign(userPreference)) {
                UacPreferencesUtil.clear(userPreference);
            }
        }
    }

    public void initCertPreference(String sn) {
        if (TextUtils.isEmpty(sn)) {
            certPreference = null;
        } else {
            certPreference = context.getSharedPreferences(sn, Context.MODE_PRIVATE);
            //如果数据被篡改则直接清空全部数据
            if (!UacPreferencesUtil.verifySign(certPreference)) {
                UacPreferencesUtil.clear(certPreference);
            }
        }
    }

    public String getUserId() {
        return UacPreferencesUtil.getStringValue(defaultPreference, KEY_USER_ID);
    }

    public void setUserId(String userId) {
        UacPreferencesUtil.setValue(defaultPreference, KEY_USER_ID, userId);
        if (!TextUtils.isEmpty(userId)) {
            userPreference = context.getSharedPreferences(userId, Context.MODE_PRIVATE);
        }
    }


    public String getUserPwd() {
        return UacPreferencesUtil.getStringValue(userPreference, KEY_USER_PASSWORD);
    }

    public void setUserPwd(String password) {
        UacPreferencesUtil.setValue(userPreference, KEY_USER_PASSWORD, password);
    }

    public long getFingerprintLockedTimestamp() {
        return UacPreferencesUtil.getLongValue(defaultPreference, KEY_FINGERPRINT_LOCKED_TIMESTAMP);
    }

    public void setFingerprintLockedTimestamp(long timestamp) {
        UacPreferencesUtil.setValue(defaultPreference, KEY_FINGERPRINT_LOCKED_TIMESTAMP, timestamp);
    }

    /* 芯片数据   start*/
    public boolean isCertFingerprintAuthAvailable() {
        return UacPreferencesUtil.getBooleanValue(certPreference, KEY_FINGERPRINT_AUTH_AVAILABLE, false);
    }

    public void setCertFingerprintAuthAvailable(boolean available) {
        UacPreferencesUtil.setValue(certPreference, KEY_FINGERPRINT_AUTH_AVAILABLE, available);
    }

    public String getCertFingerprintId() {
        return UacPreferencesUtil.getStringValue(certPreference, KEY_FINGERPRINT_ID);
    }

    public void setCertFingerprintId(String id) {
        if (TextUtils.isEmpty(id)) {
            return;
        }
        UacPreferencesUtil.setValue(certPreference, KEY_FINGERPRINT_ID, id);
    }

    public void clearCertFingerprintId() {
        UacPreferencesUtil.removeValue(certPreference, KEY_FINGERPRINT_ID);
    }

    public boolean isCertGestureAuthAvailable() {
        return UacPreferencesUtil.getBooleanValue(certPreference, KEY_GESTURE_AUTH_AVAILABLE, false);
    }

    public void setCertGestureAuthAvailable(boolean available) {
        UacPreferencesUtil.setValue(certPreference, KEY_GESTURE_AUTH_AVAILABLE, available);
    }

    public String getCertGesturePwd() {
        return UacPreferencesUtil.getStringValue(certPreference, KEY_GESTURE_PASSWORD);
    }

    public void setCertGesturePwd(String pwd) {
        UacPreferencesUtil.setValue(certPreference, KEY_GESTURE_PASSWORD, pwd);
    }

    public long getCertPrivacyProtectTimestamp() {
        return UacPreferencesUtil.getLongValue(certPreference, KEY_PRIVACY_PROTECT_TIMESTAMP);
    }

    public void setCertPrivacyProtectTimestamp(long timestamp) {
        UacPreferencesUtil.setValue(certPreference, KEY_PRIVACY_PROTECT_TIMESTAMP, timestamp);
    }

    public int getCertPrivacyProtectInterval() {
        return UacPreferencesUtil.getIntValue(certPreference, KEY_PRIVACY_PROTECT_INTERVAL);
    }

    public void setCertPrivacyProtectInterval(int expiresIn) {
        UacPreferencesUtil.setValue(certPreference, KEY_PRIVACY_PROTECT_INTERVAL, expiresIn);
    }

    public long getCertGestureLockedTimestamp() {
        return UacPreferencesUtil.getLongValue(certPreference, KEY_GESTURE_LOCKED_TIMESTAMP);
    }

    public void setCertGestureLockedTimestamp(long timestamp) {
        UacPreferencesUtil.setValue(certPreference, KEY_GESTURE_LOCKED_TIMESTAMP, timestamp);
    }

    public boolean isCertRecommendFingerprintAndGesture() {
        return UacPreferencesUtil.getBooleanValue(certPreference, KEY_RECOMMEND_FINGERPRINT_GESTURE, true);
    }

    public void setCertRecommendFingerprintAndGesture(boolean recommendFingerprintAndGesture) {
        UacPreferencesUtil.setValue(certPreference, KEY_RECOMMEND_FINGERPRINT_GESTURE, recommendFingerprintAndGesture);
    }

    /* 芯片数据   end*/


    /* 用户数据   start*/

    public String getUserName() {
        return UacPreferencesUtil.getStringValue(userPreference, KEY_USER_NAME);
    }

    public void setUserName(String userName) {
        UacPreferencesUtil.setValue(userPreference, KEY_USER_NAME, userName);
    }

    public String getUserNickname() {
        return UacPreferencesUtil.getStringValue(userPreference, KEY_USER_NICKNAME);
    }

    public void setUserNickname(String nickname) {
        UacPreferencesUtil.setValue(userPreference, KEY_USER_NICKNAME, nickname);
    }

    public int getUserIdpId() {
        return UacPreferencesUtil.getIntValue(userPreference, KEY_USER_IDP_ID, -1);
    }

    public void setUserIdpId(int idpId) {
        UacPreferencesUtil.setValue(userPreference, KEY_USER_IDP_ID, idpId);
    }

    public String getUserSN() {
        return UacPreferencesUtil.getStringValue(userPreference, KEY_USER_SN);
    }

    public void setUserSN(String sn) {
        UacPreferencesUtil.setValue(userPreference, KEY_USER_SN, sn);
    }


    public boolean isUserFingerprintAuthAvailable() {
        return UacPreferencesUtil.getBooleanValue(userPreference, KEY_FINGERPRINT_AUTH_AVAILABLE, false);
    }

    public void setUserFingerprintAuthAvailable(boolean available) {
        UacPreferencesUtil.setValue(userPreference, KEY_FINGERPRINT_AUTH_AVAILABLE, available);
    }

    public String getUserFingerprintId() {
        return UacPreferencesUtil.getStringValue(userPreference, KEY_FINGERPRINT_ID);
    }

    public void setUserFingerprintId(String id) {
        if (TextUtils.isEmpty(id)) {
            return;
        }
        UacPreferencesUtil.setValue(userPreference, KEY_FINGERPRINT_ID, id);
    }

    public void clearUserFingerprintId() {
        UacPreferencesUtil.removeValue(userPreference, KEY_FINGERPRINT_ID);
    }

    public boolean isUserGestureAuthAvailable() {
        return UacPreferencesUtil.getBooleanValue(userPreference, KEY_GESTURE_AUTH_AVAILABLE, false);
    }

    public void setUserGestureAuthAvailable(boolean available) {
        UacPreferencesUtil.setValue(userPreference, KEY_GESTURE_AUTH_AVAILABLE, available);
    }

    public String getUserGesturePwd() {
        return UacPreferencesUtil.getStringValue(userPreference, KEY_GESTURE_PASSWORD);
    }

    public void setUserGesturePwd(String pwd) {
        UacPreferencesUtil.setValue(userPreference, KEY_GESTURE_PASSWORD, pwd);
    }

    public long getUserPrivacyProtectTimestamp() {
        return UacPreferencesUtil.getLongValue(userPreference, KEY_PRIVACY_PROTECT_TIMESTAMP);
    }

    public void setUserPrivacyProtectTimestamp(long timestamp) {
        UacPreferencesUtil.setValue(userPreference, KEY_PRIVACY_PROTECT_TIMESTAMP, timestamp);
    }

    public int getUserPrivacyProtectInterval() {
        return UacPreferencesUtil.getIntValue(userPreference, KEY_PRIVACY_PROTECT_INTERVAL);
    }

    public void setUserPrivacyProtectInterval(int interval) {
        UacPreferencesUtil.setValue(userPreference, KEY_PRIVACY_PROTECT_INTERVAL, interval);
    }

    public void clearUserData() {
        UacPreferencesUtil.removeValue(defaultPreference, KEY_USER_ID);
        UacPreferencesUtil.clear(userPreference);
    }

    public long getUserGestureLockedTimestamp() {
        return UacPreferencesUtil.getLongValue(userPreference, KEY_GESTURE_LOCKED_TIMESTAMP);
    }

    public void setUserGestureLockedTimestamp(long timestamp) {
        UacPreferencesUtil.setValue(userPreference, KEY_GESTURE_LOCKED_TIMESTAMP, timestamp);
    }

    public boolean isUserRecommendFingerprintAndGesture() {
        return UacPreferencesUtil.getBooleanValue(userPreference, KEY_RECOMMEND_FINGERPRINT_GESTURE, true);
    }

    public void setUserRecommendFingerprintAndGesture(boolean recommendFingerprintAndGesture) {
        UacPreferencesUtil.setValue(userPreference, KEY_RECOMMEND_FINGERPRINT_GESTURE, recommendFingerprintAndGesture);
    }

    /* 用户数据   end*/
}
