package com.xdja.uniteauth.domain;

import android.content.Context;
import android.os.Bundle;
import android.os.RemoteException;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import com.google.gson.Gson;
import com.xdja.multichip.jniapi.JarJniApiProxy;
import com.xdja.uniteauth.UniteAuthConfig;
import com.xdja.uniteauth.data.ApiGen;
import com.xdja.uniteauth.data.net.model.ErrInfoBean;
import com.xdja.uniteauth.data.net.model.GetAuthCodeRequest;
import com.xdja.uniteauth.data.net.model.GetAuthCodeResponse;
import com.xdja.uniteauth.data.net.model.GetChallengeResponse;
import com.xdja.uniteauth.data.net.model.UASResponseBean;
import com.xdja.uniteauth.jar.ErrorCode;
import com.xdja.uniteauth.jar.ExtraBean;
import com.xdja.uniteauth.jar.IRealCallback;
import com.xdja.uniteauth.utils.CallbackAppUtil;
import com.xdja.uniteauth.utils.ErrorCodeUtil;
import com.xdja.uniteauth.utils.SafeCardUtil;

import java.util.Arrays;
import java.util.concurrent.CountDownLatch;

import it.sauronsoftware.base64.Base64;

/**
 * <br>@author: wzg@xdja.com <br/>
 * date:   2018/8/2 9:20 <br/>
 */
public class GetAuthCodeManager {

    private static GetAuthCodeManager instance;
    private Context context;

    private GetAuthCodeManager(Context context) {
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }
    }

    public static GetAuthCodeManager getInstance(Context context) {
        if (instance == null) {
            synchronized (GetAuthCodeManager.class) {
                if (instance == null) {
                    if (context == null) {
                        throw new NullPointerException("Context is null!");

                    }
                    instance = new GetAuthCodeManager(context);
                }
            }
        }
        return instance;
    }

    /**
     * 验证调用者apk是否合法
     *
     * @return first：表示是否成功,0:成功,其他：失败；
     * second：如果first=0，表示appId
     */
    Pair<Integer, String> verifyPkg(Bundle bundle) {
        final CountDownLatch latch = new CountDownLatch(1);
        final int[] ret = new int[1];
        final String[] appIdArray = new String[1];
        VerifyPkgThread verifyPkgThread = new VerifyPkgThread(context, bundle, new VerifyPkgThread.IThreadCallback() {
            @Override
            public void onStatus(int status) {
            }

            @Override
            public void onError(int code) {
                ret[0] = code;
                latch.countDown();
            }

            @Override
            public void onSuccess(String appId) {
                ret[0] = 0;
                appIdArray[0] = appId;
                latch.countDown();
            }
        });
        verifyPkgThread.run();
        try {
            latch.await();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        return Pair.create(ret[0], appIdArray[0]);
    }

    /**
     * 不启动登录界面，直接获取AuthCode
     * 并将结果通过iRealCallback返回给调用方
     */
    void getAuthCodeWithoutActivity(final String appRandom,
                                    final String appId,
                                    final IRealCallback iRealCallback) {

        final JarJniApiProxy jarJniApiProxy = SafeCardUtil.getInstance().getJniApiProxy();
        final String cardId = jarJniApiProxy.getCardId().second;

        final int containNo = UniteAuthConfig.getInstance(context).getContain();
        final int certType = UniteAuthConfig.getInstance(context).getCertType();

        final String sn = SafeCardUtil.getInstance().getSn();
        try {
            UASResponseBean challenge = ApiGen.getUnitAuthApi().getChallenge();
            handleGetChangeResponse(iRealCallback, jarJniApiProxy, appRandom, challenge,
                    containNo, certType, appId, cardId, sn);
        } catch (Exception e) {
            e.printStackTrace();
            getAuthCodeCallback(iRealCallback, appRandom,
                    ErrorCode.RET_SERVER_RETURN_ERR, "");
        }

    }

    private void handleGetChangeResponse(final IRealCallback iRealCallback,
                                         JarJniApiProxy jarJniApiProxy,
                                         final String appRandomStr,
                                         UASResponseBean uasResponseBean,
                                         int containNo,
                                         int certType,
                                         String appId,
                                         String cardId,
                                         String sn) {

        Gson gson = new Gson();
        if (uasResponseBean == null) {
            getAuthCodeCallback(iRealCallback, appRandomStr, ErrorCode.RET_SERVER_RETURN_ERR, "");
            return;
        }
        if (uasResponseBean.isIsSuccess()) {
            //获取挑战值成功

            GetChallengeResponse getChallengeResponse =
                    gson.fromJson(gson.toJson(uasResponseBean.getInfo()), GetChallengeResponse.class);

            handleGetAuthCode(iRealCallback, jarJniApiProxy, appRandomStr, getChallengeResponse,
                    containNo, certType, appId, cardId, sn);
        } else {

            ErrInfoBean errInfoBean =
                    gson.fromJson(gson.toJson(uasResponseBean.getInfo()), ErrInfoBean.class);

            getAuthCodeCallback(iRealCallback, appRandomStr, ErrorCode.RET_SERVER_RETURN_ERR,
                    String.valueOf(errInfoBean.getErrCode()));
        }

    }

    private void handleGetAuthCode(final IRealCallback iRealCallback,
                                   JarJniApiProxy jarJniApiProxy,
                                   final String appRandomStr,
                                   GetChallengeResponse getChallengeResponse,
                                   int containNo, int certType,
                                   String appId, String cardId, String sn) {
        byte[] challengeBytes = getChallengeResponse.getChallengeStr().getBytes();

        byte[] dataOut = new byte[512];
        int[] outLen = new int[2];

        //签名
        int ret = SafeCardUtil.getInstance().signByContainNo(jarJniApiProxy, challengeBytes, challengeBytes.length,
                dataOut, outLen, certType, containNo);
        //解析签名结果
        if (ret != 0) {
            getAuthCodeCallback(iRealCallback, appRandomStr, ErrorCode.RET_CHALLENGE_SIGN_FAIL, "");
            return;
        }

        //如果签名正确
        byte[] signResult = Arrays.copyOfRange(dataOut, 0, outLen[0]);

        GetAuthCodeRequest getAuthCodeRequest = new GetAuthCodeRequest();
        getAuthCodeRequest.setScope(GetAuthCodeRequest.scope_openId);
        getAuthCodeRequest.setState(appRandomStr);
        getAuthCodeRequest.setIndex(getChallengeResponse.getIndex());
        getAuthCodeRequest.setSign(new String(Base64.encode(signResult)));

        try {
            UASResponseBean uasResponseBean = ApiGen.getUnitAuthApi().getAuthCode(appId, cardId, sn, getAuthCodeRequest);
            handleGetAuthResponse(iRealCallback, uasResponseBean, appRandomStr, cardId, sn);
        } catch (Exception e) {
            e.printStackTrace();
            getAuthCodeCallback(iRealCallback, appRandomStr,
                    ErrorCode.RET_SERVER_RETURN_ERR, "");
        }
    }

    private void handleGetAuthResponse(IRealCallback iRealCallback,
                                       UASResponseBean uasResponseBean,
                                       String appRandomStr,
                                       String cardId,
                                       String sn) {
        Gson gson = new Gson();
        if (uasResponseBean == null) {
            getAuthCodeCallback(iRealCallback, appRandomStr, ErrorCode.RET_SERVER_RETURN_ERR, "");
            return;
        }

        if (uasResponseBean.isIsSuccess()) {
            //获取授权码成功
            GetAuthCodeResponse getAuthCodeResponse =
                    gson.fromJson(gson.toJson(uasResponseBean.getInfo()), GetAuthCodeResponse.class);

            ExtraBean extraBean = new ExtraBean();
            extraBean.expires_in = "" + getAuthCodeResponse.getExpires_in();
            extraBean.cardId = cardId;
            extraBean.cardSN = sn;
            extraBean.cardType = SafeCardUtil.getInstance().getJniApiProxy().getCardType();

            //调用回调方法，回调数据
            if (iRealCallback != null) {
                try {
                    iRealCallback.callback(
                            getAuthCodeResponse.getState(),
                            0,
                            getAuthCodeResponse.getCode(),
                            gson.toJson(extraBean));
                } catch (RemoteException e) {
                    e.printStackTrace();
                }
            }

        } else {

            ErrInfoBean errInfoBean = gson.fromJson(gson.toJson(uasResponseBean.getInfo()), ErrInfoBean.class);

            Log.w("auth", "server return code: " + errInfoBean.getErrCode());

            getAuthCodeCallback(iRealCallback, appRandomStr, ErrorCode.RET_SERVER_RETURN_ERR,
                    String.valueOf(errInfoBean.getErrCode()));
        }
    }

    private void getAuthCodeCallback(IRealCallback iRealCallback, String appRandomStr,
                                     int callbackRet, String s1) {
        try {
            if (iRealCallback != null) {
                iRealCallback.callback(appRandomStr, callbackRet, "", "");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    /**
     * 获取AuthCode
     */
    int getAuthCode(final String appRandom, final String appId) {

        final JarJniApiProxy jarJniApiProxy = SafeCardUtil.getInstance().getJniApiProxy();
        final String cardId = jarJniApiProxy.getCardId().second;

        final int containNo = UniteAuthConfig.getInstance(context).getContain();
        final int certType = UniteAuthConfig.getInstance(context).getCertType();

        final String sn = SafeCardUtil.getInstance().getSn();
        if (TextUtils.isEmpty(sn)) {
            return ErrorCode.RET_GET_CERT_SN_FAIL;
        }
        try {
            UASResponseBean challenge = ApiGen.getUnitAuthApi().getChallenge();
            return handleGetChangeResponse(jarJniApiProxy, appRandom, challenge, containNo, certType, appId, cardId, sn);
        } catch (Exception e) {
            e.printStackTrace();
            return ErrorCodeUtil.getErrorCode(e);
        }
    }

    private int handleGetChangeResponse(JarJniApiProxy jarJniApiProxy,
                                        final String appRandomStr,
                                        UASResponseBean uasResponseBean,
                                        int containNo,
                                        int certType,
                                        String appId,
                                        String cardId,
                                        String sn) {

        Gson gson = new Gson();
        if (uasResponseBean == null) {
            return ErrorCode.RET_SERVER_RETURN_ERR;
        }
        if (uasResponseBean.isIsSuccess()) {
            //获取挑战值成功

            GetChallengeResponse getChallengeResponse =
                    gson.fromJson(gson.toJson(uasResponseBean.getInfo()), GetChallengeResponse.class);

            return handleGetAuthCode(jarJniApiProxy,
                    appRandomStr, getChallengeResponse, containNo, certType,
                    appId, cardId, sn);
        } else {
            ErrInfoBean errInfoBean = gson.fromJson(gson.toJson(uasResponseBean.getInfo()), ErrInfoBean.class);
            return errInfoBean.getErrCode();
        }

    }

    private int handleGetAuthCode(JarJniApiProxy jarJniApiProxy,
                                  final String app_random_str,
                                  GetChallengeResponse getChallengeResponse,
                                  int containNo, int certType,
                                  String appId, String cardId, String sn) {
        byte[] challengeBytes = getChallengeResponse.getChallengeStr().getBytes();

        byte[] dataOut = new byte[512];
        int[] outLen = new int[2];

        //签名
        int ret = SafeCardUtil.getInstance().signByContainNo(jarJniApiProxy, challengeBytes, challengeBytes.length,
                dataOut, outLen, certType, containNo);
        //解析签名结果
        if (ret != 0) {
            return ErrorCode.RET_CHALLENGE_SIGN_FAIL;
        }

        //如果签名正确
        byte[] signResult = Arrays.copyOfRange(dataOut, 0, outLen[0]);

        GetAuthCodeRequest getAuthCodeRequest = new GetAuthCodeRequest();
        getAuthCodeRequest.setScope(GetAuthCodeRequest.scope_openId);
        getAuthCodeRequest.setState(app_random_str);
        getAuthCodeRequest.setIndex(getChallengeResponse.getIndex());
        getAuthCodeRequest.setSign(new String(Base64.encode(signResult)));

        try {
            UASResponseBean uasResponseBean = ApiGen.getUnitAuthApi().getAuthCode(appId, cardId, sn, getAuthCodeRequest);
            return handleGetAuthResponse(uasResponseBean, cardId, sn);
        } catch (Exception e) {
            e.printStackTrace();
            return ErrorCodeUtil.getErrorCode(e);
        }
    }

    private int handleGetAuthResponse(UASResponseBean uasResponseBean, String cardId, String sn) {
        Gson gson = new Gson();
        if (uasResponseBean == null) {
            return ErrorCode.RET_SERVER_RETURN_ERR;
        }

        if (uasResponseBean.isIsSuccess()) {
            //获取授权码成功
            GetAuthCodeResponse getAuthCodeResponse =
                    gson.fromJson(gson.toJson(uasResponseBean.getInfo()), GetAuthCodeResponse.class);

            ExtraBean extraBean = new ExtraBean();
            extraBean.expires_in = "" + getAuthCodeResponse.getExpires_in();
            extraBean.cardId = cardId;
            extraBean.cardSN = sn;
            extraBean.cardType = SafeCardUtil.getInstance().getJniApiProxy().getCardType();

            CallbackAppUtil.getInstance().saveAuthCodeResponse(getAuthCodeResponse, gson.toJson(extraBean));

            return 0;

        } else {
            ErrInfoBean errInfoBean = gson.fromJson(gson.toJson(uasResponseBean.getInfo()), ErrInfoBean.class);
            Log.w("auth", "server return code: " + errInfoBean.getErrCode());
            return errInfoBean.getErrCode();
        }
    }
}
