package com.xdja.uniteauth.domain;

import android.content.ContentProviderClient;
import android.content.ContentResolver;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.res.AssetFileDescriptor;
import android.net.Uri;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;

import com.google.gson.Gson;
import com.xdja.crypto.XDJACrypto;
import com.xdja.uniteauth.bean.ClientToken;
import com.xdja.uniteauth.data.ApiGen;
import com.xdja.uniteauth.data.net.model.CheckPkgRequest;
import com.xdja.uniteauth.data.net.model.ErrInfoBean;
import com.xdja.uniteauth.data.net.model.UASResponseBean;
import com.xdja.uniteauth.jar.ErrorCode;
import com.xdja.uniteauth.jar.ParamKeywords;
import com.xdja.uniteauth.utils.AssetsUtil;
import com.xdja.uniteauth.utils.PkgInfoUtil;
import com.xdja.uniteauth.utils.VerifyTokenUtil;
import com.xdja.xdjacrypto.XCT_CERT_INFO;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Arrays;
import java.util.HashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import retrofit.Callback;
import retrofit.RetrofitError;
import retrofit.client.Response;

/**
 * @author zhangxiaolong@xdja.com <br/>
 * @date 2018/8/2 <br/>
 * 验证应用信息的线程
 */
public class VerifyPkgThread implements Runnable {
    private static final String TAG = "VerifyPkgThreadTag";

    private Context context;
    private int callingUid;
    private int callingPid;
    private String pkgName;
    private String pkgCertFinger;
    private IThreadCallback callback;
    private boolean isSafeBoxInnerApp;

    /**
     * 验证成功的pkg和证书指纹
     * key:为包名+uid；
     * value:为证书指纹；
     */
    private static HashMap<String, String> verifySuccessMap = new HashMap<>();

    /**
     * 验证成功的pkg和appId；
     * key：为包名+uid；
     * value：为appId
     */
    private static HashMap<String, String> verifySuccessAppIdMap = new HashMap<>();

    /**
     * 验签失败的 uid#pid 和 对应的错误码
     * key:为 uid#pid
     * value: 为错误码
     */
    private static HashMap<String, Integer> verifyFailMap = new HashMap<>();

    public VerifyPkgThread(Context context,
                           Bundle bundle,
                           IThreadCallback callback) {
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }
        this.callingUid = bundle.getInt(ParamKeywords.KEY_int_uid);
        this.callingPid = bundle.getInt(ParamKeywords.KEY_int_pid);
        this.pkgName = bundle.getString(ParamKeywords.KEY_String_pkgName);
        this.callback = callback;
    }

    @Override
    public void run() {
        Log.d(TAG, "uid:" + callingUid + " pid:" + callingPid + " pkgName: " + pkgName);

        ClientToken clientToken;

        //是否为安全盒内部应用
        isSafeBoxInnerApp = isSafeBoxProviderExist(context, pkgName);
        if (isSafeBoxInnerApp) {
            clientToken = getClientTokenFromSafetyBox(pkgName, callingPid);
        } else {
            clientToken = getClientTokenFromApp(pkgName);
        }

        if (TextUtils.isEmpty(pkgCertFinger)) {
            pkgCertFinger = PkgInfoUtil.getCertFingerStr(context, pkgName);
        }

        Log.d(TAG, "pkgCertFinger = " + pkgCertFinger);

        //如果配置文件中配置是不验证签名，直接通过(本地不去验签Token文件，也不去服务端验证)
        if (!isVerifyByConfig()) {
            String appId = verifySuccessAppIdMap.get(getSuccessMapKey());
            if (TextUtils.isEmpty(appId)) {
                if (clientToken != null) {
                    appId = clientToken.getToken_data().getAppID();
                    successCallback(appId);
                    cacheVerifySuccess("", appId);
                    return;
                } else {
                    failCallback(ErrorCode.RET_FAIL_VERIFY_TOKEN);
                    return;
                }
            } else {
                successCallback(appId);
                return;
            }
        }

        //判断是否在已成功的缓存中
        if (verifySuccessMap.containsKey(getSuccessMapKey())) {
            if (verifySuccessMap.get(getSuccessMapKey()).equals(pkgCertFinger)) {
                String appId = verifySuccessAppIdMap.get(getSuccessMapKey());
                if (!TextUtils.isEmpty(appId)) {
                    successCallback(appId);
                    return;
                }
            }
        }

        //判断是否在失败的缓存中
        if (verifyFailMap.containsKey(getFailMapKey())) {
            int ret = verifyFailMap.get(getFailMapKey());
            failCallback(ret);
            return;
        }

        statusMsgCallback(IThreadCallback.START_VERIFY_PKG_TOKEN);

        // 本地验签Token文件
        int ret = verifyPkgAndToken(pkgName, clientToken);
        if (ret != 0) {
            //判断失败后，将这个uid#pid作为key，包名作为value缓存起来
            //防止这个不合法的应用，一直调用
            cacheVerifyFail(ret);
            failCallback(ret);
            return;
        }

        statusMsgCallback(IThreadCallback.START_VERIFY_BY_SERVER);
        // 从服务器端验证
        ret = verifyPkgByServer(context, clientToken, pkgCertFinger);
        if (ret != 0) {
            failCallback(ret);
        } else {
            String appId = clientToken.getToken_data().getAppID();
            cacheVerifySuccess(pkgCertFinger, appId);
            successCallback(appId);
        }
    }

    /**
     * 验证失败时缓存
     */
    private void cacheVerifyFail(int ret) {
        String key = getFailMapKey();
        verifyFailMap.put(key, ret);
    }

    /**
     * 验证成功时缓存
     */
    private void cacheVerifySuccess(String pkgCertFinger, String appId) {
        verifySuccessMap.put(getSuccessMapKey(), pkgCertFinger);
        verifySuccessAppIdMap.put(getSuccessMapKey(), appId);
    }

    /**
     * 成功时回调
     *
     * @param appId 第三方应用的appId
     */
    private void successCallback(String appId) {
        if (callback != null) {
            callback.onSuccess(appId);
        }
    }

    /**
     * 失败时回调
     *
     * @param ret
     */
    private void failCallback(int ret) {
        if (callback != null) {
            callback.onError(ret);
        }
    }

    /**
     * 发送状态
     *
     * @param status
     */
    private void statusMsgCallback(int status) {
        if (callback != null) {
            callback.onStatus(status);
        }
    }

    /**
     * 得到key<br>
     * uid#pid
     *
     * @return
     */
    private String getFailMapKey() {
        return callingUid + "#" + callingPid;
    }

    /**
     * 得到成功的map的key
     *
     * @return
     */
    private String getSuccessMapKey() {
        return pkgName + callingUid;
    }


    /**
     * 在本地验签Token文件
     *
     * @param callingPkgName
     * @param clientToken
     * @return
     */
    private int verifyPkgAndToken(String callingPkgName, ClientToken clientToken) {
        if (clientToken == null) {
            return ErrorCode.RET_FAIL_VERIFY_TOKEN;
        }

        //判断调用apk和Token文件中的apk的包名是否一致
        if (!judgePkg(callingPkgName, clientToken)) {
            return ErrorCode.RET_FAIL_VERIFY_PKG_MISMATCH;
        }

        //验证Token是否被篡改
        byte[] srcData = getSrcData(clientToken);
        byte[] pubKey = getPubKey();
        byte[] signData = it.sauronsoftware.base64.Base64.decode(clientToken.getSignature().getBytes());
        int ret = VerifyTokenUtil.signVerify(srcData, signData, pubKey);
        if (ret != 0) {
            return ErrorCode.RET_FAIL_VERIFY_TOKEN;
        }

        return 0;
    }

    /**
     * 从服务端验证应用
     *
     * @param context
     * @param clientToken
     * @param pkgCertFinger 证书指纹
     * @return
     */
    private int verifyPkgByServer(Context context, ClientToken clientToken, String pkgCertFinger) {
        final int[] ret = new int[1];
        final CountDownLatch latch = new CountDownLatch(1);
        String appId = clientToken.getToken_data().getAppID();
        CheckPkgRequest request = new CheckPkgRequest();
        String pkgName = clientToken.getToken_data().getPackageName();

        request.setPkgName(pkgName);
        request.setPkgSign(pkgCertFinger);
        Callback<UASResponseBean> callback = new Callback<UASResponseBean>() {
            @Override
            public void success(UASResponseBean uasResponseBean, Response response) {
                boolean flag = uasResponseBean.isIsSuccess();
                if (flag) {
                    ret[0] = 0;
                } else {
                    Gson gson = new Gson();
                    ErrInfoBean bean = gson.fromJson(gson.toJson(uasResponseBean.getInfo()), ErrInfoBean.class);
                    ret[0] = convertErrorCode(bean.getErrCode());
                }
                latch.countDown();
            }

            @Override
            public void failure(RetrofitError error) {
                if (error.toString().contains("failed to connect to")) {
                    ret[0] = ErrorCode.RET_TIMEOUT_REQUEST_SERVER;
                } else {
                    ret[0] = ErrorCode.RET_EXCEPTION_NETWORK;
                }
                Log.e(TAG, "VerifyPkgFromServer error: " + error.toString());
                latch.countDown();
            }
        };
        ApiGen.getUnitAuthApi().checkPkgSign(appId, request, callback);
        try {
            //最长等待55秒
            int waitTime = 55;
            boolean flag = latch.await(waitTime, TimeUnit.SECONDS);
            if (!flag) {
                ret[0] = ErrorCode.RET_TIMEOUT_REQUEST_SERVER;
            }
        } catch (InterruptedException e) {
            ret[0] = ErrorCode.RET_TIMEOUT_REQUEST_SERVER;
        }
        return ret[0];
    }


    /**
     * 查看配置文件中的标志位，是否验签。
     *
     * @return
     */
    private boolean isVerifyByConfig() {
//        String filePath = "config.properties";
//        String keyPkg = "pkgVerifySignFlag";
//        String str = "";
//
//        //modify 2018年12月12日16:16:14 weizg
//        //getValue方法不再throw异常
//        str = GetValueFromProperties.getValue(context, filePath, keyPkg);
//
//        // TODO: 2018/12/12  切记注释return false
//        if (str.equals("false")) {
////            return false;
//        }
        // modify by zhangxiaolong 一定要去验证调用的应用 2019-1-3
        return true;
    }

    private static final String TOKEN_FILE_NAME = "token.pro";

    /**
     * 判断安全盒中provider是否存在，因为安全盒项目原因（同一个设备上安装多个安全盒应用），安全盒provider的uri会变化
     */
    private boolean isSafeBoxProviderExist(Context context, String pkgName) {
        if (context == null || TextUtils.isEmpty(pkgName)) {
            return false;
        }

        boolean isExist = false;
        //安全盒提供了一个读取内部应用token.pro文件的provider，uri是{安全盒包名}.TokenProvider
        Uri uri = Uri.parse("content://" + pkgName + ".TokenProvider");

        // 如果uri有对应的provider，
        ContentResolver contentResolver = context.getContentResolver();
        ContentProviderClient contentProviderClient = null;
        try {
            contentProviderClient = contentResolver.acquireContentProviderClient(uri);
            if (contentProviderClient != null) {
                isExist = true;
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (contentProviderClient != null) {
                contentProviderClient.release();
            }
        }

        Log.d(TAG, "pkgName check " + pkgName + " safety box = " + isExist);

        return isExist;
    }

    /**
     * 判断应用是否是debug模式
     *
     * @param context
     * @return
     */
    private boolean isApkInDebug(Context context) {
        try {
            ApplicationInfo info = context.getApplicationInfo();
            return (info.flags & ApplicationInfo.FLAG_DEBUGGABLE) != 0;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 判断otherApkPkgName的签名跟自己是否相同
     *
     * @param context
     * @param otherApkPkgName
     * @return
     */
    private boolean isSignEqual(Context context, String otherApkPkgName) {
        PackageManager pm = context.getPackageManager();
        int i = pm.checkSignatures(context.getPackageName(), otherApkPkgName);
        return i == PackageManager.SIGNATURE_MATCH;
    }

    /**
     * 通过安全盒读取应用token.pro文件，生成ClientToken对象，同时把应用的证书指纹SHA1值一并返回
     */
    private ClientToken getClientTokenFromSafetyBox(String pkgName, int pid) {
        ContentResolver resolver = context.getContentResolver();
        Uri uri = Uri.parse("content://" + pkgName + ".TokenProvider");
        Bundle result = resolver.call(uri, "getToken", "" + pid, null);
        String keyTokenBytes = "tokenBytes";
        String keyTokenFd = "tokenFd";
        String keySignature = "signature";
        InputStream is = null;
        ClientToken clientToken = null;
        if (result != null) {
            int ret = result.getInt("ret");
            if (ret == 0) {
                if (result.containsKey(keyTokenBytes)) {
                    byte[] tokenBytes = result.getByteArray(keyTokenBytes);
                    is = new ByteArrayInputStream(tokenBytes);
                } else if (result.containsKey(keyTokenFd)) {
                    AssetFileDescriptor fd;
                    fd = result.getParcelable(keyTokenFd);
                    if (fd != null) {
                        is = new FileInputStream(fd.getFileDescriptor());
                    }
                }

                if (result.containsKey(keySignature)) {
                    pkgCertFinger = result.getString(keySignature);
                }
            }
        }

        if (is != null) {
            InputStreamReader reader = new InputStreamReader(is);
            clientToken = new Gson().fromJson(reader, ClientToken.class);

            try {
                is.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        return clientToken;
    }

    /**
     * 读取应用token.pro文件，生成ClientToken对象
     */
    private ClientToken getClientTokenFromApp(String pkgName) {

        ClientToken clientToken = null;
        try {
            Context cxt = this.context.createPackageContext(pkgName, Context.CONTEXT_IGNORE_SECURITY);
            InputStream is = cxt.getAssets().open(TOKEN_FILE_NAME);
            InputStreamReader reader = new InputStreamReader(is);
            clientToken = new Gson().fromJson(reader, ClientToken.class);
            is.close();
        } catch (Exception e) {
            e.printStackTrace();
        }

        return clientToken;
    }


    /**
     * 得到公钥
     *
     * @return
     */
    private byte[] getPubKey() {
        byte[] certBytes = null;
        String certPath = "auth.cer";

        byte[] certBytesTmp = AssetsUtil.getFileContent(context, certPath);
        if (certBytesTmp != null) {
            XCT_CERT_INFO certInfo = new XCT_CERT_INFO();
            int ret = XDJACrypto.getInstance().GetCertInfo(certBytesTmp, certBytesTmp.length, certInfo);
            if (ret == 0) {
                certBytes = Arrays.copyOfRange(certInfo.pubkey, 1, certInfo.getPubkeyLen());
            }
        }
        return certBytes;
    }

    /**
     * 得到签名的原数据
     *
     * @return
     */
    private byte[] getSrcData(ClientToken clientToken) {
        String contentForSign;

        int suit = clientToken.getToken_data().getSuit();
        String appId = clientToken.getToken_data().getAppID();
        String pkgName = clientToken.getToken_data().getPackageName();
        if (suit == 0) {
            contentForSign = appId + "||" + pkgName;
        } else {
            //包含加密套件编号的token文件
            contentForSign = appId + "||" + pkgName + "||" + suit;
        }
        return contentForSign.getBytes();
    }

    /**
     * 判断调用方应用的包名和Token文件中的包名是否一致
     * 如果是安全盒内部应用则直接返回true
     */
    private boolean judgePkg(String pkgName, ClientToken clientToken) {
        if (isSafeBoxInnerApp) {
            return true;
        } else {
            return pkgName.equals(clientToken.getToken_data().getPackageName());
        }

//        return true;
    }

    /**
     * 错误码转换<br>
     * 0X4201	app_not_register	应用未注册
     * 0X4202	app_package_mismatching	应用包名不匹配
     * 0X4203	app_sign_mismatching	包公钥签名不匹配
     *
     * @param errorCode
     * @return
     */
    private int convertErrorCode(int errorCode) {
        int ret;
        if (errorCode == 0X4201) {
            ret = ErrorCode.RET_FAIL_VERIFY_NO_SUCH_PKG;
        } else if (errorCode == 0X4202) {
            ret = ErrorCode.RET_FAIL_VERIFY_PKG_MISMATCH;
        } else if (errorCode == 0X4203) {
            ret = ErrorCode.RET_FAIL_VERIFY_SIGN_MISMATCH;
        } else {
            ret = errorCode;
        }
        return ret;
    }

    public interface IThreadCallback {
        /**
         * 开始验证应用的Token
         */
        int START_VERIFY_PKG_TOKEN = 1;

        /**
         * 开始向服务端验证
         */
        int START_VERIFY_BY_SERVER = 2;

        /**
         * 发送状态码
         *
         * @param status {@link #START_VERIFY_PKG_TOKEN}, {@link #START_VERIFY_BY_SERVER}
         */
        void onStatus(int status);

        /**
         * 验证应用出错了。
         *
         * @param code
         */
        void onError(int code);

        /**
         * 验证通过
         */
        void onSuccess(String appId);
    }
}
