package com.xdja.uniteauth.presenter;

import android.os.Bundle;
import android.os.Message;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.text.method.HideReturnsTransformationMethod;
import android.text.method.PasswordTransformationMethod;
import android.view.View;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.EditText;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.xdja.multichip.jniapi.JarJniApiProxy;
import com.xdja.uniteauth.R;
import com.xdja.uniteauth.ThreadPoolManager;
import com.xdja.uniteauth.bean.LoginMode;
import com.xdja.uniteauth.domain.CertLoginThread;
import com.xdja.uniteauth.jar.ErrorCode;
import com.xdja.uniteauth.utils.AuthModeUtil;
import com.xdja.uniteauth.utils.ErrorCodeUtil;
import com.xdja.uniteauth.utils.SafeCardUtil;
import com.xdja.uniteauth.utils.SettingInfoUtil;
import com.xdja.uniteauth.utils.UacActivityManager;
import com.xdja.uniteauth.utils.UserInfoUtil;
import com.xdja.uniteauth.view.UacCommonDialog;
import com.xdja.uniteauth.view.UacFingerprintDialog;
import com.xdja.uniteauth.view.UacImageView;
import com.xdja.uniteauth.view.UacInputDialog;

/**
 * @author kgg
 */
public class CertLoginActivity extends BaseLoginActivity {

    private TextView safeCardNameTv;
    private RelativeLayout pinInputLayout;
    private EditText pinEt;

    private Button loginBt;

    /**
     * 验证通过待添加的指纹ID
     */
    private String toAddFid;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.uac_activity_cert_login);
        initTitle();
        initView();
        initBottomSheetDialog();
        int ret = initSafeCardInfo();
        if (ret != 0) {
            showErrorToast(ret);
            finish();
        }

        initAuthMode();

    }

    @Override
    protected void initView() {
        super.initView();

        UacImageView appLogo = (UacImageView) findViewById(R.id.iv_app_logo);
        appLogo.setImageUrl(AuthModeUtil.getInstance().getAppLogo());

        TextView appName = (TextView) findViewById(R.id.tv_app_name);
        appName.setText(AuthModeUtil.getInstance().getAppName());

        safeCardNameTv = (TextView) findViewById(R.id.tv_safe_card_name);
        pinInputLayout = (RelativeLayout) findViewById(R.id.layout_pin_input);
        pinEt = (EditText) findViewById(R.id.et_pin);
        pinEt.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {

            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {

            }

            @Override
            public void afterTextChanged(Editable s) {
                setViewEnable(loginBt, s.length() > 0);
            }
        });
        pinEt.setTransformationMethod(PasswordTransformationMethod.getInstance());

        CheckBox checkBox = (CheckBox) findViewById(R.id.checkbox);
        checkBox.setOnCheckedChangeListener(new CompoundButton.OnCheckedChangeListener() {
            @Override
            public void onCheckedChanged(CompoundButton buttonView, boolean isChecked) {
                if (isChecked) {//显示密码
                    pinEt.setTransformationMethod(HideReturnsTransformationMethod.getInstance());
                } else {//隐藏密码
                    pinEt.setTransformationMethod(PasswordTransformationMethod.getInstance());
                }
                //光标放置到最后面
                pinEt.setSelection(pinEt.length());
            }
        });

        loginBt = (Button) findViewById(R.id.bt_login);
        loginBt.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                login();
            }
        });

    }

    private int initSafeCardInfo() {

        JarJniApiProxy jarJniApiProxy = SafeCardUtil.getInstance().getJniApiProxy(
                getApplicationContext(), "", -1);
        if (jarJniApiProxy == null) {
            return ErrorCode.RET_OPEN_SAFE_CHIP_FAIL;
        }

        String sn = SafeCardUtil.getInstance().getSn();
        if (TextUtils.isEmpty(sn)) {
            return ErrorCode.RET_GET_CERT_SN_FAIL;
        }

        String userSn = UserInfoUtil.getInstance().getSn();
        UserInfoUtil.getInstance().setSafeCardChanged(!TextUtils.isEmpty(userSn) && !userSn.equals(sn));

        String cardTypeName = getString(SafeCardUtil.cardTypeStrMap.get(jarJniApiProxy.getCardType()));
        safeCardNameTv.setText(getString(R.string.uac_input_safe_card_pin, cardTypeName));
        return 0;
    }

    private void initAuthMode() {
        //PIN为空或验证失败，只能使用PIN登录
        if (TextUtils.isEmpty(SafeCardUtil.getInstance().getPin()) ||
                SafeCardUtil.getInstance().isLastVerifyPinFailed()) {
            currentLoginMode = LoginMode.Cert;
            setViewEnable(loginBt, false);
        } else if (SettingInfoUtil.getInstance().isFingerprintLoginEnabled()) {
            currentLoginMode = LoginMode.FINGERPRINT;
            setViewEnable(loginBt, true);
            safeCardNameTv.setVisibility(View.INVISIBLE);
            pinInputLayout.setVisibility(View.INVISIBLE);
            login();
        } else if (SettingInfoUtil.getInstance().isGestureLoginEnabled()) {
            currentLoginMode = LoginMode.GESTURE;
            setViewEnable(loginBt, true);
            safeCardNameTv.setVisibility(View.INVISIBLE);
            pinInputLayout.setVisibility(View.INVISIBLE);
            login();
        } else {
            currentLoginMode = LoginMode.Cert;
            setViewEnable(loginBt, false);
        }

        refreshOtherLoginMode();
    }

    @Override
    protected void refreshOtherLoginMode() {
        super.refreshOtherLoginMode();

        if (currentLoginMode == LoginMode.Cert) {
            getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_PAN);
        } else {
            getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_NOTHING);
        }
    }

    private void login() {

        if (currentLoginMode == LoginMode.Cert) {
            String pin = pinEt.getText().toString();
            startLoginThread(pin);

        } else if (currentLoginMode == LoginMode.FINGERPRINT) {
            if (SettingInfoUtil.getInstance().isPrivacyProtectTimeout()) {
                showFingerprintLoginDialog(new UacFingerprintDialog.SuccessCallback() {
                    @Override
                    public void onSuccess(String fid) {
                        handleFingerprintSuccess(fid);
                    }
                });
            } else {
                startLoginThread();
            }

        } else if (currentLoginMode == LoginMode.GESTURE) {
            if (SettingInfoUtil.getInstance().isPrivacyProtectTimeout()) {
                gotoGestureLoginActivity(bundle);
            } else {
                startLoginThread();
            }
        }
    }

    private void handleFingerprintSuccess(String fid) {
        String addedFid = SettingInfoUtil.getInstance().getFingerprintId();
        if (TextUtils.isEmpty(fid)) {
            startLoginThread();
        } else if (addedFid.contains(fid)) {
            startLoginThread();
        } else {//引导用户添加指纹
            toAddFid = fid;
            final UacCommonDialog dialog = new UacCommonDialog(this, UacCommonDialog.TWO_BUTTON_WITH_FINGERPRINT);
            dialog.setSummary(R.string.uac_not_verify_fid);
            dialog.setCancelButton(R.string.uac_try_again, new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    dialog.dismiss();
                    login();
                }
            });

            dialog.setConfirmButton(R.string.uac_add_fid, new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    dialog.dismiss();
                    verifyPinAndAddFid();
                }
            });

            dialog.show();
        }
    }

    private void startLoginThread() {
        startLoginThread(SafeCardUtil.getInstance().getPin());
    }

    /**
     * 启动用户登录线程
     */
    private void startLoginThread(String pin) {
        showLoadingDialog(R.string.uac_login_ing);
        ThreadPoolManager.getInstance().execute(
                new CertLoginThread(getApplicationContext(), uacHandler, pin, bundle));
    }

    /**
     * 验证PIN码并添加指纹id（验证通过后获取授权码）
     */
    private void verifyPinAndAddFid() {

        showInputPinDialog(new UacInputDialog.InputCompleteCallback() {
            @Override
            public void onComplete(UacInputDialog dialog, CharSequence inputContent) {
                if (TextUtils.isEmpty(inputContent.toString().trim())) {
                    showToast(R.string.uac_pin_input_null);
                    return;
                }

                dialog.dismiss();
                startLoginThread(inputContent.toString());
            }
        });
    }

    @Override
    protected void onStart() {
        super.onStart();
        //PIN码为空或验证失败后切换到PIN码登录
        if (TextUtils.isEmpty(SafeCardUtil.getInstance().getPin()) ||
                SafeCardUtil.getInstance().isLastVerifyPinFailed()) {
            changeLoginMode(LoginMode.Cert);
            return;
        }

        //用户进入设置界面关闭了指纹登录，切换到PIN码登录
        if (currentLoginMode == LoginMode.FINGERPRINT &&
                !SettingInfoUtil.getInstance().isFingerprintLoginEnabled()) {
            changeLoginMode(LoginMode.Cert);
            return;
        }

        //用户进入设置界面关闭了指纹登录，切换到PIN码登录
        if (currentLoginMode == LoginMode.GESTURE &&
                !SettingInfoUtil.getInstance().isGestureLoginEnabled()) {
            changeLoginMode(LoginMode.Cert);
        }
    }

    @Override
    protected void changeLoginMode(LoginMode loginMode) {
        super.changeLoginMode(loginMode);
        if (currentLoginMode == loginMode) {
            return;
        }

        if (loginMode == LoginMode.Cert) {
            currentLoginMode = loginMode;
            refreshOtherLoginMode();
            safeCardNameTv.setVisibility(View.VISIBLE);
            pinInputLayout.setVisibility(View.VISIBLE);
            setViewEnable(loginBt, false);
            return;
        }


        //指纹认证未开启
        if (loginMode == LoginMode.FINGERPRINT &&
                !SettingInfoUtil.getInstance().isFingerprintLoginEnabled()) {
            showFingerprintLoginNotEnabledDialog(bundle);
            return;
        }

        //手势认证未开启
        if (loginMode == LoginMode.GESTURE &&
                !SettingInfoUtil.getInstance().isGestureLoginEnabled()) {
            showGestureLoginNotEnabledDialog(bundle);
            return;
        }

        //PIN码为空
        if (TextUtils.isEmpty(SafeCardUtil.getInstance().getPin())) {
            showUacDialog(R.string.uac_pin_modifyed);
            return;
        }

        currentLoginMode = loginMode;
        refreshOtherLoginMode();
        safeCardNameTv.setVisibility(View.INVISIBLE);
        pinInputLayout.setVisibility(View.INVISIBLE);
        pinEt.setText("");
        setViewEnable(loginBt, true);
        login();
    }


    @Override
    public void handleThreadMessage(Message msg) {
        super.handleThreadMessage(msg);
        int errorCode = msg.what;

        //证书登录线程结果处理
        if (errorCode == 0) {
            if (currentLoginMode == LoginMode.FINGERPRINT) {
                if (!TextUtils.isEmpty(toAddFid)) {
                    SettingInfoUtil.getInstance().setFingerprintId(toAddFid);
                }
                SettingInfoUtil.getInstance().updatePrivacyProtectTimestamp();
            }

            if (SettingInfoUtil.getInstance().isGestureLoginEnabled() ||
                    SettingInfoUtil.getInstance().isFingerprintLoginEnabled()) {
                UacActivityManager.getInstance().finishAllActivity();
            } else if (AuthModeUtil.getInstance().isSupportFingerprintLogin() ||
                    AuthModeUtil.getInstance().isSupportGestureLogin()) {
                recommendFingerprintGesture();
            } else {
                UacActivityManager.getInstance().finishAllActivity();
            }

        } else if (errorCode == ErrorCode.RET_VERIFY_SAFE_PIN_ERR) {
            if (currentLoginMode == LoginMode.Cert) {
                showUacDialog(ErrorCodeUtil.getPinErrorMsg());
                return;
            }

            if (ErrorCodeUtil.isVHSMNetworkError(ErrorCodeUtil.pinErrorCode)) {
                showErrorDialog(ErrorCodeUtil.pinErrorCode);
                return;
            }

            showUacDialog(R.string.uac_pin_modifyed);
            changeLoginMode(LoginMode.Cert);
        } else if (errorCode == ErrorCode.RET_NEED_VERIFY_PIN ||
                errorCode == ErrorCode.RET_GET_PIN_NULL) {
            showUacDialog(R.string.uac_pin_modifyed);
            changeLoginMode(LoginMode.Cert);
        } else {
            showErrorDialog(errorCode);
        }
    }
}
