package com.xdja.uniteauth.presenter;


import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.graphics.Color;
import android.os.Bundle;
import android.os.CountDownTimer;
import android.os.Message;
import android.util.Log;
import android.view.View;
import android.widget.TextView;

import com.andrognito.patternlockview.PatternLockView;
import com.andrognito.patternlockview.listener.PatternLockViewListener;
import com.andrognito.patternlockview.utils.PatternLockUtils;
import com.xdja.uniteauth.R;
import com.xdja.uniteauth.ThreadPoolManager;
import com.xdja.uniteauth.domain.CertLoginThread;
import com.xdja.uniteauth.domain.GetAuthCodeThread;
import com.xdja.uniteauth.domain.UserPwdLoginThread;
import com.xdja.uniteauth.jar.ErrorCode;
import com.xdja.uniteauth.jar.ParamKeywords;
import com.xdja.uniteauth.utils.AuthModeUtil;
import com.xdja.uniteauth.utils.ErrorCodeUtil;
import com.xdja.uniteauth.utils.PackageUtils_UAC;
import com.xdja.uniteauth.utils.SafeCardUtil;
import com.xdja.uniteauth.utils.SettingInfoUtil;
import com.xdja.uniteauth.utils.UacActivityManager;
import com.xdja.uniteauth.utils.UserInfoUtil;
import com.xdja.uniteauth.view.UacImageView;
import com.xdja.uniteauth.view.UacTitleBar;

import java.util.List;

/**
 * @author kgg
 */
public class GestureLoginActivity extends UacBaseActivity implements PatternLockViewListener {

    private PatternLockView patternLockView;

    private TextView gestureDrawTips;

    private int gestureDrawTipsWrongColor;

    private int gestureDrawTipsNormalColor;

    private Bundle bundle;

    private final int MAX_WRONG_COUNT = 5;

    private final long GESTURE_LOCK_TIME = 30 * 1000;

    private int gesturePwdWrongCount;

    private CountDownTimer countDownTimer;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.uac_activity_gesture_login);
        bundle = getIntent().getBundleExtra(ParamKeywords.KEY_Bundle_bundle);
        initView();
    }

    private void initView() {

        UacTitleBar titleBar = (UacTitleBar) findViewById(R.id.uac_title_bar);
        titleBar.setBackListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                finish();
            }
        });

        TextView name = (TextView) findViewById(R.id.tv_app_name);
        UacImageView logo = (UacImageView) findViewById(R.id.iv_app_logo);

        if (AuthModeUtil.getInstance().isOpenPlatformApp()) {
            String pkgName = bundle.getString(ParamKeywords.KEY_String_pkgName);
            PackageInfo packageInfo = PackageUtils_UAC.getPackageInfo(this, pkgName);
            PackageManager manager = getPackageManager();
            name.setText(packageInfo.applicationInfo.loadLabel(manager));
            logo.setImageDrawable(packageInfo.applicationInfo.loadIcon(manager));
        } else if (AuthModeUtil.getInstance().isCertAuthMode()) {
            logo.setImageUrl(AuthModeUtil.getInstance().getAppLogo());
            name.setText(AuthModeUtil.getInstance().getAppName());
        } else {
            name.setText(UserInfoUtil.getInstance().getNickname());
        }

        gestureDrawTips = (TextView) findViewById(R.id.tv_gesture_tips);

        patternLockView = (PatternLockView) findViewById(R.id.pattern_lock_view);
        patternLockView.addPatternLockListener(this);

        gestureDrawTipsNormalColor = Color.parseColor("#5F5F5F");
        gestureDrawTipsWrongColor = Color.parseColor("#CD4849");
    }

    @Override
    protected void onResume() {
        super.onResume();
        long sinceLastLockedTime = System.currentTimeMillis() - SettingInfoUtil.getInstance().getGestureLockedTimestamp();
        if (sinceLastLockedTime > 1000 && sinceLastLockedTime <= GESTURE_LOCK_TIME) {
            patternLockView.setEnabled(false);
            startCountDown(GESTURE_LOCK_TIME - sinceLastLockedTime);
        }
    }

    @Override
    public void onStarted() {
        gestureDrawTips.setTextColor(gestureDrawTipsNormalColor);
        gestureDrawTips.setText(R.string.uac_release_finger);
    }

    @Override
    public void onProgress(List<PatternLockView.Dot> progressPattern) {

    }

    @Override
    public void onComplete(List<PatternLockView.Dot> pattern) {
        handleGestureDraw(pattern);
        patternLockView.clearPattern();
    }

    @Override
    public void onCleared() {

    }

    private void handleGestureDraw(List<PatternLockView.Dot> dots) {
        if (dots.size() < 4) {
            gestureDrawTips.setTextColor(gestureDrawTipsWrongColor);
            gestureDrawTips.setText(R.string.uac_at_least_four_dots);
        } else {
            verifyGesturePwd(PatternLockUtils.patternToSha1(patternLockView, dots));
        }
    }

    /**
     * 验证手势密码
     */
    private void verifyGesturePwd(String gesturePwd) {
        if (SettingInfoUtil.getInstance().checkGesturePwd(gesturePwd)) {
            gesturePwdWrongCount = 0;
            SettingInfoUtil.getInstance().setGestureLockedTimestamp(0L);
            getAuthCode();
        } else {
            gestureDrawTips.setTextColor(gestureDrawTipsWrongColor);
            gesturePwdWrongCount++;
            if (gesturePwdWrongCount < MAX_WRONG_COUNT) {
                gestureDrawTips.setText(getString(R.string.uac_gesture_pwd_wrong, MAX_WRONG_COUNT - gesturePwdWrongCount));
            } else {
                gestureDrawTips.setText(getString(R.string.uac_verify_gesture_fail_too_much, GESTURE_LOCK_TIME));
                patternLockView.setEnabled(false);
                startCountDown(GESTURE_LOCK_TIME);
                SettingInfoUtil.getInstance().setGestureLockedTimestamp(System.currentTimeMillis());
            }
        }
    }

    private void getAuthCode() {
        showLoadingDialog(R.string.uac_login_ing);
        if (AuthModeUtil.getInstance().isOpenPlatformApp()) {
            ThreadPoolManager.getInstance().execute(
                    new GetAuthCodeThread(getApplicationContext(), uacHandler, bundle));
        } else {

            if (AuthModeUtil.getInstance().isUserPwdAuthMode()) {
                ThreadPoolManager.getInstance().execute(
                        new UserPwdLoginThread(uacHandler, UserInfoUtil.getInstance().getUserName(),
                                UserInfoUtil.getInstance().getPassword(), bundle));
                return;
            }

            if (AuthModeUtil.getInstance().isCertAuthMode()) {
                ThreadPoolManager.getInstance().execute(
                        new CertLoginThread(getApplicationContext(), uacHandler,
                                SafeCardUtil.getInstance().getPin(), bundle));
            }
        }

    }

    @Override
    public void handleThreadMessage(Message msg) {
        super.handleThreadMessage(msg);
        int errorCode = msg.what;
        if (errorCode == 0) {
            SettingInfoUtil.getInstance().updatePrivacyProtectTimestamp();
            UacActivityManager.getInstance().finishAllActivity();
        } else if (ErrorCodeUtil.isPasswordError(errorCode)) {
            showUacDialog(R.string.uac_password_modifyed, true);
            UserInfoUtil.getInstance().setPassword("");
        } else if (errorCode == ErrorCode.RET_VERIFY_SAFE_PIN_ERR) {
            //VHSM卡网络错误，直接提示，否则提示用户切换到PIN码登录
            if (ErrorCodeUtil.isVHSMNetworkError(ErrorCodeUtil.pinErrorCode)) {
                showErrorDialog(ErrorCodeUtil.pinErrorCode);
            } else {
                showUacDialog(R.string.uac_pin_modifyed, true);
            }
        } else if (errorCode == ErrorCode.RET_NEED_VERIFY_PIN ||
                errorCode == ErrorCode.RET_GET_PIN_NULL) {
            showUacDialog(R.string.uac_pin_modifyed, true);
        } else {
            showErrorDialog(msg.what);
        }
    }
    /**
     * 开始倒计时
     * 调试中发现#CountDownTimer start之后第一次onTick输出的#millisUntilFinished会比#millisInFuture少若干毫秒（视具体手机而定）
     * 导致取整的时候与预期值不符，因此对#millisInFuture采用加上二分之一#countDownInterval的处理方式
     */
    private void startCountDown(long time) {
        Log.d("GestureLoginActivity", "startCountDown(), time = " + time);
        if (countDownTimer == null) {

            final long countDownInterval = 1000L;
            countDownTimer = new CountDownTimer(time + countDownInterval / 2, countDownInterval) {
                @Override
                public void onTick(long millisUntilFinished) {
                    if (millisUntilFinished < countDownInterval) {
                        //不显示0值(某些手机默认不会输出小于#countDownInterval的值)
                        return;
                    }
                    gestureDrawTips.setTextColor(gestureDrawTipsWrongColor);
                    gestureDrawTips.setText(getString(R.string.uac_verify_gesture_fail_too_much,
                            millisUntilFinished / 1000));
                }

                @Override
                public void onFinish() {
                    Log.d("GestureLoginActivity", "onFinish()");
                    countDownTimer = null;
                    gestureDrawTips.setTextColor(gestureDrawTipsNormalColor);
                    gestureDrawTips.setText(R.string.uac_auth_gesture_tip);
                    patternLockView.setEnabled(true);
                }
            }.start();
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        if (countDownTimer != null) {
            countDownTimer.cancel();
            countDownTimer = null;
        }
    }
}
