package com.xdja.uniteauth.presenter;


import android.app.Activity;
import android.content.Intent;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.support.annotation.StringRes;
import android.support.v4.hardware.fingerprint.FingerprintManagerCompat;
import android.view.View;
import android.widget.Toast;

import com.xdja.uniteauth.R;
import com.xdja.uniteauth.jar.ParamKeywords;
import com.xdja.uniteauth.utils.AuthModeUtil;
import com.xdja.uniteauth.utils.ErrorCodeUtil;
import com.xdja.uniteauth.utils.UacActivityManager;
import com.xdja.uniteauth.utils.UacColorConfig;
import com.xdja.uniteauth.utils.UacConstant;
import com.xdja.uniteauth.utils.UserInfoUtil;
import com.xdja.uniteauth.view.UacCommonDialog;
import com.xdja.uniteauth.view.UacFingerprintDialog;
import com.xdja.uniteauth.view.UacInputDialog;
import com.xdja.uniteauth.view.UacLoadingDialog;

import java.lang.ref.WeakReference;

/**
 * @author kgg
 */
public class UacBaseActivity extends Activity {

    private UacCommonDialog uacCommonDialog;

    private UacLoadingDialog uacLoadingDialog;

    protected UacHandler uacHandler = new UacHandler(new WeakReference<>(this));

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        UacActivityManager.getInstance().addActivity(this);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            getWindow().setStatusBarColor(UacColorConfig.getInstance().getTitleBarColor());
        }

    }


    protected final void showUacDialog(String message) {
        uacCommonDialog = new UacCommonDialog(this, UacCommonDialog.SINGLE_BUTTON);
        uacCommonDialog.setSummary(message);
        uacCommonDialog.setConfirmButton(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                uacCommonDialog.dismiss();
            }
        });
        uacCommonDialog.show();
    }

    protected final void showUacDialog(@StringRes int stringId) {
        showUacDialog(stringId, false);
    }

    protected final void showUacDialog(@StringRes int stringId, final boolean finishActivity) {
        uacCommonDialog = new UacCommonDialog(this, UacCommonDialog.SINGLE_BUTTON);
        uacCommonDialog.setSummary(stringId);
        uacCommonDialog.setConfirmButton(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                uacCommonDialog.dismiss();
                if (finishActivity) {
                    finish();
                }
            }
        });
        uacCommonDialog.show();
    }

    protected final void showErrorDialog(int errorCode) {
        showUacDialog(ErrorCodeUtil.getErrorMsg(errorCode));
    }

    protected final void hideUacDialog() {
        if (uacCommonDialog != null && uacCommonDialog.isShowing()) {
            uacCommonDialog.dismiss();
        }
    }

    protected final void showErrorToast(int error) {
        showToast(ErrorCodeUtil.getErrorMsg(error));
    }

    protected final void showToast(String message) {
        Toast.makeText(this, message, Toast.LENGTH_SHORT).show();
    }

    protected final void showToast(@StringRes int stringId) {
        Toast.makeText(this, stringId, Toast.LENGTH_SHORT).show();
    }

    protected void showLoadingDialog(String msg) {
        if (uacLoadingDialog == null) {
            uacLoadingDialog = new UacLoadingDialog(this);
            uacLoadingDialog.setCancelable(false);
        }

        uacLoadingDialog.setSummary(msg);
        uacLoadingDialog.show();
    }

    protected void showLoadingDialog(@StringRes int stringId) {
        if (uacLoadingDialog == null) {
            uacLoadingDialog = new UacLoadingDialog(this);
            uacLoadingDialog.setCancelable(false);
        }

        uacLoadingDialog.setSummary(stringId);
        uacLoadingDialog.show();
    }

    protected void hideLoadingDialog() {
        if (uacLoadingDialog != null && uacLoadingDialog.isShowing()) {
            uacLoadingDialog.dismiss();
        }
    }

    /**
     * 处理子线程消息
     *
     * @param msg Message
     */
    public void handleThreadMessage(Message msg) {
        hideLoadingDialog();
    }


    @Override
    protected void onDestroy() {
        super.onDestroy();
        UacActivityManager.getInstance().removeActivity(this);
        hideUacDialog();
        hideLoadingDialog();
    }

    protected void setViewEnable(View view, boolean flag) {
        if (view == null) {
            return;
        }
        if (flag) {
            view.setEnabled(true);
            view.setAlpha(1f);
            return;
        }
        view.setEnabled(false);
        view.setAlpha(0.5f);
    }


    public static class UacHandler extends Handler {
        private WeakReference<UacBaseActivity> activityWeakReference;

        UacHandler(WeakReference<UacBaseActivity> activityWeakReference) {
            this.activityWeakReference = activityWeakReference;
        }

        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);
            UacBaseActivity baseActivity = activityWeakReference.get();
            if (baseActivity == null) {
                return;
            }

            if (baseActivity.isFinishing() || baseActivity.isDestroyed()) {
                return;
            }
            baseActivity.handleThreadMessage(msg);
        }
    }

    protected void gotoUserLoginActivity(Bundle bundle) {
        Intent intent = new Intent(this, UserPwdLoginActivity.class);
        intent.putExtra(ParamKeywords.KEY_Bundle_bundle, bundle);
        startActivity(intent);
        finish();
    }

    protected void gotoCertLoginActivity(Bundle bundle) {
        Intent intent = new Intent(this, CertLoginActivity.class);
        intent.putExtra(ParamKeywords.KEY_Bundle_bundle, bundle);
        startActivity(intent);
        finish();
    }

    protected void showFingerprintLoginDialog(UacFingerprintDialog.SuccessCallback successCallback) {

        FingerprintManagerCompat fingerprintManagerCompat = FingerprintManagerCompat.from(this);
        if (!fingerprintManagerCompat.hasEnrolledFingerprints()) {
            final UacCommonDialog uacCommonDialog = new UacCommonDialog(this, UacCommonDialog.TWO_BUTTON);
            uacCommonDialog.setSummary(R.string.uac_fingerprint_not_enroll);
            uacCommonDialog.setCancelButton(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    uacCommonDialog.dismiss();
                }
            });

            uacCommonDialog.setConfirmButton(R.string.uac_setting_now, new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    uacCommonDialog.dismiss();
                    Intent intent = new Intent(android.provider.Settings.ACTION_SETTINGS);
                    startActivity(intent); // 打开系统设置界面
                }
            });

            uacCommonDialog.show();
            return;
        }

        UacFingerprintDialog fingerprintDialog;
        if (AuthModeUtil.getInstance().isSupportSoterFingerprint()) {
            fingerprintDialog = new UacFingerprintDialog(this, UacFingerprintDialog.FINGERPRINT_TYPE_SOTER);
        } else {
            fingerprintDialog = new UacFingerprintDialog(this, UacFingerprintDialog.FINGERPRINT_TYPE_SYSTEM);
        }

        fingerprintDialog.setSuccessCallback(successCallback);

        if (AuthModeUtil.getInstance().isUserPwdAuthMode()) {
            fingerprintDialog.setDialogTitle(UserInfoUtil.getInstance().getNickname());
        }

        fingerprintDialog.show();
    }

    protected void showFingerprintLoginNotEnabledDialog(final Bundle bundle) {
        final UacCommonDialog dialog = new UacCommonDialog(this, UacCommonDialog.TWO_BUTTON);
        dialog.setSummary(R.string.uac_auth_fingerprint_not_open);
        dialog.setCancelButton(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dialog.dismiss();
            }
        });

        dialog.setConfirmButton(R.string.uac_goto_open, new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                bundle.putBoolean(UacConstant.KEY_ENABLE_FINGERPRINT, true);
                bundle.putBoolean(UacConstant.KEY_NEED_VERIFY_IDENTITY, true);
                bundle.putBoolean(UacConstant.KEY_ENABLE_GESTURE, false);
                dialog.dismiss();
                gotoSettingActivity(bundle);
            }
        });

        dialog.show();
    }

    protected void gotoGestureLoginActivity(final Bundle bundle) {
        Intent intent = new Intent(this, GestureLoginActivity.class);
        intent.putExtra(ParamKeywords.KEY_Bundle_bundle, bundle);
        startActivity(intent);
    }

    protected void showGestureLoginNotEnabledDialog(final Bundle bundle) {
        final UacCommonDialog dialog = new UacCommonDialog(this, UacCommonDialog.TWO_BUTTON);
        dialog.setSummary(R.string.uac_auth_gesture_not_open);
        dialog.setCancelButton(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dialog.dismiss();
            }
        });

        dialog.setConfirmButton(R.string.uac_goto_open, new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                bundle.putBoolean(UacConstant.KEY_ENABLE_GESTURE, true);
                bundle.putBoolean(UacConstant.KEY_NEED_VERIFY_IDENTITY, true);
                bundle.putBoolean(UacConstant.KEY_ENABLE_FINGERPRINT, false);
                dialog.dismiss();
                gotoSettingActivity(bundle);
            }
        });

        dialog.show();
    }

    protected void showInputUserPwdDialog(UacInputDialog.InputCompleteCallback callback) {
        UacInputDialog dialog = new UacInputDialog(this, UacInputDialog.INPUT_PASSWORD);
        dialog.setInputCompleteCallback(callback);
        dialog.show();
    }

    protected void showInputPinDialog(UacInputDialog.InputCompleteCallback callback) {
        UacInputDialog dialog = new UacInputDialog(this, UacInputDialog.INPUT_PIN);
        dialog.setInputCompleteCallback(callback);
        dialog.show();
    }


    protected void gotoSettingActivity(Bundle bundle) {
        Intent intent = new Intent(this, UacSettingActivity.class);
        intent.putExtra(ParamKeywords.KEY_Bundle_bundle, bundle);
        startActivity(intent);
    }

    protected void gotoIdpChangedActivity(Bundle bundle) {
        Intent intent = new Intent(this, IdpChangedActivity.class);
        intent.putExtra(ParamKeywords.KEY_Bundle_bundle, bundle);
        startActivity(intent);
    }

}
