package com.xdja.uniteauth.presenter;


import android.content.Intent;
import android.os.Bundle;
import android.os.Message;
import android.support.v4.hardware.fingerprint.FingerprintManagerCompat;
import android.text.TextUtils;
import android.view.View;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.Switch;
import android.widget.TextView;

import com.xdja.uniteauth.R;
import com.xdja.uniteauth.ThreadPoolManager;
import com.xdja.uniteauth.domain.CertLoginThread;
import com.xdja.uniteauth.domain.GetAuthCodeThread;
import com.xdja.uniteauth.domain.VerifyUserPwdThread;
import com.xdja.uniteauth.jar.ErrorCode;
import com.xdja.uniteauth.jar.ParamKeywords;
import com.xdja.uniteauth.utils.AuthModeUtil;
import com.xdja.uniteauth.utils.ErrorCodeUtil;
import com.xdja.uniteauth.utils.SafeCardUtil;
import com.xdja.uniteauth.utils.SettingInfoUtil;
import com.xdja.uniteauth.utils.UacActivityManager;
import com.xdja.uniteauth.utils.UacConstant;
import com.xdja.uniteauth.utils.UserInfoUtil;
import com.xdja.uniteauth.view.UacCommonDialog;
import com.xdja.uniteauth.view.UacInputDialog;
import com.xdja.uniteauth.view.UacTimeIntervalDialog;
import com.xdja.uniteauth.view.UacTitleBar;

/**
 * @author kgg
 */
public class UacSettingActivity extends UacBaseActivity {

    public static final int OPERATE_TYPE_ENABLE_FINGERPRINT = 1;

    public static final int OPERATE_TYPE_ENABLE_GESTURE = 2;

    public static final int OPERATE_TYPE_MODIFY_GESTURE = 3;

    public static final int OPERATE_TYPE_MODIFY_PRIVACY_PROTECT = 4;

    private final int REQUEST_CODE_VERIFY_USER_PWD = 0x01;

    private final int REQUEST_CODE_SETTING_GESTURE = 0x02;

    private final int REQUEST_CODE_SETTING_FINGERPRINT = 0x03;

    private final int REQUEST_CODE_MODIFY_GESTURE = 0x04;

    private RelativeLayout modifyGestureLayout;

    private LinearLayout privacyProtectLayout;

    private TextView timeIntervalText;

    private Switch fingerprintSwitch;

    private Switch gestureSwitch;

    private int operateType;

    private Bundle bundle;

    private boolean needVerifyIdentity = true;

    private boolean needVerifyLocalIdentity = true;

    private boolean autoEnableFingerprintLogin;

    private boolean autoEnableGestureLogin;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.uac_activity_setting);
        bundle = getIntent().getBundleExtra(ParamKeywords.KEY_Bundle_bundle);
        needVerifyIdentity = bundle.getBoolean(UacConstant.KEY_NEED_VERIFY_IDENTITY, true);
        autoEnableFingerprintLogin = bundle.getBoolean(UacConstant.KEY_ENABLE_FINGERPRINT, false);
        autoEnableGestureLogin = bundle.getBoolean(UacConstant.KEY_ENABLE_GESTURE, false);
        initView();
    }

    private void initView() {

        UacTitleBar titleBar = (UacTitleBar) findViewById(R.id.uac_title_bar);
        titleBar.setBackListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                finish();
            }
        });

        LinearLayout fingerprintLayout = (LinearLayout) findViewById(R.id.layout_fingerprint);
        if (!AuthModeUtil.getInstance().isSupportFingerprintLogin()) {
            fingerprintLayout.setVisibility(View.GONE);
        }

        fingerprintSwitch = (Switch) findViewById(R.id.switch_fingerprint);
        fingerprintSwitch.setChecked(SettingInfoUtil.getInstance().isFingerprintLoginEnabled());

        View fingerprintSwitchMask = findViewById(R.id.switch_fingerprint_mask);
        fingerprintSwitchMask.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (fingerprintSwitch.isChecked()) {
                    disableFingerprintLogin();
                } else {
                    enableFingerprintAuth();
                }
            }
        });


        LinearLayout gestureLayout = (LinearLayout) findViewById(R.id.layout_gesture);
        if (!AuthModeUtil.getInstance().isSupportGestureLogin()) {
            gestureLayout.setVisibility(View.GONE);
        }

        gestureSwitch = (Switch) findViewById(R.id.switch_gesture);
        gestureSwitch.setChecked(SettingInfoUtil.getInstance().isGestureLoginEnabled());

        View gestureSwitchMask = findViewById(R.id.switch_gesture_mask);
        gestureSwitchMask.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (gestureSwitch.isChecked()) {
                    disableGestureAuth();
                } else {
                    enableGestureAuth();
                }
            }
        });

        modifyGestureLayout = (RelativeLayout) findViewById(R.id.layout_modify_gesture);
        if (!SettingInfoUtil.getInstance().isGestureLoginEnabled()) {
            modifyGestureLayout.setVisibility(View.GONE);
        }
        modifyGestureLayout.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                modifyGesturePwd();
            }
        });

        privacyProtectLayout = (LinearLayout) findViewById(R.id.layout_time_interval);
        //指纹、手势均未开启不能设置隐私保护
        if (!SettingInfoUtil.getInstance().isFingerprintLoginEnabled() &&
                !SettingInfoUtil.getInstance().isGestureLoginEnabled()) {
            privacyProtectLayout.setVisibility(View.GONE);
        }

        timeIntervalText = (TextView) findViewById(R.id.tv_time_interval);
        timeIntervalText.setText(SettingInfoUtil.getInstance().getPrivacyProtectIntervalString());
        privacyProtectLayout.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                modifyPrivacyProtect();
            }
        });

        if (autoEnableFingerprintLogin) {
            fingerprintSwitchMask.performClick();
        }

        if (autoEnableGestureLogin) {
            gestureSwitchMask.performClick();
        }
    }

    /**
     * 开启指纹认证
     */
    private void enableFingerprintAuth() {
        operateType = OPERATE_TYPE_ENABLE_FINGERPRINT;

        FingerprintManagerCompat fingerprintManagerCompat = FingerprintManagerCompat.from(this);
        if (!fingerprintManagerCompat.hasEnrolledFingerprints()) {
            final UacCommonDialog uacCommonDialog = new UacCommonDialog(this, UacCommonDialog.TWO_BUTTON);
            uacCommonDialog.setSummary(R.string.uac_fingerprint_not_enroll);
            uacCommonDialog.setCancelButton(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    uacCommonDialog.dismiss();
                }
            });

            uacCommonDialog.setConfirmButton(R.string.uac_setting_now, new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    uacCommonDialog.dismiss();
                    Intent intent = new Intent(android.provider.Settings.ACTION_SETTINGS);
                    startActivity(intent); // 打开系统设置界面
                }
            });

            uacCommonDialog.show();
            return;
        }
        if (needVerifyIdentity) {
            verifyIdentityAndGetAuthCode();
        } else {
            handleGetAuthCodeSuccess();
        }

    }

    /**
     * 关闭指纹登录
     */
    private void disableFingerprintLogin() {
        fingerprintSwitch.setChecked(false);
        SettingInfoUtil.getInstance().disableFingerprintLogin();
        //如果手势登录未开启，则还要隐藏隐私保护设置入口并将隐私保护间隔置为0
        if (!SettingInfoUtil.getInstance().isGestureLoginEnabled()) {
            privacyProtectLayout.setVisibility(View.GONE);
            SettingInfoUtil.getInstance().setPrivacyProtectInterval(0);
        }
    }

    /**
     * 开启手势认证
     */
    private void enableGestureAuth() {
        operateType = OPERATE_TYPE_ENABLE_GESTURE;
        if (needVerifyIdentity) {
            verifyIdentityAndGetAuthCode();
        } else {
            handleGetAuthCodeSuccess();
        }

    }

    /**
     * 关闭手势认证
     */
    private void disableGestureAuth() {
        gestureSwitch.setChecked(false);
        modifyGestureLayout.setVisibility(View.GONE);
        SettingInfoUtil.getInstance().disableGestureLogin();
        //如果指纹登录未开启，则还要隐藏隐私保护设置入口并将隐私保护间隔置为0
        if (!SettingInfoUtil.getInstance().isFingerprintLoginEnabled()) {
            privacyProtectLayout.setVisibility(View.GONE);
            SettingInfoUtil.getInstance().setPrivacyProtectInterval(0);
        }
    }

    /**
     * 修改手势密码
     */
    private void modifyGesturePwd() {
        operateType = OPERATE_TYPE_MODIFY_GESTURE;
        if (needVerifyLocalIdentity) {
            verifyLocalIdentity();
        } else {
            handleVerifyLocalIdentitySuccess();
        }

    }

    /**
     * 修改隐私保护间隔
     */
    private void modifyPrivacyProtect() {
        operateType = OPERATE_TYPE_MODIFY_PRIVACY_PROTECT;
        if (needVerifyLocalIdentity) {
            verifyLocalIdentity();
        } else {
            handleVerifyLocalIdentitySuccess();
        }
    }

    /**
     * 验证身份并获取授权码
     */
    private void verifyIdentityAndGetAuthCode() {
        if (AuthModeUtil.getInstance().isUserPwdAuthMode()) {
            verifyUserPwdAndGetAuthCode();
        } else {
            verifyPinAndGetAuthCode();
        }
    }

    /**
     * 验证PIN并获取授权码
     */
    private void verifyPinAndGetAuthCode() {
        showInputPinDialog(new UacInputDialog.InputCompleteCallback() {
            @Override
            public void onComplete(UacInputDialog dialog, CharSequence inputContent) {
                if (TextUtils.isEmpty(inputContent.toString().trim())) {
                    showToast(R.string.uac_pin_input_null);
                    return;
                }

                dialog.dismiss();
                showLoadingDialog(R.string.uac_pin_verify_ing);
                if (AuthModeUtil.getInstance().isOpenPlatformApp()) {
                    ThreadPoolManager.getInstance().execute(
                            new GetAuthCodeThread(getApplicationContext(), uacHandler, bundle, inputContent.toString()));
                } else {
                    ThreadPoolManager.getInstance().execute(
                            new CertLoginThread(getApplicationContext(), uacHandler, inputContent.toString(), bundle));
                }

            }
        });
    }

    /**
     * 验证用户密码并获取授权码
     */
    private void verifyUserPwdAndGetAuthCode() {
        final String userName = UserInfoUtil.getInstance().getUserName();
        //用户名为空则进入新界面验证用户名密码，不空则在当前界面弹框输入密码进行验证
        if (TextUtils.isEmpty(userName)) {
            Intent intent = new Intent(this, VerifyUserPwdActivity.class);
            intent.putExtra(ParamKeywords.KEY_Bundle_bundle, bundle);
            intent.putExtra(UacConstant.KEY_OPERATE_TYPE, operateType);
            startActivityForResult(intent, REQUEST_CODE_VERIFY_USER_PWD);
        } else {
            showInputUserPwdDialog(new UacInputDialog.InputCompleteCallback() {
                @Override
                public void onComplete(UacInputDialog dialog, CharSequence inputContent) {
                    if (TextUtils.isEmpty(inputContent.toString().trim())) {
                        showToast(R.string.uac_password_is_null);
                        return;
                    }

                    dialog.dismiss();
                    showLoadingDialog(R.string.uac_verify_password_ing);
                    ThreadPoolManager.getInstance().execute(
                            new VerifyUserPwdThread(uacHandler, userName, inputContent.toString(), bundle));

                }
            });
        }
    }

    /**
     * 验证本地身份，如果是信大捷安芯片模式验证PIN码，否则验证用户密码
     */
    private void verifyLocalIdentity() {
        if (AuthModeUtil.getInstance().isUserPwdAuthMode()) {
            verifyUserPwd();
        } else {
            verifyPin();
        }
    }

    /**
     * 验证PIN
     */
    private void verifyPin() {
        showInputPinDialog(new UacInputDialog.InputCompleteCallback() {
            @Override
            public void onComplete(UacInputDialog dialog, final CharSequence inputContent) {
                if (TextUtils.isEmpty(inputContent.toString().trim())) {
                    showToast(R.string.uac_pin_input_null);
                    return;
                }

                dialog.dismiss();
                showLoadingDialog(R.string.uac_verify_ing);
                ThreadPoolManager.getInstance().execute(new Runnable() {
                    @Override
                    public void run() {
                        final int ret = SafeCardUtil.getInstance().verifyPin(inputContent.toString());
                        runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                hideLoadingDialog();
                                if (ret == 0) {
                                    handleVerifyLocalIdentitySuccess();
                                } else {
                                    ErrorCodeUtil.pinErrorCode = ret;
                                    showUacDialog(ErrorCodeUtil.getPinErrorMsg());
                                }
                            }
                        });
                    }
                });

            }
        });
    }

    private void verifyUserPwd() {
        showInputUserPwdDialog(new UacInputDialog.InputCompleteCallback() {
            @Override
            public void onComplete(UacInputDialog dialog, CharSequence inputContent) {
                if (TextUtils.isEmpty(inputContent.toString().trim())) {
                    showToast(R.string.uac_password_is_null);
                    return;
                }

                dialog.dismiss();
                if (inputContent.toString().equals(UserInfoUtil.getInstance().getPassword())) {
                    handleVerifyLocalIdentitySuccess();
                } else {
                    showUacDialog(R.string.uac_password_is_error);
                }
            }
        });


    }

    @Override
    public void handleThreadMessage(Message msg) {
        super.handleThreadMessage(msg);
        int errorCode = msg.what;
        if (errorCode == 0) {
            handleGetAuthCodeSuccess();
        } else if (errorCode == ErrorCode.RET_VERIFY_SAFE_PIN_ERR) {
            showUacDialog(ErrorCodeUtil.getPinErrorMsg());
        } else {
            showErrorDialog(errorCode);
        }
    }

    /**
     * 获取授权码成功之后进行下一步操作（设置指纹或者设置手势密码）
     */
    private void handleGetAuthCodeSuccess() {
        hideLoadingDialog();
        if (operateType == OPERATE_TYPE_ENABLE_FINGERPRINT) {
            //验证指纹, 成功后开启指纹认证
            startActivityForResult(new Intent(UacSettingActivity.this,
                    FingerprintSettingActivity.class), REQUEST_CODE_SETTING_FINGERPRINT);
        } else if (operateType == OPERATE_TYPE_ENABLE_GESTURE) {
            //设置手势密码，成功后开启手势认证
            startActivityForResult(new Intent(UacSettingActivity.this,
                    GestureSettingActivity.class), REQUEST_CODE_SETTING_GESTURE);
        }
    }

    /**
     * 验证本地身份成功后的下一步操作
     */
    private void handleVerifyLocalIdentitySuccess() {
        needVerifyLocalIdentity = false;
        if (operateType == OPERATE_TYPE_MODIFY_GESTURE) {
            //修改手势密码
            startActivityForResult(new Intent(UacSettingActivity.this,
                    GestureSettingActivity.class), REQUEST_CODE_MODIFY_GESTURE);
        } else if (operateType == OPERATE_TYPE_MODIFY_PRIVACY_PROTECT) {
            UacTimeIntervalDialog uacTimeIntervalDialog = new UacTimeIntervalDialog(UacSettingActivity.this);
            uacTimeIntervalDialog.setIntervalCheckListener(new UacTimeIntervalDialog.IntervalCheckListener() {
                @Override
                public void onChecked(int interval) {
                    timeIntervalText.setText(SettingInfoUtil.getInstance().getPrivacyProtectIntervalString());
                }
            });
            uacTimeIntervalDialog.show();
        }
    }


    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        if (resultCode == RESULT_OK) {
            if (requestCode == REQUEST_CODE_VERIFY_USER_PWD) {
                //验证用户密码成功
                handleGetAuthCodeSuccess();
            }

            if (requestCode == REQUEST_CODE_SETTING_FINGERPRINT) {
                //指纹设置成功
                fingerprintSwitch.setChecked(true);
                privacyProtectLayout.setVisibility(View.VISIBLE);
                SettingInfoUtil.getInstance().updatePrivacyProtectTimestamp();
                UacActivityManager.getInstance().finishAllActivity();
            }

            if (requestCode == REQUEST_CODE_SETTING_GESTURE) {
                //手势设置成功
                gestureSwitch.setChecked(true);
                privacyProtectLayout.setVisibility(View.VISIBLE);
                modifyGestureLayout.setVisibility(View.VISIBLE);
                SettingInfoUtil.getInstance().updatePrivacyProtectTimestamp();
                UacActivityManager.getInstance().finishAllActivity();
            }

            if (requestCode == REQUEST_CODE_MODIFY_GESTURE) {
                //手势修改成功
                showToast(R.string.uac_modify_gesture_pwd_success);
            }
        }
    }
}
