package com.xdja.uniteauth.presenter;


import android.os.Bundle;
import android.os.Message;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.text.method.HideReturnsTransformationMethod;
import android.text.method.PasswordTransformationMethod;
import android.view.View;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.TextView;

import com.xdja.uniteauth.R;
import com.xdja.uniteauth.ThreadPoolManager;
import com.xdja.uniteauth.bean.LoginMode;
import com.xdja.uniteauth.domain.UserPwdLoginThread;
import com.xdja.uniteauth.utils.AuthModeUtil;
import com.xdja.uniteauth.utils.ErrorCodeUtil;
import com.xdja.uniteauth.utils.SettingInfoUtil;
import com.xdja.uniteauth.utils.UacActivityManager;
import com.xdja.uniteauth.utils.UserInfoUtil;
import com.xdja.uniteauth.view.UacCommonDialog;
import com.xdja.uniteauth.view.UacFingerprintDialog;
import com.xdja.uniteauth.view.UacImageView;
import com.xdja.uniteauth.view.UacInputDialog;

/**
 * @author kgg
 */
public class UserPwdLoginActivity extends BaseLoginActivity {

    private View userInfoLayout;
    private View userNameInputLayout;
    private View userPwdInputLayout;

    private EditText userNameEt;

    private EditText passwordEt;

    private Button loginBt;

    /**
     * 验证通过待添加的指纹ID
     */
    private String toAddFid;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.uac_activity_user_login);
        initTitle();
        initView();
        initBottomSheetDialog();
        initAuthMode();
    }


    @Override
    protected void initView() {
        super.initView();

        UacImageView appLogo = (UacImageView) findViewById(R.id.iv_app_logo);
        appLogo.setImageUrl(AuthModeUtil.getInstance().getAppLogo());

        TextView appName = (TextView) findViewById(R.id.tv_app_name);
        appName.setText(AuthModeUtil.getInstance().getAppName());

        userInfoLayout = findViewById(R.id.layout_user_info);
        userNameInputLayout = findViewById(R.id.layout_user_name_input);
        userPwdInputLayout = findViewById(R.id.layout_user_pwd_input);

        userNameEt = (EditText) findViewById(R.id.et_user_name);
        userNameEt.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {

            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {

            }

            @Override
            public void afterTextChanged(Editable s) {
                setViewEnable(loginBt, s.length() > 0 && passwordEt.length() > 0);
            }
        });

        passwordEt = (EditText) findViewById(R.id.et_password);
        passwordEt.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {

            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {

            }

            @Override
            public void afterTextChanged(Editable s) {
                setViewEnable(loginBt, s.length() > 0 && userNameEt.length() > 0);
            }
        });
        passwordEt.setTransformationMethod(PasswordTransformationMethod.getInstance());

        CheckBox checkBox = (CheckBox) findViewById(R.id.checkbox);
        checkBox.setOnCheckedChangeListener(new CompoundButton.OnCheckedChangeListener() {
            @Override
            public void onCheckedChanged(CompoundButton buttonView, boolean isChecked) {
                if (isChecked) {//显示密码
                    passwordEt.setTransformationMethod(HideReturnsTransformationMethod.getInstance());
                } else {//隐藏密码
                    passwordEt.setTransformationMethod(PasswordTransformationMethod.getInstance());
                }
                //光标放置到最后面
                passwordEt.setSelection(passwordEt.length());
            }
        });

        loginBt = (Button) findViewById(R.id.bt_login);
        loginBt.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                login();
            }
        });

        //用户名不存在，展示用户名输入框，否则展示用户信息框
        if (TextUtils.isEmpty(UserInfoUtil.getInstance().getUserName())) {
            userNameInputLayout.setVisibility(View.VISIBLE);
            userInfoLayout.setVisibility(View.GONE);
        } else {
            userNameInputLayout.setVisibility(View.GONE);
            userInfoLayout.setVisibility(View.VISIBLE);
            userNameEt.setText(UserInfoUtil.getInstance().getUserName());

            TextView userName = (TextView) findViewById(R.id.tv_user_name);
            userName.setText(UserInfoUtil.getInstance().getNickname());

            ImageView deleteUser = (ImageView) findViewById(R.id.iv_delete_user);
            deleteUser.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    deleteUser();
                }
            });
        }
    }



    private void initAuthMode() {
        //用户名或密码为空，只能使用密码登录
        if (TextUtils.isEmpty(UserInfoUtil.getInstance().getUserName()) ||
                TextUtils.isEmpty(UserInfoUtil.getInstance().getPassword())) {
            currentLoginMode = LoginMode.UserPwd;
            setViewEnable(loginBt, false);
        } else if (SettingInfoUtil.getInstance().isFingerprintLoginEnabled()) {
            currentLoginMode = LoginMode.FINGERPRINT;
            setViewEnable(loginBt, true);
            userPwdInputLayout.setVisibility(View.INVISIBLE);
            login();
        } else if (SettingInfoUtil.getInstance().isGestureLoginEnabled()) {
            currentLoginMode = LoginMode.GESTURE;
            setViewEnable(loginBt, true);
            userPwdInputLayout.setVisibility(View.INVISIBLE);
            login();
        } else {
            currentLoginMode = LoginMode.UserPwd;
            setViewEnable(loginBt, false);
        }

        refreshOtherLoginMode();
    }

    @Override
    protected void onStart() {
        super.onStart();
        //后台修改密码后切换到用户名密码登录
        if (TextUtils.isEmpty(UserInfoUtil.getInstance().getPassword()) ||
                TextUtils.isEmpty(UserInfoUtil.getInstance().getPassword())) {
            changeLoginMode(LoginMode.UserPwd);
            return;
        }

        //用户进入设置界面关闭了指纹登录，切换到用户名密码登录
        if (currentLoginMode == LoginMode.FINGERPRINT &&
                !SettingInfoUtil.getInstance().isFingerprintLoginEnabled()) {
            changeLoginMode(LoginMode.UserPwd);
            return;
        }

        //用户进入设置界面关闭了指纹登录，切换到用户名密码登录
        if (currentLoginMode == LoginMode.GESTURE &&
                !SettingInfoUtil.getInstance().isGestureLoginEnabled()) {
            changeLoginMode(LoginMode.UserPwd);
        }
    }

    @Override
    protected void changeLoginMode(LoginMode loginMode) {
        if (currentLoginMode == loginMode) {
            return;
        }

        if (loginMode == LoginMode.UserPwd) {
            currentLoginMode = loginMode;
            refreshOtherLoginMode();
            userPwdInputLayout.setVisibility(View.VISIBLE);
            setViewEnable(loginBt, false);
            return;
        }


        //指纹认证未开启
        if (loginMode == LoginMode.FINGERPRINT &&
                !SettingInfoUtil.getInstance().isFingerprintLoginEnabled()) {
            showFingerprintLoginNotEnabledDialog(bundle);
            return;
        }

        //手势认证未开启
        if (loginMode == LoginMode.GESTURE &&
                !SettingInfoUtil.getInstance().isGestureLoginEnabled()) {
            showGestureLoginNotEnabledDialog(bundle);
            return;
        }

        //用户密码为空
        if (TextUtils.isEmpty(UserInfoUtil.getInstance().getPassword())) {
            showUacDialog(R.string.uac_password_modifyed);
            return;
        }

        currentLoginMode = loginMode;
        refreshOtherLoginMode();
        userPwdInputLayout.setVisibility(View.INVISIBLE);
        passwordEt.setText("");
        setViewEnable(loginBt, true);
        login();
    }

    @Override
    protected void refreshOtherLoginMode() {
        super.refreshOtherLoginMode();

        if (currentLoginMode == LoginMode.UserPwd) {
            getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_PAN);
        } else {
            getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_NOTHING);
        }
    }

    private void login() {

        if (currentLoginMode == LoginMode.UserPwd) {
            String userName = userNameEt.getText().toString();
            String password = passwordEt.getText().toString();
            startLoginThread(userName, password);

        } else if (currentLoginMode == LoginMode.FINGERPRINT) {
            if (SettingInfoUtil.getInstance().isPrivacyProtectTimeout()) {
                showFingerprintLoginDialog(new UacFingerprintDialog.SuccessCallback() {
                    @Override
                    public void onSuccess(String fid) {
                        handleFingerprintSuccess(fid);
                    }
                });
            } else {
                startLoginThread();
            }

        } else if (currentLoginMode == LoginMode.GESTURE) {
            if (SettingInfoUtil.getInstance().isPrivacyProtectTimeout()) {
                gotoGestureLoginActivity(bundle);
            } else {
                startLoginThread();
            }

        }
    }

    private void handleFingerprintSuccess(String fid) {
        String addedFid = SettingInfoUtil.getInstance().getFingerprintId();
        if (TextUtils.isEmpty(fid)) {
            startLoginThread();
        } else if (addedFid.contains(fid)) {
            startLoginThread();
        } else {//引导用户添加指纹
            toAddFid = fid;
            final UacCommonDialog dialog = new UacCommonDialog(this, UacCommonDialog.TWO_BUTTON_WITH_FINGERPRINT);
            dialog.setSummary(R.string.uac_not_verify_fid);
            dialog.setCancelButton(R.string.uac_try_again, new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    dialog.dismiss();
                    login();
                }
            });

            dialog.setConfirmButton(R.string.uac_add_fid, new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    dialog.dismiss();
                    verifyUserPwdAndAddFid();
                }
            });

            dialog.show();
        }
    }

    private void startLoginThread() {
        startLoginThread(UserInfoUtil.getInstance().getUserName(), UserInfoUtil.getInstance().getPassword());
    }

    /**
     * 启动用户登录线程
     */
    private void startLoginThread(String userName, String password) {
        showLoadingDialog(R.string.uac_login_ing);
        ThreadPoolManager.getInstance().execute(
                new UserPwdLoginThread(uacHandler, userName, password, bundle));
    }

    /**
     * 验证用户密码并添加指纹id（验证通过后获取授权码）
     */
    private void verifyUserPwdAndAddFid() {

        showInputUserPwdDialog(new UacInputDialog.InputCompleteCallback() {
            @Override
            public void onComplete(UacInputDialog dialog, CharSequence inputContent) {
                if (TextUtils.isEmpty(inputContent.toString().trim())) {
                    showToast(R.string.uac_password_is_null);
                    return;
                }

                String userName = UserInfoUtil.getInstance().getUserName();
                dialog.dismiss();
                showLoadingDialog(R.string.uac_verify_password_ing);
                startLoginThread(userName, inputContent.toString());
            }
        });
    }

    /**
     * 删除当前用户
     */
    private void deleteUser() {
        final UacCommonDialog uacCommonDialog = new UacCommonDialog(this, UacCommonDialog.TWO_BUTTON);
        uacCommonDialog.setSummary(getString(R.string.uac_delete_user_tips, UserInfoUtil.getInstance().getNickname()));
        uacCommonDialog.setCancelButton(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                uacCommonDialog.dismiss();
            }
        });

        uacCommonDialog.setConfirmButton(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                UserInfoUtil.getInstance().clearUserInfo();
                changeLoginMode(LoginMode.UserPwd);
                userNameInputLayout.setVisibility(View.VISIBLE);
                userNameEt.setText("");
                userNameEt.requestFocus();
                passwordEt.setText("");
                userInfoLayout.setVisibility(View.GONE);
                uacCommonDialog.dismiss();
            }
        });

        uacCommonDialog.show();
    }

    @Override
    public void handleThreadMessage(Message msg) {
        super.handleThreadMessage(msg);
        int errorCode = msg.what;

        //用户登录线程结果处理
        if (errorCode == 0) {
            if (currentLoginMode == LoginMode.FINGERPRINT) {
                if (!TextUtils.isEmpty(toAddFid)) {
                    SettingInfoUtil.getInstance().setFingerprintId(toAddFid);
                }
                SettingInfoUtil.getInstance().updatePrivacyProtectTimestamp();
            }

            if (SettingInfoUtil.getInstance().isGestureLoginEnabled() ||
                    SettingInfoUtil.getInstance().isFingerprintLoginEnabled()) {
                UacActivityManager.getInstance().finishAllActivity();
            } else if (AuthModeUtil.getInstance().isSupportFingerprintLogin() ||
                    AuthModeUtil.getInstance().isSupportGestureLogin()) {
                recommendFingerprintGesture();
            } else {
                UacActivityManager.getInstance().finishAllActivity();
            }

        } else if (ErrorCodeUtil.isPasswordError(errorCode)) {
            if (currentLoginMode == LoginMode.UserPwd) {
                showErrorDialog(errorCode);
            } else {
                showUacDialog(R.string.uac_password_modifyed);
                UserInfoUtil.getInstance().setPassword("");
                changeLoginMode(LoginMode.UserPwd);
            }
        } else {
            showErrorDialog(msg.what);
        }
    }

}
