package com.xdja.uniteauth.utils;

import android.text.TextUtils;

import com.xdja.uniteauth.bean.LoginMode;
import com.xdja.uniteauth.data.net.model.GetAuthModeResponse;

import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.List;

/**
 * @author kgg
 * Date:2019/8/12
 * Time:10:48
 * Summary:
 * @author kgg
 */
public class AuthModeUtil {
    private static final AuthModeUtil INSTANCE = new AuthModeUtil();

    public static AuthModeUtil getInstance() {
        return INSTANCE;
    }

    /**
     * 认证方式，当前支持用户名密码（仅支持新平台）和数字证书（包括新平台和老平台）两种方式
     * 在这两种认证方式的基础上扩展了多种登录方式（用户名密码、证书PIN码，指纹、手势等）
     * 最终都是通过这两种认证方式完成身份认证
     */
    private String authMode;

    private String appName;

    private String appLogo;

    private List<LoginMode> loginModeList = new ArrayList<>();

    /**
     * 是否为open平台（老平台）注册的应用
     */
    private boolean openPlatformApp;

    private boolean differentIdp;

    /**
     * 手机设备是否支持腾讯soter指纹
     */
    private boolean supportSoterFingerprint;

    /**
     * 是够支持指纹登录
     */
    private boolean supportFingerprintLogin;

    /**
     * 是够支持手势登录
     */
    private boolean supportGestureLogin;

    /**
     * 是够有指纹硬件模块
     */
    private boolean hasFingerprintHardware;

    private AuthModeUtil() {
    }

    public void saveAuthModeResponse(GetAuthModeResponse response) {

        appName = response.getSpName();
        appLogo = response.getLogo();
        authMode = response.getAuthMode();

        if (response.getOpenSupportFlag() == 1) {
            openPlatformApp = true;
            loginModeList.add(LoginMode.OldCert);
        } else {
            openPlatformApp = false;

            if (TextUtils.isEmpty(response.getAuthMode()) ||
                    response.getAuthMode().contains(UacConstant.AUTH_MODE_USER_PWD)) {
                loginModeList.add(LoginMode.UserPwd);
                authMode = UacConstant.AUTH_MODE_USER_PWD;
            } else if (response.getAuthMode().contains(UacConstant.AUTH_MODE_CERT)) {
                loginModeList.add(LoginMode.Cert);
                authMode = UacConstant.AUTH_MODE_CERT;
            } else {
                loginModeList.add(LoginMode.UserPwd);
                authMode = UacConstant.AUTH_MODE_USER_PWD;
            }
        }

        String machineAuthModeFlag = response.getMachineAuthMode();
        if (!TextUtils.isEmpty(machineAuthModeFlag)) {
            if (machineAuthModeFlag.contains(UacConstant.MACHINE_AUTH_MODE_FINGERPRINT) && hasFingerprintHardware) {
                loginModeList.add(LoginMode.FINGERPRINT);
                supportFingerprintLogin = true;
            }

            if (machineAuthModeFlag.contains(UacConstant.MACHINE_AUTH_MODE_GESTURE)) {
                loginModeList.add(LoginMode.GESTURE);
                supportGestureLogin = true;
            }
        }

        int userIdpId = UserInfoUtil.getInstance().getIdpId();
        differentIdp = userIdpId != -1 && userIdpId != response.getIdpId();

        String redirectUri = response.getRedirect_uri();
        if (!openPlatformApp && !TextUtils.isEmpty(redirectUri)) {
            try {
                redirectUri = URLDecoder.decode(redirectUri, "UTF-8");
            } catch (Exception e) {
                e.printStackTrace();
            }
            CallbackAppUtil.getInstance().setRedirectUri(redirectUri);
        }

        SettingInfoUtil.getInstance().initSettingInfo();
        if (!supportFingerprintLogin && !differentIdp) {
            SettingInfoUtil.getInstance().disableFingerprintLogin();
        }

        if (!supportGestureLogin && !differentIdp) {
            SettingInfoUtil.getInstance().disableGestureLogin();
        }
    }

    public String getAppName() {
        return appName;
    }

    public String getAppLogo() {
        return appLogo;
    }

    public List<LoginMode> getLoginModeList() {
        return loginModeList;
    }

    public boolean isOpenPlatformApp() {
        return openPlatformApp;
    }

    public boolean isCertAuthMode() {
        return UacConstant.AUTH_MODE_CERT.equals(authMode);
    }

    public boolean isUserPwdAuthMode() {
        return UacConstant.AUTH_MODE_USER_PWD.equals(authMode);
    }

    public boolean isDifferentIdp() {
        return differentIdp;
    }

    public boolean isSupportFingerprintLogin() {
        return supportFingerprintLogin;
    }

    public boolean isSupportGestureLogin() {
        return supportGestureLogin;
    }

    public boolean isSupportSoterFingerprint() {
        return supportSoterFingerprint;
    }

    public void setSupportSoterFingerprint(boolean supportSoterFingerprint) {
        this.supportSoterFingerprint = supportSoterFingerprint;
    }

    public void setHasFingerprintHardware(boolean hasFingerprintHardware) {
        this.hasFingerprintHardware = hasFingerprintHardware;
    }

    public void reset() {
        authMode = "";
        appName = "";
        appLogo = "";
        differentIdp = false;
        openPlatformApp = false;
        supportGestureLogin = false;
        supportFingerprintLogin = false;
        loginModeList.clear();
    }
}
