package com.xdja.uniteauth.utils;

import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.RemoteException;
import android.text.TextUtils;
import android.util.Log;
import android.widget.Toast;

import com.xdja.uniteauth.data.net.model.GetAuthCodeResponse;
import com.xdja.uniteauth.data.net.model.LoginResponse;
import com.xdja.uniteauth.jar.ErrorCode;
import com.xdja.uniteauth.jar.IRealCallback;

import java.net.URLDecoder;
import java.net.URLEncoder;

/**
 * @author kgg
 * Date:2019/8/22
 * Time:9:45
 * Summary:回调三方App授权码结果工具类，
 */
public class CallbackAppUtil {

    private final String TAG = "CallbackAppUtilTag";

    /**
     * Binder方式回调，用于老平台应用
     */
    private static final int CALLBACK_MODE_BINDER = 1;
    /**
     * Scheme方式回调，用于新平台应用
     */
    private static final int CALLBACK_MODE_SCHEME = 2;

    private static final CallbackAppUtil INSTANCE = new CallbackAppUtil();

    public static CallbackAppUtil getInstance() {
        return INSTANCE;
    }

    private CallbackAppUtil() {
    }

    /**
     * 新平台应用scheme uri
     */
    private String redirectUri;

    /**
     * 代理应用应用scheme uri，当前代理应用为安全盒
     * 应用安装在安全盒内部，盒外无法通过redirectUri访问，只能使用agentUri
     */
    private String agentUri;

    /**
     * 老平台应用Binder，通过该Binder回调
     */
    private IRealCallback realCallback;

    private String state;

    /**
     * 回调给老平台应用的错误码
     */
    private int errorCode = ErrorCode.RET_GET_UAC_CODE_FAIL;

    private String extra;

    /**
     * 回调给新平台应用的错误信息
     */
    private String errorMsg;

    private String authCode;

    private int callbackMode;

    public void setRealCallback(IRealCallback realCallback) {
        Log.d(TAG, "setRealCallback()");
        this.realCallback = realCallback;
        callbackMode = CALLBACK_MODE_BINDER;
    }

    public void setRedirectUri(String redirectUri) {
        Log.d(TAG, "setRedirectUri(), redirectUri = " + redirectUri);
        this.redirectUri = redirectUri;
        callbackMode = CALLBACK_MODE_SCHEME;
    }

    public void setAgentUri(String agentUri) {
        Log.d(TAG, "setAgentUri(), agentUri = " + agentUri);
        this.agentUri = agentUri;
    }

    public void setState(String state) {
        this.state = state;
    }

    /**
     * 设置错误码（仅用于老平台应用数字证书登录场景）
     */
    public void setErrorCode(int errorCode) {
        Log.d(TAG, "setErrorCode(), errorCode = " + errorCode);
        //服务器端返回的业务错误码统一转换为RET_SERVER_RETURN_ERR
        if (ErrorCodeUtil.isServerErrCode(errorCode)) {
            this.errorCode = ErrorCode.RET_SERVER_RETURN_ERR;
            if (ErrorCodeUtil.isInvalidScope(errorCode)) {
                setErrorMsg("invalid_scope");
            }
            if (ErrorCodeUtil.isUnsupportedResponseType(errorCode)) {
                setErrorMsg("unsupported_response_type");
            }
        } else {
            this.errorCode = errorCode;
        }
    }

    private void setErrorMsg(String errorMsg) {
        this.errorMsg = errorMsg;
    }

    public void saveAuthCodeResponse(GetAuthCodeResponse authCodeResponse, String extra) {
        Log.d(TAG, "saveAuthCodeResponse()");
        this.errorCode = 0;
        this.extra = extra;
        this.state = authCodeResponse.getState();
        this.authCode = authCodeResponse.getCode();
    }

    public void saveLoginResponse(LoginResponse loginResponse) {
        Log.d(TAG, "saveLoginResponse()");
        this.errorCode = 0;
        this.authCode = loginResponse.getCode();

    }

    public void callbackApp(Context context) {
        if (callbackMode == CALLBACK_MODE_BINDER) {
            callbackAppThroughBinder();
        } else if (callbackMode == CALLBACK_MODE_SCHEME) {
            callbackAppThroughScheme(context);
        }
    }

    private void callbackAppThroughBinder() {
        Log.d(TAG, "callbackAppThroughBinder()");
        if (realCallback == null) {
            return;
        }

        //检查调用方进程是否在运行
        if (!realCallback.asBinder().pingBinder()) {
            return;
        }

        try {
            realCallback.callback(state, errorCode, authCode, extra);
        } catch (RemoteException e) {
            e.printStackTrace();
        } finally {
            realCallback = null;
            errorCode = ErrorCode.RET_GET_UAC_CODE_FAIL;
            authCode = "";
            extra = "";
            callbackMode = 0;
        }
    }

    private void callbackAppThroughScheme(Context context) {
        Log.d(TAG, "callbackAppThroughScheme()");
        if (TextUtils.isEmpty(redirectUri)) {
            return;
        }
        try {
            String uri = null;
            if (!TextUtils.isEmpty(authCode)) {
                uri = redirectUri + "?code=" + authCode + "&state=" + state;
            } else if (!TextUtils.isEmpty(errorMsg)) {
                uri = redirectUri + "?error=" + errorMsg + "&state=" + state;
            }

            //如果是通过agentUri回调，则要把uri作为参数，参数名为real_uri
            if (!TextUtils.isEmpty(agentUri)){
                uri = agentUri + "?real_uri=" + URLEncoder.encode(uri, "UTF-8");
            }

            Log.d(TAG, "uri = " + uri);

            if (!TextUtils.isEmpty(uri)){
                Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(uri));
                context.startActivity(intent);
            }

        } catch (Exception e) {
            Log.e(TAG, "redirect_uri error");
            Toast.makeText(context, "redirect_uri错误", Toast.LENGTH_SHORT).show();
        } finally {
            redirectUri = "";
            errorMsg = "";
            authCode = "";
            callbackMode = 0;
        }
    }
}
