package com.xdja.uniteauth.utils;

import android.content.Context;
import android.util.SparseArray;

import com.xdja.uniteauth.ApplicationContext;
import com.xdja.uniteauth.R;
import com.xdja.uniteauth.jar.ErrorCode;

import java.net.SocketTimeoutException;

/**
 * @author kgg
 * Date:2019/8/16
 * Time:10:22
 * Summary:
 */
public class ErrorCodeUtil {

    /**
     * PIN码验证错误码
     */
    public static int pinErrorCode;

    public static String idpErrorMsg;

    /**
     * idp认证失败，原因有多种，可能是密码错误，也肯能是别的原因
     * 将错误信息暂存到idpErrorMsg，客户端不再进行翻译
     */
    private static final int IDP_ERROR = 0X4212;

    public static int getErrorCode(Exception e) {
        int errorCode = ErrorCode.RET_EXCEPTION_NETWORK;
        if (e.getCause() instanceof SocketTimeoutException || e.toString().contains("timeout")) {
            errorCode = ErrorCode.RET_TIMEOUT_REQUEST_SERVER;
        }

        return errorCode;
    }

    public static SparseArray<String> serverErrCodeMap = new SparseArray<>();

    static {
        //系统公共异常(4001—0x4100)
        serverErrCodeMap.append(0X4001, "缺少必要请求参数");
        serverErrCodeMap.append(0X4002, "非法请求参数");
        serverErrCodeMap.append(0X4003, "请求方法不支持");
        serverErrCodeMap.append(0X4004, "服务内部异常");

        //业务异常(0x4201—暂无限制)
        serverErrCodeMap.append(0x4201, "应用未注册");
        serverErrCodeMap.append(0X4202, "应用包名不匹配");
        serverErrCodeMap.append(0X4203, "包公钥签名不匹配");
        serverErrCodeMap.append(0X4204, "挑战值无效");
        serverErrCodeMap.append(0X4205, "证书未录入资产库");
        serverErrCodeMap.append(0X4206, "验签失败");
        serverErrCodeMap.append(0X4207, "未知授权模式");
        serverErrCodeMap.append(0X4208, "授权码无效");
        serverErrCodeMap.append(0X4209, "应用密钥不一致");
        serverErrCodeMap.append(0X420a, "无效的刷新凭证");
        serverErrCodeMap.append(0X420b, "证书状态异常");
        serverErrCodeMap.append(0X420c, "无效的授权凭证");
        serverErrCodeMap.append(0X4210, "应用被停用");
        serverErrCodeMap.append(0X4211, "idp未注册");
        serverErrCodeMap.append(0X4213, "用户已认证");
        serverErrCodeMap.append(0X4214, "用户名或密码错误");
        serverErrCodeMap.append(0X4215, "人员安全卡不匹配");
        serverErrCodeMap.append(0X4216, "安全卡不存在");
        serverErrCodeMap.append(0X4217, "认证状态已过期");
        serverErrCodeMap.append(0X4218, "应用信息不匹配");

        //需回调给三方应用的业务异常
        serverErrCodeMap.append(0X4219, "scope不合法，无法识别或其他错误");
        serverErrCodeMap.append(0X4220, "response_type不匹配");
    }

    public static String getErrorMsg(int errorCode) {
        Context context = ApplicationContext.getContext();
        String errorMsg = "未知错误（" + errorCode + "）";
        switch (errorCode) {
            case ErrorCode.RET_OPEN_SAFE_CHIP_FAIL:
                errorMsg = context.getString(R.string.uac_info_chip_init);
                break;
            case ErrorCode.RET_CHALLENGE_SIGN_FAIL:
                errorMsg = context.getString(R.string.uac_challenge_sign_fail);
                break;
            case ErrorCode.RET_GET_CERT_SN_FAIL:
                errorMsg = context.getString(R.string.uac_get_cert_sn_fail);
                break;
            case ErrorCode.RET_APP_ID_NULL:
                errorMsg = context.getString(R.string.uac_app_id_null);
                break;
            case ErrorCode.RET_NEED_VERIFY_PIN:
            case ErrorCode.RET_GET_PIN_NULL:
            case ErrorCode.RET_VERIFY_SAFE_PIN_ERR:
                errorMsg = context.getString(R.string.uac_pin_timeout);
                break;
            case ErrorCode.RET_TIMEOUT_REQUEST_SERVER:
            case ErrorCode.RET_VHSM_TIMEOUT_REQUEST_SERVER:
                errorMsg = context.getString(R.string.uac_net_timeout);
                break;
            case ErrorCode.RET_EXCEPTION_NETWORK:
            case ErrorCode.RET_VHSM_EXCEPTION_NETWORK:
                errorMsg = context.getString(R.string.uac_net_error);
                break;
            case IDP_ERROR:
                errorMsg = idpErrorMsg;
                break;
            default:
                if (isServerErrCode(errorCode)) {
                    //服务器返回的错误码
                    errorMsg = serverErrCodeMap.get(errorCode);
                }
        }

        return errorMsg;
    }

    public static boolean isServerErrCode(int errorCode) {
        return serverErrCodeMap.indexOfKey(errorCode) >= 0;
    }

    public static boolean isInvalidScope(int errorCode) {
        return errorCode == 0x4219;
    }

    public static boolean isUnsupportedResponseType(int errorCode) {
        return errorCode == 0x4220;
    }

    public static boolean isPasswordError(int errorCode) {
        return errorCode == 0X4214;
    }

    public static boolean isIdpError(int errorCode) {
        return errorCode == IDP_ERROR;
    }

    public static boolean isVHSMNetworkError(int errorCode) {
        return errorCode == ErrorCode.RET_VHSM_TIMEOUT_REQUEST_SERVER ||
                errorCode == ErrorCode.RET_VHSM_EXCEPTION_NETWORK;
    }

    public static String getPinErrorMsg() {
        Context context = ApplicationContext.getContext();
        String msg;
        if (pinErrorCode > 0) {
            if (pinErrorCode > 3) {
                msg = context.getString(R.string.uac_security_password_error);
            } else {
                msg = context.getString(R.string.uac_security_password_error_left, pinErrorCode + "");
            }
        } else if (pinErrorCode == -10 || pinErrorCode == -16) {
            //芯片锁死
            msg = context.getString(R.string.uac_pin_lock);

        } else if (pinErrorCode == -31 || pinErrorCode == -32) {
            msg = context.getString(R.string.uac_pin_lost);
        } else if (pinErrorCode == -132) {
            msg = context.getString(R.string.uac_net_error);
        } else if (pinErrorCode == -121) {
            msg = context.getString(R.string.uac_vhsm_net_timeout);
        } else {
            msg = context.getString(R.string.uac_verify_pin_error, pinErrorCode);
        }

        return msg;
    }

}
