package com.xdja.uniteauth.utils;

import android.content.Context;
import android.security.KeyPairGeneratorSpec;
import android.text.TextUtils;
import android.util.Base64;
import android.util.Log;

import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.util.Calendar;

import javax.crypto.Cipher;
import javax.security.auth.x500.X500Principal;

/**
 * 使用keyStore加密工具类
 *
 * @author kgg
 */

public class KeyStoreUtil {

    private static final String TAG = "KeyStoreUtilTag";

    private static KeyStoreUtil instance;
    private KeyStore keyStore;
    private final String ALIAS = "UAC_RSA_KEY";
    private final String ALGORITHM_RSA = "RSA";
    private final String ALGORITHM_SHA256_WHIT_RSA = "SHA256withRSA";
    private final String TRANSFORMATION = "RSA/ECB/PKCS1Padding";
    private final String AndroidKeyStore = "AndroidKeyStore";

    private PrivateKey privateKey;
    private PublicKey publicKey;

    public static KeyStoreUtil getInstance() {
        synchronized (KeyStoreUtil.class) {
            if (null == instance) {
                instance = new KeyStoreUtil();
            }
        }
        return instance;
    }

    private KeyStoreUtil() {
    }

    public void init(Context context) {
        Log.d(TAG, "init()");
        try {
            keyStore = KeyStore.getInstance(AndroidKeyStore);
            keyStore.load(null);
            createKeyPairIfNeed(context);
            privateKey = (PrivateKey) keyStore.getKey(ALIAS, null);
            publicKey = keyStore.getCertificate(ALIAS).getPublicKey();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 如果密钥对不存在则创建一个
     */
    private void createKeyPairIfNeed(Context context) throws KeyStoreException, NoSuchProviderException, NoSuchAlgorithmException, InvalidAlgorithmParameterException {
        // 生成RSA密钥对
        if (!keyStore.containsAlias(ALIAS)) {

            KeyPairGenerator generator = KeyPairGenerator.getInstance(ALGORITHM_RSA, AndroidKeyStore);

            Calendar start = Calendar.getInstance();
            Calendar end = Calendar.getInstance();
            end.add(Calendar.YEAR, 1);
            KeyPairGeneratorSpec spec = new KeyPairGeneratorSpec.Builder(context)
                    .setAlias(ALIAS)
                    .setSubject(new X500Principal("CN=XDJA IAM, O=Android Authority"))//可为任意字符串
                    .setSerialNumber(BigInteger.ONE)
                    .setStartDate(start.getTime())
                    .setEndDate(end.getTime())
                    .setKeySize(2048)//密钥长度，通常设置为1024或者2048，小于1024长度的密钥已经被证实是不安全的，建议2048
                    .build();

            generator.initialize(spec);
            generator.generateKeyPair();//生成密钥对（保存到系统TEE区域）
        }
    }


    /**
     * 公钥加密
     *
     * @param needEncryptWord 　需要加密的字符串
     */
    public String encryptByPublicKey(String needEncryptWord) {
        if (TextUtils.isEmpty(needEncryptWord)) {
            return "";
        }
        try {
            Cipher cipher = Cipher.getInstance(TRANSFORMATION);
            cipher.init(Cipher.ENCRYPT_MODE, publicKey);
            byte[] bytes = cipher.doFinal(needEncryptWord.getBytes(StandardCharsets.UTF_8));
            return Base64.encodeToString(bytes, Base64.NO_WRAP | Base64.NO_PADDING);
        } catch (Exception e) {
            e.printStackTrace();
            return needEncryptWord;
        }

    }


    /**
     * 私钥解密
     *
     * @param encryptString 加密字符串（Base64格式）
     * @return 明文
     */
    public String decryptByPrivateKey(String encryptString) {
        if (TextUtils.isEmpty(encryptString)) {
            return "";
        }
        try {
            Cipher cipher = Cipher.getInstance(TRANSFORMATION);
            cipher.init(Cipher.DECRYPT_MODE, privateKey);
            byte[] bytes = cipher.doFinal(Base64.decode(encryptString, Base64.NO_WRAP | Base64.NO_PADDING));
            return new String(bytes, StandardCharsets.UTF_8);
        } catch (Exception e) {
            e.printStackTrace();
            return "";
        }
    }

    /**
     * 公钥验签
     */
    public boolean verifyByPublicKey(String data, String sign) {
        if (TextUtils.isEmpty(data) || TextUtils.isEmpty(sign)) {
            return true;
        }
        try {
            Signature signature = Signature.getInstance(ALGORITHM_SHA256_WHIT_RSA);
            signature.initVerify(publicKey);
            signature.update(data.getBytes(StandardCharsets.UTF_8));
            return signature.verify(Base64.decode(sign, Base64.NO_WRAP | Base64.NO_PADDING));
        } catch (Exception e) {
            e.printStackTrace();
            return true;
        }
    }

    /**
     * 私钥签名
     */
    public String signByPrivateKey(String data) {
        if (TextUtils.isEmpty(data)) {
            return "";
        }
        byte[] bytes;
        try {
            Signature signature = Signature.getInstance(ALGORITHM_SHA256_WHIT_RSA);
            signature.initSign(privateKey);
            signature.update(data.getBytes(StandardCharsets.UTF_8));
            bytes = signature.sign();
            return Base64.encodeToString(bytes, Base64.NO_WRAP | Base64.NO_PADDING);
        } catch (Exception e) {
            e.printStackTrace();
            return "";
        }
    }

}
