package com.xdja.uniteauth.utils;

import android.content.Context;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;
import android.util.SparseIntArray;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.multi.unitepin.jar.MultiChipUnitePinManager;
import com.xdja.multichip.jniapi.Arithmetic;
import com.xdja.multichip.jniapi.JarJniApiProxy;
import com.xdja.multichip.jniapi.JarMultiChipStatusManager;
import com.xdja.multichip.jniapi.JarMultiJniApiManager;
import com.xdja.multichip.param.JniApiParam;
import com.xdja.uniteauth.R;
import com.xdja.uniteauth.UniteAuthConfig;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPublicKey;
import java.util.Arrays;
import java.util.List;

/**
 * Author:kongguoguang
 * Date:2019/10/21
 * Time:19:30
 * Summary:
 *
 * @author kgg
 */
public class SafeCardUtil {

    private static final String TAG = "SafeCardUtilTag";

    private static final SafeCardUtil INSTANCE = new SafeCardUtil();

    public static SafeCardUtil getInstance() {
        return INSTANCE;
    }

    private SafeCardUtil() {
    }

    private JarJniApiProxy jniApiProxy;
    private Context context;

    private static final int CERT_TYPE_RSA = 1;
    private static final int CERT_TYPE_SM2 = 2;

    /**
     * 优先级最高的代理对象
     */
    private JarJniApiProxy highestPriorityJarJniApiProxy;

    public void init(Context context) {
        this.context = context;
    }

    /**
     * 记录上次验证PIN码是否失败
     */
    private boolean lastVerifyPinFailed;

    public boolean isLastVerifyPinFailed() {
        return lastVerifyPinFailed;
    }

    /**
     * 卡类型名称
     */
    public static SparseIntArray cardTypeStrMap = new SparseIntArray(5);

    static {
        cardTypeStrMap.put(JniApiParam.TYPE_ONBOARD, R.string.uac_chip_name);
        cardTypeStrMap.put(JniApiParam.TYPE_TF, R.string.uac_tf_name);
        cardTypeStrMap.put(JniApiParam.TYPE_COVERED, R.string.uac_cover_card_name);
        cardTypeStrMap.put(JniApiParam.TYPE_BLUETOOTH, R.string.uac_blue_key_name);
        cardTypeStrMap.put(JniApiParam.TYPE_VHSM_NET, R.string.uac_net_vhsm);
    }

    /**
     * 卡类型的优先级，
     * 优先级 板载芯片 > TF卡 > 贴膜卡 > 蓝牙key
     */
    public static SparseIntArray priorityMap = new SparseIntArray(7);

    static {
        priorityMap.put(JniApiParam.TYPE_ONBOARD, 100);
        priorityMap.put(JniApiParam.TYPE_TF, 99);
        priorityMap.put(JniApiParam.TYPE_COVERED, 98);
        priorityMap.put(JniApiParam.TYPE_VHSM_NET, 97);
        priorityMap.put(JniApiParam.TYPE_VHSM, 96);
        priorityMap.put(JniApiParam.TYPE_BLUETOOTH, 95);
        priorityMap.put(0, 0);

    }

    /**
     * 根据安全卡Id和类型获取JniApiProxy代理对象，如果代理对象已存在
     */
    public JarJniApiProxy getJniApiProxy(@NonNull Context context, @NonNull String cardId, int cardType) {
        jniApiProxy = null;
        if (!TextUtils.isEmpty(cardId)) {

            if (highestPriorityJarJniApiProxy != null &&
                    cardId.equals(highestPriorityJarJniApiProxy.getCardId().second)) {
                jniApiProxy = highestPriorityJarJniApiProxy;
            } else {
                Pair<Integer, JarJniApiProxy> pair = JarMultiJniApiManager.getInstance().make(context, cardId);
                jniApiProxy = pair.second;
            }

        } else if (cardType > 0) {

            if (highestPriorityJarJniApiProxy != null &&
                    cardType == highestPriorityJarJniApiProxy.getCardType()) {
                jniApiProxy = highestPriorityJarJniApiProxy;
            } else {
                Pair<Integer, JarJniApiProxy> pair = JarMultiJniApiManager.getInstance().make(context, cardType);
                jniApiProxy = pair.second;
            }
        } else {
            jniApiProxy = getHighestPriorityJarJniApiProxy();
        }
        return jniApiProxy;
    }


    /**
     * 根据优先级得到JniApiProxy代理对象，如果代理对象已存在则直接使用
     */
    public JarJniApiProxy getJniApiProxy() {
        if (jniApiProxy != null) {
            return jniApiProxy;
        }
//        synchronized (SafeCardUtil.class) {
//            Pair<Integer, List<JniApiParam>> pair = JarMultiJniApiManager.getInstance().getAll(context);
//            if (pair.first == 0) {
//                String cardId = "";
//                int cardType = 0;
//                int priority = 0;
//                for (JniApiParam param : pair.second) {
//                    Log.d(TAG, "JniApiParam chipType = " + param.chipType);
//                    //默认是最低优先级；如果当前芯片类型没有配置优先级；
//                    int tmpPriority = priorityMap.get(param.chipType);
//                    if (tmpPriority > priority) {
//                        priority = tmpPriority;
//                        cardType = param.chipType;
//                        cardId = param.cardId;
//                    }
//                }
//                if (cardType == 0) {
//                    return null;
//                }
//                Pair<Integer, JarJniApiProxy> makePair = JarMultiJniApiManager.getInstance().make(context, cardId);
//                if (makePair == null || makePair.first != 0) {
//                    return null;
//                }
//                jniApiProxy = makePair.second;
//                register();
//            }
//        }
        return getHighestPriorityJarJniApiProxy();
    }

    private JarJniApiProxy getHighestPriorityJarJniApiProxy() {
        if (highestPriorityJarJniApiProxy != null) {
            return highestPriorityJarJniApiProxy;
        }
        synchronized (SafeCardUtil.class) {
            Pair<Integer, List<JniApiParam>> pair = JarMultiJniApiManager.getInstance().getAll(context);
            if (pair.first == 0) {
                String cardId = "";
                int cardType = 0;
                int priority = 0;
                for (JniApiParam param : pair.second) {
                    Log.d(TAG, "JniApiParam chipType = " + param.chipType);
                    //默认是最低优先级；如果当前芯片类型没有配置优先级；
                    int tmpPriority = priorityMap.get(param.chipType);
                    if (tmpPriority > priority) {
                        priority = tmpPriority;
                        cardType = param.chipType;
                        cardId = param.cardId;
                    }
                }
                if (cardType == 0) {
                    return null;
                }
                Pair<Integer, JarJniApiProxy> makePair = JarMultiJniApiManager.getInstance().make(context, cardId);
                if (makePair == null || makePair.first != 0) {
                    return null;
                }
                highestPriorityJarJniApiProxy = makePair.second;
                register();
            }
        }
        return highestPriorityJarJniApiProxy;
    }

    private JarMultiChipStatusManager.MultiChipStatusListener listener;

    private void register() {
        JarMultiChipStatusManager.getInstance().registerListener(context, getListener());
    }

    private JarMultiChipStatusManager.MultiChipStatusListener getListener() {
        if (listener == null) {
            listener = new JarMultiChipStatusManager.MultiChipStatusListener() {
                @Override
                public void chipAdd(String cardId, int cardType) {
                    Log.d(TAG, "MultiChipStatusListener, chipAdd(), cardType = " + cardType);
                    //jniApiProxy为空，或者新添加的芯片设备优先级高于当前使用设备，则根据新添加的芯片设备创建jniApiProxy
                    if (highestPriorityJarJniApiProxy == null ||
                            priorityMap.get(cardType) > priorityMap.get(highestPriorityJarJniApiProxy.getCardType())) {
                        highestPriorityJarJniApiProxy = JarMultiJniApiManager.getInstance().make(context, cardType).second;
                    }
                }

                @Override
                public void chipRemove(int cardType) {
                    Log.d(TAG, "MultiChipStatusListener, chipRemove(), cardType = " + cardType);
                    if (highestPriorityJarJniApiProxy != null) {
                        if (cardType == highestPriorityJarJniApiProxy.getCardType()) {
                            highestPriorityJarJniApiProxy = null;
                        }
                    }
                }

                @Override
                public void serviceDeath() {
                    Log.d(TAG, "MultiChipStatusListener, serviceDeath()");
                    highestPriorityJarJniApiProxy = null;
                    jniApiProxy = null;
                }

                @Override
                public void serviceStart() {
                    Log.d(TAG, "MultiChipStatusListener, serviceStart()");
                }
            };
        }
        return listener;
    }

    /**
     * 获取使用的卡号
     */
    private String getCardId() {
        JarJniApiProxy jniApiProxy = getJniApiProxy();
        if (jniApiProxy == null) {
            return "";
        }

        Pair<Integer, String> cardIdPair = jniApiProxy.getCardId();
        if (cardIdPair == null || cardIdPair.first != 0) {
            return "";
        }

        return cardIdPair.second;
    }

    /**
     * 获取PIN码
     */
    public String getPin() {
        JarJniApiProxy jarJniApiProxy = getJniApiProxy();
        if (jarJniApiProxy == null) {
            return "";
        }

        String cardId = jarJniApiProxy.getCardId().second;
        if (TextUtils.isEmpty(cardId)) {
            return "";
        }

        int role = UniteAuthConfig.getInstance(context).getRole();
        Pair<Integer, String> pinPair = MultiChipUnitePinManager.getInstance().getPin(context, cardId, role);
        if (pinPair == null || pinPair.first != 0) {
            return "";
        }
        return pinPair.second;
    }

    /**
     * 验证PIN码
     */
    public int verifyPin(String pin) {
        JarJniApiProxy jarJniApiProxy = getJniApiProxy();
        if (jarJniApiProxy == null) {
            return JNIAPI.XKR_NO_KEY;
        }
        int role = UniteAuthConfig.getInstance(context).getRole();
        int ret = jarJniApiProxy.VerifyPIN(role, pin.getBytes(), pin.length());
        lastVerifyPinFailed = ret != 0;
        return ret;
    }

    public int signByContainNo(JarJniApiProxy jarJniApiProxy, byte[] dataIn, int inLen,
                               byte[] dataOut, int[] outLen,
                               int certType, int containNo) {
        //首先判断证书类型
        //调用相应接口进行运算
        int certAlgTypeInContain = parseCertAlgType(jarJniApiProxy, containNo, certType);
        if (certAlgTypeInContain == CERT_TYPE_RSA) {
            return RSASignByContain(jarJniApiProxy, dataIn, inLen, dataOut, outLen, certType, containNo);
        } else if (certAlgTypeInContain == CERT_TYPE_SM2) {
            return sm2Sign(jarJniApiProxy, dataIn, inLen, dataOut, outLen, certType, containNo);
        }

        return -1;
    }

    //根据参数进行rsa签名
    private int RSASignByContain(JarJniApiProxy jarJniApiProxy, byte[] dataIn, int inLen,
                                 byte[] dataOut, int[] outLen,
                                 int certType, int containNo) {
        //首先获取证书模长
        int pubkeyBits = getRSACertPubkeyBits(jarJniApiProxy, containNo, certType);
        if (pubkeyBits == -1) {
            Log.w(TAG, "RSA getRSACertPubkeyBits ret: " + pubkeyBits);
            return -1;
        }

        int ret = -1;
        //对数据进行sha1运算

        //调用接口运算
        byte[] certId;
        if (certType == 1) {
            certId = Arithmetic.getExchangeCertFid(containNo);
        } else {// 如果是签名证书
            certId = Arithmetic.getSignCertFid(containNo);
        }
        byte[] prifid = new byte[2];
        prifid[0] = 0x00;
        prifid[1] = (byte) (certId[1] + 2);
        ret = jarJniApiProxy.RSASign(pubkeyBits, prifid, JNIAPI.SIGN_NOHASH, dataIn, inLen, dataOut, outLen);

        return ret;
    }

    public int getRSACertPubkeyBits(JarJniApiProxy jarJniApiProxy, int containNo, int certType) {

        byte[] certBuf = new byte[2 * 1024];
        int[] certLen = new int[1];

        byte[] fid = convertToCardFid(containNo, certType);
        int ret = jarJniApiProxy.ReadCert(fid, certBuf, certLen);
        if (ret != 0) {
            Log.w("Auth", "RS ReadCert ret: " + ret);
            return -1;
        }
        byte[] result = Arrays.copyOfRange(certBuf, 0, certLen[0]);

        return getRsaPubkeyBits(result, certLen[0]);
    }

    public int getRsaPubkeyBits(byte[] cert, int certlen) {

        ByteArrayInputStream in = new ByteArrayInputStream(cert, 0, certlen);

        X509Certificate x509;
        try {
            x509 = (X509Certificate) CertificateFactory.getInstance("X.509").generateCertificate(in);
            in.close();
        } catch (Exception e) {
            return 1024;
        }
        RSAPublicKey key = (RSAPublicKey) x509.getPublicKey();
        BigInteger inte = key.getModulus();
        // key = x509.getPublicKey();
        return inte.bitLength();
    }

    private int sm2Sign(JarJniApiProxy jarJniApiProxy, byte[] dataIn, int inLen,
                        byte[] dataOut, int[] outLen,
                        int certType, int containNo) {

        byte[] pubfid = new byte[2];
        byte[] prifid = new byte[2];

//        int SIGN_NOHASH = 1;
        int dataType = 1;

        //如果是交换证书
        byte[] certId;
        if (certType == 1) {
            certId = Arithmetic.getExchangeCertFid(containNo);
        } else {// 如果是签名证书
            certId = Arithmetic.getSignCertFid(containNo);
        }
        pubfid[0] = 0x00;
        pubfid[1] = (byte) (certId[1] + 1);

        prifid[0] = 0x00;
        prifid[1] = (byte) (certId[1] + 2);

        int ret = jarJniApiProxy.SM2Sign(pubfid, prifid, dataType, dataIn, inLen, dataOut, outLen);
        Log.w("Auth", "SM2SignByDefault ret: " + ret);

        return ret;
    }

    /**
     * 将 容器号 和 类型 转换成 fid
     *
     * @param containNo 容器号
     * @param certType  类型(容器名称、交换证书、交换公钥、交换私钥、签名证书、签名公钥、签名私钥)
     * @return
     */
    public byte[] convertToFid(int containNo, int certType) {
        byte[] certId = new byte[2];

        certId[0] = 0x00;
        certId[1] = (byte) (0x27 + certType + containNo * 7);

        return certId;
    }

    public byte[] convertToCardFid(int containNo, int certType) {
        byte[] fId = new byte[2];
        fId[0] = 0x00;

        byte[] certId;
        if (certType == 1) {
            certId = Arithmetic.getExchangeCertFid(containNo);
        } else {// 如果是签名证书
            certId = Arithmetic.getSignCertFid(containNo);
        }

        fId[1] = certId[1];

        return fId;
    }


    //证书算法类型
    public int parseCertAlgType(JarJniApiProxy jarJniApiProxy, int containNo, int certType) {

        //首先从指定容器中读取证书内容
        byte[] certBuf = new byte[2 * 1024];
        int[] certLen = new int[1];

        byte[] fid = convertToCardFid(containNo, certType);

        int ret = jarJniApiProxy.ReadCert(fid, certBuf, certLen);
        if (ret != 0) {
            Log.w("Auth", "RS ReadCert ret: " + ret);
            return -1;
        }
        byte[] result = Arrays.copyOfRange(certBuf, 0, certLen[0]);

        //开始解析证书信息
        ByteArrayInputStream in = new ByteArrayInputStream(result, 0, certLen[0]);

        X509Certificate x509 = null;
        try {
            x509 = (X509Certificate) CertificateFactory.getInstance("X.509", "BC").generateCertificate(in);
            in.close();
        } catch (Exception e) {
            e.printStackTrace();
            try {
                in.close();
            } catch (IOException e1) {
                e1.printStackTrace();
            }
        }

        if (x509 != null) {
            String alg = x509.getSigAlgName();
            Log.d("Auth", "Cert alg name ==== " + alg);
            if (!TextUtils.isEmpty(alg)) {
                if (alg.contains("RSA")) {
                    return CERT_TYPE_RSA;
                } else if (alg.contains("SM2")) {
                    return CERT_TYPE_SM2;
                }
            }
        }
        return CERT_TYPE_SM2;
    }

    /**
     * 获得SN
     */
    public String getSn() {
        return getSn(getJniApiProxy());
    }

    public String getSn(JarJniApiProxy jarJniApiProxy) {
        int containNo = UniteAuthConfig.getInstance(context).getContain();
        int certType = UniteAuthConfig.getInstance(context).getCertType();
        if (jarJniApiProxy == null) {
            return "";
        }
        Pair<Integer, String> snPair = jarJniApiProxy.getSn(containNo, certType);
        if (snPair.first != 0) {
            Log.d(TAG, "get sn failed： " + snPair.first);
            return "";
        }

        //modify 2019年1月10日15:21:42 weizg
        //sn开头有零的话，则删除；
        String snStr = snPair.second;
        while (snStr.startsWith("0")) {
            snStr = snStr.substring(snStr.indexOf("0") + 1);
        }

        return snStr;
    }


    /**
     * 获得SN
     *
     * @return
     */
    public Pair<Integer, String> getSnPair() {
        String sn = getSn();
        if (TextUtils.isEmpty(sn)) {
            return Pair.create(-1, "");
        } else {
            return Pair.create(0, sn);
        }
    }
}
