package com.xdja.uniteauth.utils;

import android.text.TextUtils;
import android.util.Log;

import com.xdja.uniteauth.data.local.UacPreference;

/**
 * @author kgg
 * Date:2019/8/15
 * Time:15:49
 * Summary:设置信息工具类
 */
public class SettingInfoUtil {

    private final String TAG = "SettingInfoUtilTag";

    private static final SettingInfoUtil INSTANCE = new SettingInfoUtil();

    public static SettingInfoUtil getInstance() {
        return INSTANCE;
    }

    private SettingInfoUtil() {
    }

    private String fingerprintId;

    private boolean fingerprintLoginEnabled;

    private long fingerprintLockedTimestamp;

    private String gesturePwd;

    private boolean gestureLoginEnabled;

    private long gestureLockedTimestamp;

    private long privacyProtectTimestamp;

    private int privacyProtectInterval;

    private boolean recommendFingerprintAndGesture = true;

    public void initSettingInfo() {
        if (AuthModeUtil.getInstance().isOpenPlatformApp()) {
            fingerprintId = UacPreference.getInstance().getCertFingerprintId();
            fingerprintLoginEnabled = UacPreference.getInstance().isCertFingerprintAuthAvailable();

            gesturePwd = UacPreference.getInstance().getCertGesturePwd();
            gestureLoginEnabled = UacPreference.getInstance().isCertGestureAuthAvailable();
            gestureLockedTimestamp = UacPreference.getInstance().getCertGestureLockedTimestamp();

            privacyProtectTimestamp = UacPreference.getInstance().getCertPrivacyProtectTimestamp();
            privacyProtectInterval = UacPreference.getInstance().getCertPrivacyProtectInterval();

            recommendFingerprintAndGesture = UacPreference.getInstance().isCertRecommendFingerprintAndGesture();
        } else {
            fingerprintId = UacPreference.getInstance().getUserFingerprintId();
            fingerprintLoginEnabled = UacPreference.getInstance().isUserFingerprintAuthAvailable();

            gesturePwd = UacPreference.getInstance().getUserGesturePwd();
            gestureLoginEnabled = UacPreference.getInstance().isUserGestureAuthAvailable();
            gestureLockedTimestamp = UacPreference.getInstance().getUserGestureLockedTimestamp();

            privacyProtectTimestamp = UacPreference.getInstance().getUserPrivacyProtectTimestamp();
            privacyProtectInterval = UacPreference.getInstance().getUserPrivacyProtectInterval();

            recommendFingerprintAndGesture = UacPreference.getInstance().isUserRecommendFingerprintAndGesture();
        }

        fingerprintLockedTimestamp = UacPreference.getInstance().getFingerprintLockedTimestamp();
    }

    public void resetSettingInfo() {
        fingerprintId = "";
        fingerprintLoginEnabled = false;
        fingerprintLockedTimestamp = 0L;
        gesturePwd = "";
        gestureLoginEnabled = false;
        gestureLockedTimestamp = 0L;
        privacyProtectTimestamp = 0L;
        privacyProtectInterval = 0;
        recommendFingerprintAndGesture = true;
    }


    public String getFingerprintId() {
        return fingerprintId;
    }

    public void setFingerprintId(String fid) {
        if (TextUtils.isEmpty(fingerprintId)) {
            fingerprintId = fid;
        } else if (!fingerprintId.contains(fid)) {
            fingerprintId = fingerprintId + "," + fid;
        }
        fingerprintLoginEnabled = true;
        if (AuthModeUtil.getInstance().isOpenPlatformApp()) {
            UacPreference.getInstance().setCertFingerprintId(fingerprintId);
            UacPreference.getInstance().setCertFingerprintAuthAvailable(true);
        } else {
            UacPreference.getInstance().setUserFingerprintId(fingerprintId);
            UacPreference.getInstance().setUserFingerprintAuthAvailable(true);
        }
    }

    public void disableFingerprintLogin() {
        fingerprintId = "";
        fingerprintLoginEnabled = false;
        if (AuthModeUtil.getInstance().isOpenPlatformApp()) {
            UacPreference.getInstance().clearCertFingerprintId();
            UacPreference.getInstance().setCertFingerprintAuthAvailable(false);
        } else {
            UacPreference.getInstance().clearUserFingerprintId();
            UacPreference.getInstance().setUserFingerprintAuthAvailable(false);
        }
    }

    public boolean isFingerprintLoginEnabled() {
        return fingerprintLoginEnabled;
    }

    public void setGesturePwd(String pwd) {
        gesturePwd = pwd;
        gestureLoginEnabled = true;
        if (AuthModeUtil.getInstance().isOpenPlatformApp()) {
            UacPreference.getInstance().setCertGesturePwd(pwd);
            UacPreference.getInstance().setCertGestureAuthAvailable(true);
        } else {
            UacPreference.getInstance().setUserGesturePwd(pwd);
            UacPreference.getInstance().setUserGestureAuthAvailable(true);
        }
    }

    public void disableGestureLogin() {
        gesturePwd = "";
        gestureLoginEnabled = false;
        if (AuthModeUtil.getInstance().isOpenPlatformApp()) {
            UacPreference.getInstance().setCertGesturePwd("");
            UacPreference.getInstance().setCertGestureAuthAvailable(false);
        } else {
            UacPreference.getInstance().setUserGesturePwd("");
            UacPreference.getInstance().setUserGestureAuthAvailable(false);
        }
    }

    public boolean isGestureLoginEnabled() {
        return gestureLoginEnabled;
    }

    /**
     * 校验手势密码是否正确
     */
    public boolean checkGesturePwd(String gesturePwd) {
        return this.gesturePwd.equals(gesturePwd);
    }

    /**
     * 隐私保护是否超时
     */
    public boolean isPrivacyProtectTimeout() {
        Log.d(TAG, "isPrivacyProtectTimeout()");
        long timeDiff = System.currentTimeMillis() - privacyProtectTimestamp;
        return timeDiff >= privacyProtectInterval * 60 * 1000;
    }

    public void updatePrivacyProtectTimestamp() {
        privacyProtectTimestamp = System.currentTimeMillis();
        if (AuthModeUtil.getInstance().isOpenPlatformApp()) {
            UacPreference.getInstance().setCertPrivacyProtectTimestamp(privacyProtectTimestamp);
        } else {
            UacPreference.getInstance().setUserPrivacyProtectTimestamp(privacyProtectTimestamp);
        }

    }

    public int getPrivacyProtectInterval() {
        return privacyProtectInterval;
    }

    public String getPrivacyProtectIntervalString() {

        String msg;
        if (privacyProtectInterval == 0) {
            msg = "每次";
        } else {
            msg = privacyProtectInterval + "分钟";
        }

        return msg;
    }

    public void setPrivacyProtectInterval(int interval) {
        this.privacyProtectInterval = interval;
        if (AuthModeUtil.getInstance().isOpenPlatformApp()) {
            UacPreference.getInstance().setCertPrivacyProtectInterval(interval);
        } else {
            UacPreference.getInstance().setUserPrivacyProtectInterval(interval);
        }
    }

    public long getFingerprintLockedTimestamp() {
        return fingerprintLockedTimestamp;
    }

    public void setFingerprintLockedTimestamp(long timestamp) {
        this.fingerprintLockedTimestamp = timestamp;
        UacPreference.getInstance().setFingerprintLockedTimestamp(timestamp);
    }

    public long getGestureLockedTimestamp() {
        return gestureLockedTimestamp;

    }

    public void setGestureLockedTimestamp(long timestamp) {
        this.gestureLockedTimestamp = timestamp;
        if (AuthModeUtil.getInstance().isOpenPlatformApp()) {
            UacPreference.getInstance().setCertGestureLockedTimestamp(timestamp);
        } else {
            UacPreference.getInstance().setUserGestureLockedTimestamp(timestamp);
        }
    }

    public boolean isRecommendFingerprintAndGesture() {
        return recommendFingerprintAndGesture;
    }

    public void setRecommendFingerprintAndGesture(boolean recommendFingerprintAndGesture) {
        this.recommendFingerprintAndGesture = recommendFingerprintAndGesture;
        if (AuthModeUtil.getInstance().isOpenPlatformApp()) {
            UacPreference.getInstance().setCertRecommendFingerprintAndGesture(recommendFingerprintAndGesture);
        } else {
            UacPreference.getInstance().setUserRecommendFingerprintAndGesture(recommendFingerprintAndGesture);
        }
    }
}
