package com.xdja.uniteauth.utils;


import android.content.SharedPreferences;
import android.text.TextUtils;

import com.xdja.uniteauth.ThreadPoolManager;

import java.util.Comparator;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

/**
 * 通过KeyStoreUtil加密存取数据的SharedPreferences工具类
 * 每次存入数据后计算一下签名
 *
 * @author kgg
 */
public class UacPreferencesUtil {

    private static final String KEY_SIGN = "key_sign";

    /**
     * 保存数据到SharedPreferences，因加密比较耗时，放到子线程中执行
     */
    public static void setValue(final SharedPreferences sharedPreferences, final String key, final Object value) {

        ThreadPoolManager.getInstance().execute(new Runnable() {
            @Override
            public void run() {

                if (value instanceof String) {
                    setStringValue(sharedPreferences, key, (String) value);
                } else if (value instanceof Boolean) {
                    setBooleanValue(sharedPreferences, key, (Boolean) value);
                } else if (value instanceof Float) {
                    setFloatValue(sharedPreferences, key, (Float) value);
                } else if (value instanceof Integer) {
                    setIntValue(sharedPreferences, key, (Integer) value);
                } else if (value instanceof Long) {
                    setLongValue(sharedPreferences, key, (Long) value);
                } else {
                    return;
                }

                updateSign(sharedPreferences);
            }
        });


    }

    /**
     * Set a preference string value
     *
     * @param key   the preference key to set
     * @param value the value for this key
     */
    private static void setStringValue(SharedPreferences sharedPreferences, String key, String value) {
        if (sharedPreferences == null) {
            return;
        }
        String encryptString = KeyStoreUtil.getInstance().encryptByPublicKey(value);
        sharedPreferences.edit().putString(key, encryptString).apply();
    }

    /**
     * Set a preference boolean value
     *
     * @param key   the preference key to set
     * @param value the value for this key
     */
    private static void setBooleanValue(SharedPreferences sharedPreferences, String key, boolean value) {
        if (sharedPreferences == null) {
            return;
        }
        String encryptString = KeyStoreUtil.getInstance().encryptByPublicKey(String.valueOf(value));
        sharedPreferences.edit().putString(key, encryptString).apply();
    }

    /**
     * Set a preference float value
     *
     * @param key   the preference key to set
     * @param value the value for this key
     */
    private static void setFloatValue(SharedPreferences sharedPreferences, String key, float value) {
        if (sharedPreferences == null) {
            return;
        }
        String encryptString = KeyStoreUtil.getInstance().encryptByPublicKey(String.valueOf(value));
        sharedPreferences.edit().putString(key, encryptString).apply();
    }

    /**
     * Set a preference int value
     *
     * @param key   the preference key to set
     * @param value the value for this key
     */
    private static void setIntValue(SharedPreferences sharedPreferences, String key, int value) {
        if (sharedPreferences == null) {
            return;
        }
        String encryptString = KeyStoreUtil.getInstance().encryptByPublicKey(String.valueOf(value));
        sharedPreferences.edit().putString(key, encryptString).apply();
    }

    /**
     * Set a preference long value
     *
     * @param key   the preference key to set
     * @param value the value for this key
     */
    private static void setLongValue(SharedPreferences sharedPreferences, String key, long value) {
        if (sharedPreferences == null) {
            return;
        }
        String encryptString = KeyStoreUtil.getInstance().encryptByPublicKey(String.valueOf(value));
        sharedPreferences.edit().putString(key, encryptString).apply();
    }

    //Private static getters
    // For string
    public static String getStringValue(SharedPreferences sharedPreferences, String key) {
        if (sharedPreferences == null) {
            return "";
        }
        return KeyStoreUtil.getInstance().decryptByPrivateKey(sharedPreferences.getString(key, ""));
    }

    // For boolean
    public static boolean getBooleanValue(SharedPreferences sharedPreferences, String key, boolean defaultValue) {
        if (sharedPreferences == null) {
            return defaultValue;
        }
        String encryptData = sharedPreferences.getString(key, "");
        String decryptData = KeyStoreUtil.getInstance().decryptByPrivateKey(encryptData);
        if (TextUtils.isEmpty(decryptData)) {
            return defaultValue;
        }
        return Boolean.parseBoolean(decryptData);
    }

    // For int
    public static int getIntValue(SharedPreferences sharedPreferences, String key) {
        if (sharedPreferences == null) {
            return 0;
        }
        String encryptData = sharedPreferences.getString(key, "");
        String decryptData = KeyStoreUtil.getInstance().decryptByPrivateKey(encryptData);
        if (TextUtils.isEmpty(decryptData)) {
            return 0;
        }
        try {
            return Integer.parseInt(decryptData);
        } catch (Exception e) {
            return 0;
        }

    }

    // For int
    public static int getIntValue(SharedPreferences sharedPreferences, String key, int defaultValue) {
        if (sharedPreferences == null) {
            return defaultValue;
        }
        String encryptData = sharedPreferences.getString(key, "");
        String decryptData = KeyStoreUtil.getInstance().decryptByPrivateKey(encryptData);
        if (TextUtils.isEmpty(decryptData)) {
            return defaultValue;
        }
        try {
            return Integer.parseInt(decryptData);
        } catch (Exception e) {
            return defaultValue;
        }

    }

    // For long
    public static long getLongValue(SharedPreferences sharedPreferences, String key) {
        if (sharedPreferences == null) {
            return 0L;
        }
        String encryptData = sharedPreferences.getString(key, "");
        String decryptData = KeyStoreUtil.getInstance().decryptByPrivateKey(encryptData);
        if (TextUtils.isEmpty(decryptData)) {
            return 0L;
        }
        try {
            return Long.parseLong(decryptData);
        } catch (Exception e) {
            return 0L;
        }
    }

    // For float
    public static float getFloatValue(SharedPreferences sharedPreferences, String key) {
        if (sharedPreferences == null) {
            return 0F;
        }
        String encryptData = sharedPreferences.getString(key, "");
        String decryptData = KeyStoreUtil.getInstance().decryptByPrivateKey(encryptData);
        if (TextUtils.isEmpty(decryptData)) {
            return 0F;
        }
        try {
            return Float.parseFloat(decryptData);
        } catch (Exception e) {
            return 0F;
        }
    }

    public static void removeValue(SharedPreferences sharedPreferences, String key) {
        if (sharedPreferences == null) {
            return;
        }
        sharedPreferences.edit().remove(key).apply();
    }

    public static void clear(SharedPreferences sharedPreferences) {
        if (sharedPreferences == null) {
            return;
        }
        sharedPreferences.edit().clear().apply();
    }

    /**
     * 验证签名值，防止数据被篡改
     */
    public static boolean verifySign(SharedPreferences sharedPreferences) {
        if (sharedPreferences == null) {
            return true;
        }
        String sign = sharedPreferences.getString(KEY_SIGN, "");
        return KeyStoreUtil.getInstance().verifyByPublicKey(getAllData(sharedPreferences), sign);
    }

    /**
     * 对SP中的全部数据进行签名，并将签名值保存到SP中
     *
     * @param sharedPreferences SP
     */
    private static void updateSign(SharedPreferences sharedPreferences) {
        if (sharedPreferences == null) {
            return;
        }
        String sign = KeyStoreUtil.getInstance().signByPrivateKey(getAllData(sharedPreferences));
        sharedPreferences.edit().putString(KEY_SIGN, sign).apply();
    }

    /**
     * 获取SP中保存的所有数据（签名值除外），拼接成一个字符串并用逗号分隔
     *
     * @param sharedPreferences Sp
     * @return String
     */
    private static String getAllData(SharedPreferences sharedPreferences) {

        if (sharedPreferences == null) {
            return "";
        }

        Map<String, ?> map = sharedPreferences.getAll();
        //降序排列
        Set<String> sortKeySet = new TreeSet<>(new Comparator<String>() {
            @Override
            public int compare(String o1, String o2) {
                return o2.compareTo(o1);
            }
        });
        sortKeySet.addAll(map.keySet());
        //签名值不计算在内
        sortKeySet.remove(KEY_SIGN);

        StringBuilder builder = new StringBuilder();
        for (String key : sortKeySet) {
            builder.append(map.get(key)).append(",");
        }

        return builder.toString();
    }

}
