package com.xdja.uniteauth.view;

import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.res.Resources;
import android.hardware.fingerprint.FingerprintManager;
import android.os.CountDownTimer;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.annotation.StringRes;
import android.support.v4.content.ContextCompat;
import android.support.v4.hardware.fingerprint.FingerprintManagerCompat;
import android.support.v4.os.CancellationSignal;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.TextView;

import com.tencent.soter.core.model.ConstantsSoter;
import com.tencent.soter.wrapper.SoterWrapperApi;
import com.tencent.soter.wrapper.wrap_biometric.SoterBiometricCanceller;
import com.tencent.soter.wrapper.wrap_biometric.SoterBiometricStateCallback;
import com.tencent.soter.wrapper.wrap_callback.SoterProcessAuthenticationResult;
import com.tencent.soter.wrapper.wrap_callback.SoterProcessCallback;
import com.tencent.soter.wrapper.wrap_core.SoterProcessErrCode;
import com.tencent.soter.wrapper.wrap_task.AuthenticationParam;
import com.xdja.uniteauth.ApplicationContext;
import com.xdja.uniteauth.R;
import com.xdja.uniteauth.utils.SettingInfoUtil;
import com.xdja.uniteauth.utils.UacConstant;

/**
 * Author:kongguoguang
 * Date:2018-10-17
 * Time:9:58
 * Summary:
 * @author kgg
 */
public class UacFingerprintDialog extends Dialog {

    public static final int FINGERPRINT_TYPE_SOTER = 1;
    public static final int FINGERPRINT_TYPE_SYSTEM = 0;

    private String TAG = "UacFingerprintDialogTag";

    // 默认宽度
    private static int default_width = 260;

    // 默认样式
    private static int default_style = R.style.UacDialog;

    private TextView title, summary;

    private View titleLine;

    private SoterBiometricCanceller soterBiometricCanceller = null;

    private AuthenticationParam authenticationParam;

    private SoterProcessCallback<SoterProcessAuthenticationResult> soterProcessCallback;

    private SuccessCallback successCallback;

    private CountDownTimer countDownTimer;

    private final long FINGERPRINT_SENSOR_LOCK_TIME = 30 * 1000;

    private int fingerprintType;

    private FingerprintManagerCompat fingerprintManagerCompat;

    private CancellationSignal cancellationSignal;

    private FingerprintManagerCompat.AuthenticationCallback authenticationCallback;

    public UacFingerprintDialog(@NonNull Context context, int fingerprintType) {
        this(context, default_style, fingerprintType);
    }

    private UacFingerprintDialog(@NonNull Context context, int themeResId, int fingerprintType) {
        super(context, themeResId);
        this.fingerprintType = fingerprintType;
        View dialogContentView = LayoutInflater.from(context).inflate(R.layout.uac_fingerfrint_dialog, null);
        setContentView(dialogContentView);
        setCanceledOnTouchOutside(false);
        setOnDismissListener(new OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                Log.d(TAG, "onDismiss()");
                stopVerifyFingerprint();
            }
        });
        setOnCancelListener(new OnCancelListener() {
            @Override
            public void onCancel(DialogInterface dialog) {
                Log.d(TAG, "onCancel()");
                stopVerifyFingerprint();
            }
        });
        Window window = getWindow();
        if (window != null) {
            WindowManager.LayoutParams params = window.getAttributes();
            Resources resources = context.getResources();
            DisplayMetrics dm = resources.getDisplayMetrics();
            params.width = (int) (default_width * dm.density + 0.5f);
            window.setAttributes(params);
        }

        title = (TextView) dialogContentView.findViewById(R.id.uac_dialog_title);
        titleLine = findViewById(R.id.title_line);

        summary = (TextView) dialogContentView.findViewById(R.id.uac_dialog_summary);
        TextView confirm = (TextView) dialogContentView.findViewById(R.id.uac_dialog_confirm);
        confirm.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dismiss();
            }
        });

        init();

    }

    private void init() {
        if (fingerprintType == FINGERPRINT_TYPE_SOTER) {
            initSoterParam();
        } else {
            initFingerprint();
        }
    }

    private void initSoterParam() {
        soterBiometricCanceller = new SoterBiometricCanceller();
        authenticationParam = new AuthenticationParam.AuthenticationParamBuilder() // 通过Builder来构建认证请求
                .setScene(UacConstant.TENCENT_SOTER_SCENE) // 指定需要认证的场景。必须在init中初始化。必填
                .setBiometricType(ConstantsSoter.FINGERPRINT_AUTH)
                .setContext(getContext()) // 指定当前上下文。必填。
                .setSoterBiometricCanceller(soterBiometricCanceller)
                .setPrefilledChallenge("uac challenge")// 直接设置此字段，忽略获取挑战因子网络封装结构体的设置
                .setSoterBiometricStateCallback(new SoterBiometricStateCallback() {
                    @Override
                    public void onStartAuthentication() {

                    }

                    @Override
                    public void onAuthenticationHelp(int helpCode, CharSequence helpString) {
                        Log.d(TAG, "onAuthenticationHelp(), helpMsgId = " + helpCode + ", helpString = " + helpString);
                        if (TextUtils.isEmpty(helpString)) {
                            setSummary(R.string.uac_please_verify_fingerprint, false);
                        } else {
                            setSummary(helpString, false);
                        }

                    }

                    @Override
                    public void onAuthenticationSucceed() {
                        Log.d(TAG, "onAuthenticationSucceed()");
                    }

                    @Override
                    public void onAuthenticationFailed() {
                        Log.d(TAG, "onAuthenticationFailed()");
                        setSummary(R.string.uac_verify_fingerprint_fail, true);
                    }

                    @Override
                    public void onAuthenticationCancelled() {
                        Log.d(TAG, "onAuthenticationCancelled()");
                    }

                    @Override
                    public void onAuthenticationError(int errorCode, CharSequence errorString) {
                        Log.d(TAG, "onAuthenticationError(), errorCode = " + errorCode + ", errString = " + errorString);

                    }
                })
                .build();

        soterProcessCallback = new SoterProcessCallback<SoterProcessAuthenticationResult>() {
            @Override
            public void onResult(@NonNull SoterProcessAuthenticationResult result) {
                Log.d(TAG, "verify fingerprint result = " + result.getErrCode());
                if (result.isSuccess()) {
                    Log.d(TAG, "verify fingerprint success");
                    SettingInfoUtil.getInstance().setFingerprintLockedTimestamp(0L);
                    dismiss();
                    if (successCallback != null) {
                        successCallback.onSuccess(result.getExtData().getFid());
                    }
                } else {
                    Log.d(TAG, "verify fingerprint failed");
                    if (result.errCode == SoterProcessErrCode.ERR_FINGERPRINT_LOCKED) {
                        setSummary(getContext().getString(R.string.uac_verify_fingerprint_fail_too_much, 30), false);
                        SettingInfoUtil.getInstance().setFingerprintLockedTimestamp(System.currentTimeMillis());
                        startCountDown(FINGERPRINT_SENSOR_LOCK_TIME);
                    } else if (result.errCode == SoterProcessErrCode.ERR_INIT_SIGN_FAILED) {
                        setSummary("初始化指纹功能，请稍等", false);
                        stopVerifyFingerprint();
                        ApplicationContext.initSoterWrapperApi();
                        new Handler().postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                setSummary(R.string.uac_please_verify_fingerprint, false);
                                startVerifyFingerprint();
                            }
                        }, 4000);

                    } else if (result.errCode == SoterProcessErrCode.ERR_USER_CANCELLED) {
                        dismiss();
                    } else {
                        setSummary(getContext().getString(R.string.uac_verify_fingerprint_error_code, result.getErrCode()), false);
                    }
                }
            }
        };
    }

    private void initFingerprint() {
        fingerprintManagerCompat = FingerprintManagerCompat.from(getContext());
        cancellationSignal = new CancellationSignal();
        authenticationCallback = new FingerprintManagerCompat.AuthenticationCallback() {
            @Override
            public void onAuthenticationError(int errMsgId, CharSequence errString) {
                super.onAuthenticationError(errMsgId, errString);
                if (errMsgId == FingerprintManager.FINGERPRINT_ERROR_LOCKOUT) {
                    setSummary(getContext().getString(R.string.uac_verify_fingerprint_fail_too_much, 30), false);
                    SettingInfoUtil.getInstance().setFingerprintLockedTimestamp(System.currentTimeMillis());
                    startCountDown(FINGERPRINT_SENSOR_LOCK_TIME);
                }

            }

            @Override
            public void onAuthenticationHelp(int helpMsgId, CharSequence helpString) {
                Log.d(TAG, "onAuthenticationHelp(), helpMsgId = " + helpMsgId + ", helpString = " + helpString);
                if (TextUtils.isEmpty(helpString)) {
                    setSummary(R.string.uac_please_verify_fingerprint, false);
                } else {
                    setSummary(helpString, false);
                }
            }

            @Override
            public void onAuthenticationSucceeded(FingerprintManagerCompat.AuthenticationResult result) {
                super.onAuthenticationSucceeded(result);
                Log.d(TAG, "verify fingerprint success");
                SettingInfoUtil.getInstance().setFingerprintLockedTimestamp(0L);
                dismiss();
                if (successCallback != null) {
                    successCallback.onSuccess("");
                }
            }

            @Override
            public void onAuthenticationFailed() {
                Log.d(TAG, "onAuthenticationFailed()");
                setSummary(R.string.uac_verify_fingerprint_fail, true);
            }
        };
    }

    private void setSummary(@StringRes int stringId, boolean isErrorState) {
        int textColor;
        if (isErrorState) {
            textColor = ContextCompat.getColor(getContext(), R.color.uac_fingerprint_error_red);
        } else {
            textColor = ContextCompat.getColor(getContext(), R.color.uac_text_color_333);
        }
        summary.setTextColor(textColor);
        summary.setText(stringId);
    }

    private void setSummary(CharSequence message, boolean isErrorState) {
        int textColor;
        if (isErrorState) {
            textColor = ContextCompat.getColor(getContext(), R.color.uac_fingerprint_error_red);
        } else {
            textColor = ContextCompat.getColor(getContext(), R.color.uac_text_color_333);
        }
        summary.setTextColor(textColor);
        summary.setText(message);
    }

    @Override
    public void show() {
        super.show();
        //距离上次锁定时间
        long sinceLastLockedTime = System.currentTimeMillis() - SettingInfoUtil.getInstance().getFingerprintLockedTimestamp();

        if (sinceLastLockedTime > 1000 && sinceLastLockedTime <= FINGERPRINT_SENSOR_LOCK_TIME) {
            startCountDown(FINGERPRINT_SENSOR_LOCK_TIME - sinceLastLockedTime);
        } else {
            setSummary(R.string.uac_please_verify_fingerprint, false);
            startVerifyFingerprint();
        }

    }

    private void startVerifyFingerprint() {
        Log.d(TAG, "startVerifyFingerprint()");
        if (fingerprintType == FINGERPRINT_TYPE_SOTER) {
            if (soterBiometricCanceller != null) {
                soterBiometricCanceller = null;
            }

            initSoterParam();
            SoterWrapperApi.requestAuthorizeAndSign(soterProcessCallback, authenticationParam);
        } else {
            if (cancellationSignal == null || cancellationSignal.isCanceled()) {
                cancellationSignal = new CancellationSignal();
            }
            fingerprintManagerCompat.authenticate(null, 0, cancellationSignal, authenticationCallback, null);
        }

    }

    private void stopVerifyFingerprint() {
        Log.d(TAG, "stopVerifyFingerprint()");
        if (soterBiometricCanceller != null) {
            soterBiometricCanceller.asyncCancelBiometricAuthentication();
        }

        if (cancellationSignal != null && !cancellationSignal.isCanceled()) {
            cancellationSignal.cancel();
        }

        if (countDownTimer != null) {
            countDownTimer.cancel();
            countDownTimer = null;
        }
    }

    public interface SuccessCallback {
        void onSuccess(String fid);
    }

    public void setSuccessCallback(SuccessCallback successCallback) {
        this.successCallback = successCallback;
    }

    /**
     * 开始倒计时
     * 调试中发现#CountDownTimer start之后第一次onTick输出的#millisUntilFinished会比#millisInFuture少若干毫秒（视具体手机而定）
     * 导致取整的时候与预期值不符，因此对#millisInFuture采用加上二分之一#countDownInterval的处理方式
     */
    private void startCountDown(final long time) {
        Log.d(TAG, "startCountDown(), time = " + time);
        final long countDownInterval = 1000L;
        if (countDownTimer == null) {
            countDownTimer = new CountDownTimer(time + countDownInterval / 2, countDownInterval) {
                @Override
                public void onTick(long millisUntilFinished) {
                    if (millisUntilFinished < countDownInterval)
                        return;//不显示0值(某些手机默认不会输出小于#countDownInterval的值)
                    setSummary(getContext().getString(R.string.uac_verify_fingerprint_fail_too_much,
                            millisUntilFinished / 1000), false);
                }

                @Override
                public void onFinish() {
                    Log.d(TAG, "onFinish()");
                    countDownTimer = null;
                    setSummary(R.string.uac_please_verify_fingerprint, false);
                    startVerifyFingerprint();
                }
            }.start();
        }
    }

    public void setDialogTitle(String userName) {
        title.setVisibility(View.VISIBLE);
        title.setText(userName);
        titleLine.setVisibility(View.VISIBLE);
    }

}
