package com.xdja.ckms.chip;

import android.content.Context;
import android.util.Log;
import android.util.Pair;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.XDJA_DEVINFO;
import com.xdja.SafeKey.XDJA_SM2_PRIKEY;
import com.xdja.SafeKey.XDJA_SM2_PUBKEY;
import com.xdja.multi.unitepin.jar.MultiChipUnitePinManager;
import com.xdja.multichip.jniapi.JarJniApiProxy;
import com.xdja.multichip.jniapi.JarMultiChipStatusManager;
import com.xdja.multichip.jniapi.JarMultiJniApiCCManager;
import com.xdja.multichip.jniapi.JarMultiJniApiManager;
import com.xdja.multichip.param.JniApiParam;
import com.xdja.safekeyjar.Base64.Base64;
import com.xdja.safekeyjar.ErrorCode;
import com.xdja.safekeyjar.util.Arithmetic;
import com.xdja.safekeyjar.util.StringResult;
//import com.xdja.unitepin.UnitePinErrorCode;

import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/9/1 <br/>
 */

public class CKMSChip {
    private static CKMSChip instance;
    private Context context;


    /**
     * 贴膜key
     */
    public static final int TYPE_COVERKEY = JniApiParam.TYPE_COVERED;

    /**
     * 蓝牙key
     */
    public static final int TYPE_BLUEKEY = JniApiParam.TYPE_BLUETOOTH;
    /**
     * 有卡，但不是蓝牙key
     */
    public static final int TYPE_NOT_BLUEKEY = 3;
    /**
     * 什么也没有
     */
    public static final int TYPE_NULL = -1;


    private CKMSChip() {
    }

    /**
     * 得到实例
     * （CKMSChip在一个进程中为单例）
     *
     * @return
     */
    public static CKMSChip getInstance() {
        if (instance == null) {
            synchronized (CKMSChip.class) {
                if (instance == null) {
                    instance = new CKMSChip();

                    Log.w("GetAll", "ckms new CKMSChip()");
                }
            }
        }
        return instance;
    }

    public static void setInstanceNull(Context context) {

        Log.w("GetAll", "ckms setInstanceNull ");

        instance = null;

        //add 2018年5月22日09:41:48 weizg 重新初始化出实例，并设置context
        //解决使用贴膜卡时，重启手机后，安通+开启自启调用CKMS接口获取挑战值失败问题
        //该问题是因为，重启手机后，SIM卡插入广播处理时，会调用该方法，把instance设置为null，
        //后面没有调用init方法，而直接调用业务接口时，context为空，返回-10000
        getInstance().setContext(context);
    }

    /**
     * 设置context
     *
     * @param context context
     * @return 0:成功；其他：失败
     */
    public int setContext(Context context) {
        if (context == null) {
            return ErrorCode.CONTEXT_NULL;
        }
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }
        JarMultiChipStatusManager.getInstance().registerListener(this.context, listener);
        return 0;
    }

    /**
     * 得到操作的卡类型
     *
     * @param context context
     * @return 返回值有三种：{@link JniApiParam#TYPE_ONBOARD},{@link JniApiParam#TYPE_TF},
     * {@link JniApiParam#TYPE_COVERED},{@link JniApiParam#TYPE_BLUETOOTH}
     */
    public int getType(Context context) {
        if (this.context == null) {
            if (context == null) {
                return ErrorCode.CONTEXT_NULL;
            }
            init(context);
        }

        if (getOperate().getProxy() != null) {
            return getOperate().cardType;
        }
        return TYPE_NULL;
    }

    /**
     * 初始化；
     * 0：表示发现了安全卡；
     * -1： 表示没有发现任何安全卡
     *
     * @param context
     * @param isSync  是否同步； true，同步，false 异步；默认不传入参数是同步；
     * @return
     */
    public int init(Context context, Boolean... isSync) {
        if (context == null) {
            return ErrorCode.CONTEXT_NULL;
        }
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }
        int ret;

        //modify 2017年10月19日16:07:48 weizg
        //因为多芯片manager getAll方法会执行时间比较长，且该方法在
        //application的oncreate调用了，为了不阻塞启动，把下面执行时间长的代码
        //放到线程中执行
        if (isSync != null && isSync.length > 0 && !isSync[0]) {
            StartInitOtherChipThread();//异步
            ret = 0;
        } else {
            ret = initOtherChipManager();
            Log.w("GetAll", "ckms initOper ret: " + ret);
        }

        Log.w("GetAll", "ckms init end: ");
        JarMultiChipStatusManager.getInstance().registerListener(this.context, listener);
        return ret;
    }

    private ExecutorService cachedThreadPool = Executors.newSingleThreadExecutor();

    void StartInitOtherChipThread() {
        cachedThreadPool.execute(new InitOtherChipRunnable());
    }

    class InitOtherChipRunnable implements Runnable {

        @Override
        public void run() {

            synchronized (CKMSChip.class) {
                if (getOperate().getProxy() == null) {
                    initOtherChipManager();
                }
            }

        }
    }

    private int initOtherChipManager() {
        //init card result
        int ret = -1;

        Log.w("GetAll", "CKMS getMultiJniApi start: ");
        Pair<Integer, List<JniApiParam>> pair = JarMultiJniApiManager.getInstance().getAll(CKMSChip.this.context);
        if (pair.first == 0) {
            //用于记录是否有
            int[] typeArrays = new int[4];
            Arrays.fill(typeArrays, -1);
            String cardId = "";
            int cardType = 0;
            int priority = 0;
            for (JniApiParam param : pair.second) {
                int tmpPriority = priorityMap.get(param.chipType);
                if (tmpPriority > priority) {
                    priority = tmpPriority;
                    cardType = param.chipType;
                    cardId = param.cardId;
                }
            }
            if (cardType == 0) {
                return ret;
            }
            Pair<Integer, JarJniApiProxy> makePair = JarMultiJniApiManager.getInstance().make(CKMSChip.this.context,cardId);
            if (makePair.first == 0) {
                getOperate().setProxy(makePair.second, cardType, cardId);
                ret = 0;
            }
        }

        Log.w("GetAll", "CKMS getMultiJniApi ret: " + ret);

        return ret;
    }


    /**
     * 得到PIN角色的重试次数
     *
     * @param pinRole 角色
     * @return {@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     * 大于等于1： 表示剩余次数；
     * -10或-16： 表示锁死；
     * 其他（-1 到 -99）：参看JNIAPI中以 'XKR_' 开头的定义
     */
    public int GetPinTryCount(int pinRole) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        return getOperate().GetPinTryCount(pinRole);
    }

    /**
     * 得到卡号
     * (此接口准备废弃，如有用到，请使用其他接口代替)
     *
     * @return
     */
    @Deprecated
    public StringResult getSafeCardID() {
        if (!judgeIsInit()) {
            return new StringResult(ErrorCode.CONTEXT_NULL, "");
        }
        int ret;
        XDJA_DEVINFO devInfo = new XDJA_DEVINFO();
        String cardId = "";
        ret = getOperate().GetDevInfo(devInfo);
        if (ret == JNIAPI.XKR_OK) {
            cardId = new String(devInfo.cardid, 0, 32);
        }
        StringResult result = new StringResult(ret, cardId);
        return result;
    }

    /**
     * 得到证书sn
     * (此接口准备废弃，如有用到，请使用其他接口代替)
     *
     * @param containNo 容器号
     * @param certType  证书类型，1：交换证书；2：签名证书
     * @return
     */
    @Deprecated
    public StringResult getSafeCardSn(int containNo, int certType) {
        if (!judgeIsInit()) {
            return new StringResult(ErrorCode.CONTEXT_NULL, "");
        }
        int ret;
        if (certType == 1) {
            certType = Arithmetic.EXC_CERT;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_CERT;
        } else {
            return new StringResult(ErrorCode.PARAM_ERROR, "");
        }
        byte[] certId = Arithmetic.convertToFid(containNo, certType);
        byte[] certBuf = new byte[2 * 1024];
        int[] certLen = new int[2];
        StringResult result;
        ret = getOperate().ReadCert(certId, certBuf, certLen);
        if (ret != JNIAPI.XKR_OK) {
            result = new StringResult(ret, "");
            return result;
        }

        X509Certificate x509Cert = Arithmetic.getX509Certificate(certBuf, certLen[0]);
        if (x509Cert != null) {
            byte[] certSn = x509Cert.getSerialNumber().toByteArray();
            System.arraycopy(certSn, 0, certBuf, 0, certSn.length);
            certLen[0] = certSn.length;
        } else {
            result = new StringResult(ErrorCode.GET_SAFE_CARD_SN_ERROR, "");
            return result;
        }

        String stringCertSn = "";
        for (int i = 0; i < certLen[0]; i++) {
            stringCertSn += String.format("%02x", certBuf[i]);
        }

        result = new StringResult(ret, stringCertSn);
        return result;
    }

    /**
     * SM2签名
     *
     * @param role      pin码角色
     * @param pin       PIN码
     * @param containNo 容器号
     * @param certType  证书类型，1：交换证书；2：签名证书
     * @param dataType  数据类型，参看{@link JNIAPI#SIGN_NOHASH} 和 {@link JNIAPI#SIGN_HASH}
     * @param dataIn    要签名的数据
     * @param inLen     要签名数据的长度
     * @param dataOut   [out]签名后的数据
     * @param outLen    [out]签名后数据的长度，签名后的数据长度在outLen[0]中
     * @return 0：成功；{@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     * 其他（-1 到 -99）：参看JNIAPI中以 'XKR_' 开头的定义
     */
    public int SM2Sign(int role, String pin, int containNo, int certType, int dataType, byte[] dataIn, int inLen,
                       byte[] dataOut, int[] outLen) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        byte[] pubfid = new byte[2];
        byte[] prifid = new byte[2];

        if (certType == 1) {
            certType = Arithmetic.EXC_CERT;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_CERT;
        } else {
            return ErrorCode.PARAM_ERROR;
        }

        byte[] certId = Arithmetic.convertToFid(containNo, certType);
        pubfid[0] = 0x00;
        pubfid[1] = (byte) (certId[1] + 1);

        prifid[0] = 0x00;
        prifid[1] = (byte) (certId[1] + 2);

        int ret;
        ret = getOperate().SM2Sign(pubfid, prifid, dataType, dataIn, inLen, dataOut, outLen);
        if (ret == JNIAPI.XKR_NO_POWER) {
            ret = getOperate().VerifyPIN(role, pin.getBytes(), pin.length());
            if (ret == JNIAPI.XKR_OK) {
                ret = getOperate().SM2Sign(pubfid, prifid, dataType, dataIn, inLen, dataOut, outLen);
            }
        }
        return ret;
    }

    /**
     * SM2验签
     *
     * @param role          pin码角色
     * @param pin           PIN码
     * @param containNo     容器号
     * @param certType      证书类型，1：交换证书；2：签名证书
     * @param dataType      数据类型，参看{@link JNIAPI#SIGN_NOHASH} 和 {@link JNIAPI#SIGN_HASH}
     * @param dataIn        签名的数据
     * @param inLen         签名数据的长度
     * @param signatureData 要验签的数据
     * @return 0：验签成功；{@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     * 其他（-1 到 -99）：参看JNIAPI中以 'XKR_' 开头的定义
     */
    public int SM2SignVerify(int role, String pin, int containNo, int certType, int dataType, byte[] dataIn,
                             int inLen, byte[] signatureData) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        byte[] pubfid = new byte[2];
        if (certType == 1) {
            certType = Arithmetic.EXC_CERT;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_CERT;
        } else {
            return ErrorCode.PARAM_ERROR;
        }

        byte[] certId = Arithmetic.convertToFid(containNo, certType);
        pubfid[0] = 0x00;
        pubfid[1] = (byte) (certId[1] + 1);

        int ret;
        ret = getOperate().SM2SignVerify(pubfid, dataType, null, dataIn, inLen, signatureData);
        if (ret == JNIAPI.XKR_NO_POWER) {
            ret = getOperate().VerifyPIN(role, pin.getBytes(), pin.length());
            if (ret == JNIAPI.XKR_OK) {
                ret = getOperate().SM2SignVerify(pubfid, dataType, null, dataIn, inLen, signatureData);
            }
        }
        return ret;
    }

    /**
     * SM3算法
     *
     * @param dataIn  要做运算的数据
     * @param inLen   要做运算数据的长度
     * @param dataOut [out]出参，代码中不能为空，大小为32
     * @return 0：成功；{@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     * 其他（-1 到 -99）：参看JNIAPI中以 'XKR_' 开头的定义
     */
    public int SM3(byte[] dataIn, int inLen, byte[] dataOut) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        return getOperate().SM3(dataIn, inLen, dataOut);
    }

    /**
     * SM1KEY算法
     * （贴膜卡不支持此方法调用）
     *
     * @param tmpKey  key
     * @param dataIn  要运算的数据
     * @param inLen   数据长度
     * @param flag    运算标志位（参看{@link JNIAPI#ECB_DECRYPT},{@link JNIAPI#ECB_ENCRYPT},{@link JNIAPI#CBC_DECRYPT},{@link JNIAPI#CBC_DECRYPT}）
     * @param dataOut [out]加密后的数据，使用时不能为空
     * @param IV      IV
     * @return 0：成功；{@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     * 其他（-1 到 -99）：参看JNIAPI中以 'XKR_' 开头的定义
     */
    public int SM1KEY(byte[] tmpKey, byte[] dataIn, int inLen, int flag, byte[] dataOut, byte[] IV) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        return getOperate().SM1KEY(tmpKey, dataIn, inLen, flag, dataOut, IV);
    }

    /**
     * SM1算法
     * （贴膜卡不支持此方法调用）
     *
     * @param dataIn  要运算的数据
     * @param inLen   数据长度
     * @param flag    运算标志位（参看{@link JNIAPI#ECB_DECRYPT},{@link JNIAPI#ECB_ENCRYPT},{@link JNIAPI#CBC_DECRYPT},{@link JNIAPI#CBC_DECRYPT}）
     * @param dataOut [out]加密后的数据，使用时不能为空
     * @param kID     密钥id
     * @param IV      iv
     * @return 0：成功；{@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     * 其他（-1 到 -99）：参看JNIAPI中以 'XKR_' 开头的定义
     */
    public int SM1(byte[] dataIn, int inLen, int flag, byte[] dataOut, byte kID, byte[] IV) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        return getOperate().SM1(dataIn, inLen, flag, dataOut, kID, IV);
    }

    /**
     * 是否激活
     * （一般只有芯片才会存在未激活的情况）
     *
     * @return {@link JNIAPI#XKR_NOT_ACTIVATED}: 未激活;<br>
     * {@link JNIAPI#XKR_OK}: 已激活<br>
     * {@link JNIAPI#XKR_BACK_DATA}, {@link JNIAPI#XKR_PARAMETER}, {@link JNIAPI#XKR_NO_THIS_CMD}: 如果返回这三个值可以认为芯片已激活，返回这些值的原因是：一些安全芯片COS不支持激活状态。<br>
     * {@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     */
    public int GetActivateState() {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        return getOperate().GetActivateState();
    }

    /**
     * 在线通过url激活芯片
     *
     * @param url [in]激活的地址，比如： https://11.12.110.205:8099/scas-web/api
     * @return 错误码
     * {@link JNIAPI#XKR_OK}: 成功<br>
     * 其他：表示错误，请参见其他错误码定义
     */
    public int ActivateCardByURL(String url) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        return getOperate().ActivateCardByURL(url);
    }

    /**
     * 产生SM2公私钥对
     *
     * @param pubfid [in]公钥文件ID,为0x00 0x00时表示公钥导出卡外
     * @param prifid [in]私钥文件ID,公私钥ID均为0x00 0x00时表示私钥导出卡外
     * @param pubkey [out]SM2公钥结构，公钥文件ID为0x00 0x00时有效
     * @param prikey [out]SM2私钥结构，公私钥文件ID均为0x00 0x00时有效
     * @return 0：成功；{@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     * 其他（-1 到 -99）：参看JNIAPI中以 'XKR_' 开头的定义
     */
    public int GenSM2KeyPair(byte[] pubfid, byte[] prifid, XDJA_SM2_PUBKEY pubkey, XDJA_SM2_PRIKEY prikey) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        return getOperate().GenSM2KeyPair(pubfid, prifid, pubkey, prikey);
    }

    /**
     * 获取安全卡中证书
     *
     * @param containNo [in]容器编号[0-9]
     * @param certType  [in]证书类型[1加密证书, 2签名证书]
     * @return 返回含安全卡中用户证书对象，调用getErrorInfo方法获取失败的原因
     */
    public StringResult getSafeCardCertification(int containNo, int certType) {
        if (!judgeIsInit()) {
            return new StringResult(ErrorCode.CONTEXT_NULL, "");
        }
        int ret;
        if (certType == 1) {
            certType = Arithmetic.EXC_CERT;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_CERT;
        } else {
            return new StringResult(ErrorCode.PARAM_ERROR, "");
        }
        byte[] certId = Arithmetic.convertToFid(containNo, certType);

        StringResult result;

        byte[] certBuf = new byte[2 * 1024];
        int[] certLen = new int[2];
        ret = getOperate().ReadCert(certId, certBuf, certLen);
        if (ret != JNIAPI.XKR_OK) {
            result = new StringResult(ret, "");
            return result;
        }

        String stringCertContent;
        stringCertContent = Base64.encodeBytes(certBuf, 0, certLen[0], Base64.DONT_BREAK_LINES);
        result = new StringResult(ret, stringCertContent);
        return result;
    }

    /**
     * 获取芯片信息
     *
     * @param devInfo [out]芯片信息
     * @return 0：成功；{@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     * 其他（-1 到 -99）：参看JNIAPI中以 'XKR_' 开头的定义
     */
    public int GetDevInfo(XDJA_DEVINFO devInfo) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        return getOperate().GetDevInfo(devInfo);
    }

    /**
     * 检查卡是否存在
     *
     * @return 0：存在，其他：不存在；{@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     */
    public int detectSafeCard() {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        XDJA_DEVINFO devInfo = new XDJA_DEVINFO();
        int result = GetDevInfo(devInfo);
        return result;
    }

    /**
     * 验证PIN码
     *
     * @param pinRole pin码角色
     * @param pin     pin码
     * @param pinLen  pin码长度
     * @return 0：成功；{@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     * 其他（-1 到 -99）：参看JNIAPI中以 'XKR_' 开头的定义
     */
    public int VerifyPIN(int pinRole, byte[] pin, int pinLen) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        return getOperate().VerifyPIN(pinRole, pin, pinLen);
    }

    /**
     * 修改PIN码
     *
     * @param pinRole PIN码角色
     * @param oldPin  老PIN码
     * @param oldLen  老PIN码长度
     * @param newPin  新PIN码
     * @param newLen  新PIN码长度
     * @return 0：成功；{@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     * 其他（-1 到 -99）：参看JNIAPI中以 'XKR_' 开头的定义
     */
    public int ChangePIN(int pinRole, byte[] oldPin, int oldLen, byte[] newPin, int newLen) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        return getOperate().ChangePIN(pinRole, oldPin, oldLen, newPin, newLen);
    }

    /**
     * 得到PIN码
     *
     * @param role PIN码角色
     * @param pin  得到的PIN码，如果成功，PIN码在pin[0]中
     * @return 0：成功；其他：失败；{@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     */
    public int uniteGetPin(int role, String[] pin) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        return getOperate().uniteGetPin(role, pin);
    }

    /**
     * 贴膜卡使用的SM4卡内算法
     *
     * @param dataIn  要运算的数据
     * @param dataLen 要运算数据的长度
     * @param flag    运算标志位（参看{@link JNIAPI#ECB_DECRYPT},{@link JNIAPI#ECB_ENCRYPT},{@link JNIAPI#CBC_DECRYPT},{@link JNIAPI#CBC_DECRYPT}）
     * @param dataOut [out]运算结果
     * @param kID     密钥id
     * @param IV      iv
     * @return 0：成功；其他：失败；{@link ErrorCode#CONTEXT_NULL}:表示没有初始化;
     */
    public int SM4ForCoverKey(byte[] dataIn, int dataLen, int flag, byte[] dataOut, byte kID, byte[] IV) {
        if (!judgeIsInit()) {
            return ErrorCode.CONTEXT_NULL;
        }
        return getOperate().SM4ForCoverKey(dataIn, dataLen, flag, dataOut, kID, IV);
    }

    /**
     * 当没有getOperate()中的XDJASafeTF和JarJniApiProxy的实例为空时，再次init一下，
     * 用于第一次init时没有插入安全芯片，而真正用时已经插入安全芯片，而实例却还为空的情况。
     *
     * @return true：表示可以继续调用；false：表示不能继续调用
     */
    private boolean judgeIsInit() {
        if (context == null) {
            Log.w("GetAll", "CKMSChip judgeIsInit context null ");
            return false;
        }
        if (getOperate().getProxy() == null) {
            Log.w("GetAll", "CKMSChip operate.getProxy() null ");
            return reInit(context);
        }
        return true;
    }


    private boolean reInit(Context context) {
        synchronized (CKMSChip.class) {
            if (getOperate().getProxy() == null) {
                init(context, true);
                if (getOperate().getProxy() != null) {
                    return true;
                }
            } else {//add 2018年4月20日14:55:38 weizg 如果后进入同步块的子线程发现变量不为空，则直接返回true
                return true;
            }
        }

        return false;
    }

    //========================================================================================================

    private CardOperate operate = new CardOperate();

    private CardOperate getOperate() {
        return operate;
    }

    class CardOperate {
        private JarJniApiProxy proxy;
        private int cardType = 0;
        private String cardId = "";

        public JarJniApiProxy getProxy() {
            return proxy;
        }

        public void setProxy(JarJniApiProxy proxy, int typeProxy, String cardId) {
            this.proxy = proxy;
            this.cardType = typeProxy;
            this.cardId = cardId;
        }


        public int GetPinTryCount(int pinRole) {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.GetPinTryCount(pinRole);
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int GetDevInfo(XDJA_DEVINFO devInfo) {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.GetDevInfo(devInfo);
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int ReadCert(byte[] fid, byte[] certBuf, int[] certLen) {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.ReadCert(fid, certBuf, certLen);
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int SM2Sign(byte[] pubfid, byte[] prifid, int dataType, byte[] dataIn, int inLen, byte[] dataOut,
                           int[] outLen) {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.SM2Sign(pubfid, prifid, dataType, dataIn, inLen, dataOut, outLen);
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int SM2SignVerify(byte[] pubfid, int dataType, XDJA_SM2_PUBKEY pubkey, byte[] dataIn, int inLen,
                                 byte[] signatureData) {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.SM2SignVerify(pubfid, dataType, pubkey, dataIn, inLen, signatureData);
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int VerifyPIN(int pinRole, byte[] pin, int pinLen) {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.VerifyPIN(pinRole, pin, pinLen);
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int ChangePIN(int pinRole, byte[] oldPin, int oldLen, byte[] newPin, int newLen) {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.ChangePIN(pinRole, oldPin, oldLen, newPin, newLen);
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int SM3(byte[] dataIn, int inLen, byte[] dataOut) {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.SM3(dataIn, inLen, dataOut);
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int SM1KEY(byte[] tmpKey, byte[] datain, int inLen, int flag, byte[] dataOut, byte[] IV) {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.SM1KEY(tmpKey, datain, inLen, flag, dataOut, IV);
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int SM1(byte[] dataIn, int inLen, int flag, byte[] dataOut, byte kID, byte[] IV) {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.SM1(dataIn, inLen, flag, dataOut, kID, IV);
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int GetActivateState() {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.GetActivateState();
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int ActivateCardByURL(String url) {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.ActivateCardByURL(url);
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int GenSM2KeyPair(byte[] pubfid, byte[] prifid, XDJA_SM2_PUBKEY pubkey, XDJA_SM2_PRIKEY prikey) {
            if (!judge()) {
                return JNIAPI.XKR_NO_KEY;
            }
            if (proxy != null) {
                return proxy.GenSM2KeyPair(pubfid, prifid, pubkey, prikey);
            } else {
                return JNIAPI.XKR_NO_KEY;
            }
        }

        public int SM4ForCoverKey(byte[] dataIn, int dataLen, int flag, byte[] dataOut, byte kID, byte[] IV) {
            return JarMultiJniApiCCManager.getInstance().SM4(proxy, dataIn, dataLen, flag, dataOut, kID, IV);
        }

        public int uniteGetPin(int role, String[] pin) {
            if (pin == null || pin.length < 1) {
                return com.xdja.ckms.chip.ErrorCode.ERROR_PARAM;
            }
            if (!judge()) {
                return com.xdja.ckms.chip.ErrorCode.ERROR_PIN_NULL;
            }
            if (proxy != null) {
                Pair<Integer, String> cardIdPair = proxy.getCardId();
                if (cardIdPair.first != 0) {
                    return com.xdja.ckms.chip.ErrorCode.ERROR_PIN_NULL;
                }
                Pair<Integer, String> pinPair = MultiChipUnitePinManager.getInstance().getPin(context, cardIdPair.second, role);
                if (pinPair.first != 0) {
                    return com.xdja.ckms.chip.ErrorCode.ERROR_PIN_NULL;
                }
                pin[0] = pinPair.second;
                return 0;
            } else {
                return com.xdja.ckms.chip.ErrorCode.ERROR_PIN_NULL;
            }
        }

        private boolean judge() {
            if (proxy == null) {
                int ret = init(context);
                if (ret != 0) {
                    return false;
                }
            }
            return true;
        }
    }

    CKMSMultiChipStatusListener listener = new CKMSMultiChipStatusListener();

    class CKMSMultiChipStatusListener implements JarMultiChipStatusManager.MultiChipStatusListener {
        @Override
        public void chipAdd(String cardId, int cardType) {
            int flag = judgeCardTypePriority(cardType, getOperate().cardType);
            //如果新增的卡类型 优先级大于 现在的卡类型，则重置现在的代理为空
            if (flag > 0) {
                operate = new CardOperate();
            }
            //如果新增的卡类型 和 现在的卡类型 一样，再判断 卡号是否相同
            else if (flag == 0) {
                //如果卡号不相等，也需要重置
                if (!cardId.equals(getOperate().cardId)) {
                    operate = new CardOperate();
                }
            }
        }

        @Override
        public void chipRemove(int cardType) {
            if (cardType == getOperate().cardType){
                operate = new CardOperate();
            }
        }

        @Override
        public void serviceDeath() {
        }

        @Override
        public void serviceStart() {
        }
    }

    /**
     * 判断两个卡类型的优先级;
     * 如果 firstType 大于 secondType，返回 1；
     * 如果 firstType 等于 secondType，返回 0；
     * 如果 firstType 小于 secondType，返回 -1；
     *
     * @param firstType
     * @param secondType
     * @return
     */
    private int judgeCardTypePriority(int firstType, int secondType) {
        int diff = priorityMap.get(firstType) - priorityMap.get(secondType);
        if (diff > 0) {
            return 1;
        } else if (diff == 0) {
            return 0;
        } else {
            return -1;
        }
    }

    /**
     * 卡类型的优先级，
     * 优先级 板载芯片 > TF卡 > 贴膜卡 > 蓝牙key
     */
    private static HashMap<Integer, Integer> priorityMap = new HashMap<>();

    static {
        priorityMap.put(JniApiParam.TYPE_ONBOARD, 100);
        priorityMap.put(JniApiParam.TYPE_TF, 99);
        priorityMap.put(JniApiParam.TYPE_COVERED, 98);
        priorityMap.put(JniApiParam.TYPE_BLUETOOTH, 97);
        priorityMap.put(0, 0);

    }
}



