package com.xdja.ckms.chip;

import com.xdja.multichip.jniapi.JarJniApiProxy;
import com.xdja.safekeyjar.ErrorCode;
import com.xdja.safekeyjar.util.Arithmetic;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/10/23 <br/>
 */

public class MultiJniApiExtra {

    /**
     * 私钥运算
     *
     * @param pin       [in]用户口令
     * @param role      [in]用户角色
     * @param pDataIn   [in]待运算数据
     * @param pDataOut  [out]运算结果
     * @param outLen    [out]运算结果长度
     * @param containNo [in]容器编号[0-9]
     * @param certType  [in]证书类型，1交换证书，2签名证书
     * @return 成功返回0，非0表示失败，调用getErrorInfo方法获取失败的原因
     */
    public int rsaPriKeyCalc(JarJniApiProxy proxy, String pin, int role, byte[] pDataIn,
                             byte[] pDataOut, int[] outLen, int containNo, int certType) {
        byte[] prikeyId = new byte[2];
        if (certType == 1) {
            certType = Arithmetic.EXC_PRIKEY;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_PRIKEY;
        } else {
            return ErrorCode.PUB_KEY_CALCULATE_PARAM_ERROR;
        }

        prikeyId = Arithmetic.convertToFid(containNo, certType);
        int ret;
        int padLen;
        int inLen;
        byte[] tempDataIn = new byte[128];
        byte[] tempDataOut = new byte[128];
        int[] tempOutLen = new int[1];

        ret = proxy.VerifyPIN(role, pin.getBytes(), pin.length());
        if (ret != 0) {
            return ret;
        }
        inLen = pDataIn.length;
        //私钥运算输入数据长度小于128字节需要按照pkcs1pading方式填充
        if (inLen < 128) {
            padLen = 128 - 3 - inLen;

            tempDataIn[0] = 0x00;
            tempDataIn[1] = 0x01;
            for (int i = 2; i < 2 + padLen; i++) {
                tempDataIn[i] = (byte) 0xff;
            }
            tempDataIn[2 + padLen] = 0x00;
            System.arraycopy(pDataIn, 0, tempDataIn, 3 + padLen, inLen);
            ret = proxy.RSAPriKeyCalc(prikeyId, tempDataIn, tempDataIn.length, tempDataOut, tempOutLen);
        } else {
            ret = proxy.RSAPriKeyCalc(prikeyId, pDataIn, pDataIn.length, tempDataOut, tempOutLen);
        }
        //私钥运算成功
        if (ret == 0) {
            boolean pkcs1padding = false;
            //pkcs1去填充
            if (tempOutLen[0] == 128) {
                if (tempDataOut[0] == 0x00 && tempDataOut[1] == 0x02) {
                    int i = 0;
                    //查找私钥解密后有效数据起始位置的前一个字节
                    for (i = 2; i < 128; i++) {
                        if (0x00 == tempDataOut[i]) {
                            break;
                        }
                    }

                    if (128 != i) {
                        //计算数据有效长度
                        outLen[0] = tempOutLen[0] - (i + 1);
                        System.arraycopy(tempDataOut, i + 1, pDataOut, 0, outLen[0]);
                    }
                }
            }

            //未按照pkcs1方式填充原数据返回
            if (!pkcs1padding) {
                outLen[0] = tempOutLen[0];
                System.arraycopy(tempDataOut, 0, pDataOut, 0, outLen[0]);
            }
        }
        return ret;
    }

    /**
     * 公钥运算
     *
     * @param pDataIn   [in]待运算数据
     * @param pDataOut  [out]运算结果
     * @param outLen    [out]运算结果长度
     * @param containNo [in]容器编号[0-9]
     * @param certType  [in]证书类型，1交换公钥，2签名公钥
     * @return 成功返回0，非0表示失败，调用getErrorInfo方法获取失败的原因
     */
    public int rsaPubKeyCalc(JarJniApiProxy proxy, byte[] pDataIn, byte[] pDataOut, int[] outLen, int containNo, int certType) {
        byte[] pubkeyId = new byte[2];
        if (certType == 1) {
            certType = Arithmetic.EXC_PUBKEY;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_PUBKEY;
        } else {
            return ErrorCode.PUB_KEY_CALCULATE_PARAM_ERROR;
        }
        pubkeyId = Arithmetic.convertToFid(containNo, certType);
        byte[] tempDataIn = new byte[128];
        byte[] tempDataOut = new byte[128];
        int[] tempOutLen = new int[1];
        int ret = -1;
        int inLen = 0;
        int padLen = 0;

        inLen = pDataIn.length;
        //公钥运算输入数据长度小于128字节需要按照pkcs1pading方式填充
        if (inLen < 128) {
            padLen = 128 - 3 - inLen;

            tempDataIn[0] = 0x00;
            tempDataIn[1] = 0x02;
            for (int i = 2; i < 2 + padLen; i++) {
                tempDataIn[i] = (byte) (Math.random() * 254 + 1);
            }
            tempDataIn[2 + padLen] = 0x00;
            System.arraycopy(pDataIn, 0, tempDataIn, 3 + padLen, inLen);

            ret = proxy.RSAPubKeyCalc(pubkeyId, null, tempDataIn, 128, tempDataOut, tempOutLen);
        } else {
            ret = proxy.RSAPubKeyCalc(pubkeyId, null, pDataIn, pDataIn.length, tempDataOut, tempOutLen);
        }

        //公钥运算成功
        if (ret == 0) {
            boolean pkcs1padding = false;
            //pkcs1去填充
            if (tempOutLen[0] == 128) {
                if (tempDataOut[0] == 0x00 && tempDataOut[1] == 0x01) {
                    int i = 0;
                    //查找公钥解密后有效数据起始位置的前一个字节
                    for (i = 2; i < 128; i++) {
                        if (0x00 == tempDataOut[i]) {
                            pkcs1padding = true;
                            break;
                        }
                    }

                    //计算数据有效长度
                    if (128 != i) {
                        outLen[0] = tempOutLen[0] - (i + 1);
                        System.arraycopy(tempDataOut, i + 1, pDataOut, 0, outLen[0]);
                    }
                }
            }

            //未按照pkcs1方式填充原数据返回
            if (!pkcs1padding) {
                outLen[0] = tempOutLen[0];
                System.arraycopy(tempDataOut, 0, pDataOut, 0, outLen[0]);
            }
        }
        return ret;
    }

    /**
     * SM2公钥加密
     *
     * @param role      [in]角色
     * @param pin       [in]pin码
     * @param containNo [in]容器编号[0-9]
     * @param certType  [in]证书类型[1交换证书, 2签名证书]
     * @param dataIn    [in]明文数据,最大长度不超过158
     * @param inLen     [in]数据长度
     * @param dataOut   [out]加密后密文,缓冲长度至少为inLen+97
     * @param outLen    [out]加密后数据长度
     * @return 错误码
     * XKR_OK 成功
     */
    private int sm2Encrypt(JarJniApiProxy proxy, int role, String pin, int containNo, int certType, byte[] dataIn, int inLen,
                           byte[] dataOut, int[] outLen) {
        byte[] pubfid = new byte[2];

        if (certType == 1) {
            certType = Arithmetic.EXC_CERT;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_CERT;
        } else {
            return ErrorCode.PARAM_ERROR;
        }

        byte[] certId = Arithmetic.convertToFid(containNo, certType);
        pubfid[0] = 0x00;
        pubfid[1] = (byte) (certId[1] + 1);

        int ret;
        ret = proxy.VerifyPIN(role, pin.getBytes(), pin.length());
        if (ret == 0) {
            ret = proxy.SM2Encrypt(pubfid, null, dataIn, inLen, dataOut, outLen);
        }
        return ret;
    }

    /**
     * SM2私钥解密
     *
     * @param role      [in]角色
     * @param pin       [in]pin码
     * @param containNo [in]容器编号[0-9]
     * @param certType  [in]证书类型[1交换证书, 2签名证书]
     * @param dataIn    [in]密文数据,最大长度不超过SM2_BLOCK_MAX+97
     * @param inLen     [in]数据长度
     * @param dataOut   [out]解密后的明文数据,缓冲区至少为inLen-97
     * @param outLen    [out]解密后数据长度
     * @return 错误码
     * XKR_OK        成功
     * XKR_NO_POWER  权限不够
     */
    public int sm2Decrypt(JarJniApiProxy proxy, int role, String pin, int containNo, int certType,
                          byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        byte[] prifid = new byte[2];

        if (certType == 1) {
            certType = Arithmetic.EXC_CERT;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_CERT;
        } else {
            return ErrorCode.PARAM_ERROR;
        }
        byte[] certId = Arithmetic.convertToFid(containNo, certType);
        prifid[0] = 0x00;
        prifid[1] = (byte) (certId[1] + 2);
        int ret;
        ret = proxy.VerifyPIN(role, pin.getBytes(), pin.length());
        if (ret == 0) {
            ret = proxy.SM2Decrypt(prifid, dataIn, inLen, dataOut, outLen);
        }
        return ret;
    }


    /**
     * SM2数据签名验证
     *
     * @param role          [in]角色
     * @param pin           [in]pin码
     * @param containNo     [in]容器编号[0-9]
     * @param certType      [in]证书类型[1交换证书, 2签名证书]
     * @param dataType      [in]数据类型SIGN_DATA_TYPE
     * @param dataIn        [in]数据
     * @param inLen         [in]摘要数据长度必须为32
     * @param signatureData [in]验签数据
     * @return 错误码
     * XKR_OK        成功
     * XKR_NO_POWER  权限不够
     */
    public int sm2SignVerify(JarJniApiProxy proxy, int role, String pin, int containNo, int certType,
                             int dataType, byte[] dataIn, int inLen, byte[] signatureData) {
        byte[] pubfid = new byte[2];

        if (certType == 1) {
            certType = Arithmetic.EXC_CERT;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_CERT;
        } else {
            return ErrorCode.PARAM_ERROR;
        }

        byte[] certId = Arithmetic.convertToFid(containNo, certType);
        pubfid[0] = 0x00;
        pubfid[1] = (byte) (certId[1] + 1);

        int ret;
        ret = proxy.VerifyPIN(role, pin.getBytes(), pin.length());
        if (ret == 0) {
            ret = proxy.SM2SignVerify(pubfid, dataType, null, dataIn, inLen, signatureData);
        }
        return ret;
    }

    /**
     * SM2数据签名
     *
     * @param role      [in]角色
     * @param pin       [in]pin码
     * @param containNo [in]容器编号[0-9]
     * @param certType  [in]证书类型[1交换证书, 2签名证书]
     * @param dataType  [in]数据类型SIGN_DATA_TYPE
     * @param dataIn    [in]数据
     * @param inLen     [in]摘要数据数据长度必须为32
     * @param dataOut   [out]输出签名数据 缓冲区长度必须大于64字节。加密后的长度为64位。
     * @param outLen    [out]输出数据长度 当dataOut为NULL，此值必须为0。虽然dataOut长度固定，但建议通过outLen[0]来判断是否是64，来判断范围值是否正确
     * @return 错误码
     * XKR_OK 成功
     */
    public int sm2Sign(JarJniApiProxy proxy, int role, String pin, int containNo, int certType,
                       int dataType, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {

        byte[] pubfid = new byte[2];
        byte[] prifid = new byte[2];

        if (certType == 1) {
            certType = Arithmetic.EXC_CERT;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_CERT;
        } else {
            return ErrorCode.PARAM_ERROR;
        }

        byte[] certId = Arithmetic.convertToFid(containNo, certType);
        pubfid[0] = 0x00;
        pubfid[1] = (byte) (certId[1] + 1);

        prifid[0] = 0x00;
        prifid[1] = (byte) (certId[1] + 2);

        int ret;
        ret = proxy.VerifyPIN(role, pin.getBytes(), pin.length());
        if (ret == 0) {
            ret = proxy.SM2Sign(pubfid, prifid, dataType, dataIn, inLen, dataOut, outLen);
        }
        return ret;
    }
}
