package com.xdja.multichip.jniapi;

import android.content.Context;
import android.os.Binder;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;
import android.os.Message;
import android.os.Messenger;
import android.util.Pair;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.XDJA_DEVINFO;
import com.xdja.SafeKey.XDJA_DIR;
import com.xdja.SafeKey.XDJA_FILE;
import com.xdja.SafeKey.XDJA_KEY_ATTR;
import com.xdja.SafeKey.XDJA_RSA_PRIKEY;
import com.xdja.SafeKey.XDJA_RSA_PUBKEY;
import com.xdja.SafeKey.XDJA_SM2_PARAM;
import com.xdja.SafeKey.XDJA_SM2_PRIKEY;
import com.xdja.SafeKey.XDJA_SM2_PUBKEY;
import com.xdja.multichip.IMultiJniApi;

import java.security.cert.X509Certificate;

import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.GET_SAFE_CARD_SN_ERROR;
import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_EXCEPTION;
import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_PARAM_ERROR;


/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/7/24 <br/>
 */

public class JarJniApiProxy {

    private Context context;
    protected IMultiJniApi binder;
    private String cardId;
    private long handle = 0;
    protected int cardType;

    protected JarJniApiProxy(Context context, IMultiJniApi binder, int cardType) {
        this.context = context.getApplicationContext();
        if (this.context == null){
            this.context = context;
        }
        this.binder = binder;
        this.cardType = cardType;
    }

    public JarJniApiProxy(Context context, IMultiJniApi binder, String cardId, int cardType) {
        this.context = context.getApplicationContext();
        if (this.context == null) {
            this.context = context;
        }
        this.binder = binder;
        this.cardId = cardId;
        this.cardType = cardType;
        binderDeathLink();
    }

    /**
     * 获取安全卡硬件ID号
     *
     * @return 其中：Pair.first 是错误码
     * Pair.second 返回安全卡硬件ID号的对象<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, String> getCardId() {
        XDJA_DEVINFO devInfo = new XDJA_DEVINFO();
        int ret = GetDevInfo(devInfo);
        if (ret == JNIAPI.XKR_OK) {
            String cardId = new String(devInfo.cardid, 0, 32);
            return Pair.create(ret, cardId);
        }
        return Pair.create(ret, "");
    }

    /**
     * 获取卡类型
     *
     * @return
     */
    public int getCardType() {
        return cardType;
    }

    /**
     * 获取证书SN号
     *
     * @param containNo 容器编号[0-9]
     * @param certType  证书类型[1加密证书, 2签名证书]
     * @return 其中：Pair.first 是错误码
     * Pair.second 返回安全卡硬件ID号的对象<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, String> getSn(int containNo, int certType) {
        int ret = 0;
        if (certType == 1) {
            certType = Arithmetic.EXC_CERT;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_CERT;
        } else {
            return Pair.create(RET_PARAM_ERROR, "");
        }
        byte[] certId = Arithmetic.convertToFid(containNo, certType);

        byte[] certBuf = new byte[2 * 1024];
        int[] certLen = new int[2];
        ret = ReadCert(certId, certBuf, certLen);
        if (ret != JNIAPI.XKR_OK) {
            return Pair.create(ret, "");
        }

        X509Certificate x509Cert = Arithmetic.getX509Certificate(certBuf, certLen[0]);
        if (x509Cert != null) {
            byte[] certSn = x509Cert.getSerialNumber().toByteArray();
            System.arraycopy(certSn, 0, certBuf, 0, certSn.length);
            certLen[0] = certSn.length;
        } else {
            return Pair.create(GET_SAFE_CARD_SN_ERROR, "");
        }

        String stringCertSn = "";
        for (int i = 0; i < certLen[0]; i++) {
            stringCertSn += String.format("%02x", certBuf[i]);
        }
        return Pair.create(ret, stringCertSn);
    }

    /**
     * 获取卡库版本
     *
     * @param verion  卡库版本
     * @param ver_len 长度
     * @return 错误码  0 成功
     */
    public int GetVersion(byte[] verion, int[] ver_len) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.GetVersion(verion, ver_len);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 获得设备独占使用权（事务锁）
     * 超时时间30秒
     *
     * @return 错误码  0 成功
     */
    public int LockDev() {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.LockDev(handle);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 释放设备独占使用权（事务锁）
     *
     * @return 错误码 0 成功
     */
    public int UnlockDev() {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.UnlockDev(handle);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 设备命令传输
     *
     * @param cmd     设备命令（APDU指令）
     * @param cmdLen  命令长度
     * @param dataOut 返回结果数据
     * @param outLen  输入表示结果数据缓冲大小，输出表示结果数据实际长度
     * @return 错误码 0 成功
     */
    public int Transmit(byte[] cmd, int cmdLen, byte[] dataOut, int[] outLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.Transmit(handle, cmd, cmdLen, dataOut, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 获取设备信息，包括卡ID、COS版本等
     *
     * @param devInfo 返回设备信息
     * @return 错误码 0 成功
     */
    public int GetDevInfo(XDJA_DEVINFO devInfo) {
        if (devInfo == null) {
            return RET_PARAM_ERROR;
        }
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.GetDevInfo(handle, devInfo);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 激活芯片设备
     *
     * @param param    激活因子
     * @param paramLen 激活因子长度(一般为256字节)
     * @return 错误码 0: 成功<br>
     */
    public int ActivateCard(byte[] param, int paramLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.ActivateCard(handle, param, paramLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 获取芯片激活状态
     *
     * @return 错误码：<br>
     * {@link JNIAPI#XKR_NOT_ACTIVATED }: 未激活;<br>
     * {@link JNIAPI#XKR_OK }: 已激活<br>
     * {@link JNIAPI#XKR_PARAMETER }, {@link JNIAPI#XKR_PARAMETER},
     * {@link JNIAPI#XKR_NO_THIS_CMD}: 如果返回这三个值可以认为芯片已激活，返回这些值的原因是：一些安全芯片COS不支持激活状态。<br>
     * 其他：表示错误，请参见其他错误码定义
     */
    public int GetActivateState() {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.GetActivateState(handle);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 修改PIN
     *
     * @param pinRole [in]PIN码角色
     * @param oldPin  [in]旧PIN码
     * @param oldLen  [in]旧PIN码长度
     * @param newPin  [in]新PIN码
     * @param newLen  [in]新PIN码长度
     * @return 错误码
     * {@link JNIAPI#XKR_OK}          成功
     * 正数,剩余重试次数（N为1-9）
     * {@link JNIAPI#XKR_PASSWORD}    密码错误,剩余重试次数0
     * {@link JNIAPI#XKR_KEY_LOCKED}  密钥被锁死
     */
    public int ChangePIN(int pinRole, byte[] oldPin, int oldLen, byte[] newPin, int newLen) {
        if (oldPin == null || oldPin.length < oldLen || newPin == null || newPin.length < newLen) {
            return RET_PARAM_ERROR;
        }
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.ChangePIN(handle, pinRole, oldPin, oldLen, newPin, newLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 获取PIN码信息
     *
     * @param pinRole PIN码角色
     * @return 正数 PIN码重试次数
     */
    public int GetPinTryCount(int pinRole) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.GetPinTryCount(handle, pinRole);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 校验PIN
     *
     * @param pinRole [in]PIN码角色
     * @param pin     [in]PIN码
     * @param pinLen  [in]PIN码长度
     * @return 错误码
     * {@link JNIAPI#XKR_OK}          成功
     * 正数,剩余重试次数（N为1-9）
     */
    public int VerifyPIN(int pinRole, byte[] pin, int pinLen) {
        if (pin == null || pin.length < pinLen) {
            return RET_PARAM_ERROR;
        }
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.VerifyPIN(handle, pinRole, pin, pinLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * 解锁PIN
     *
     * @param id     [in] 解锁密钥ID
     * @param key    [in] 解锁码
     * @param keyLen [in] 解锁码长度
     * @param newPin [in] 新PIN码
     * @param newLen [in] 新PIN码长度
     * @return 错误码
     * @retval {@link JNIAPI#XKR_OK} 成功
     */
    public int UnlockPIN(int id, byte[] key, int keyLen, byte[] newPin, int newLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.UnlockPIN(handle, id, key, keyLen, newPin, newLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * 重装PIN
     *
     * @param pinRole [in]PIN码角色
     * @param key     [in]PIN码重装码
     * @param keyLen  [in]重装码长度
     * @param newPin  [in]新PIN码
     * @param newLen  [in]新PIN码长度
     * @return {@link JNIAPI#XKR_OK} 成功
     */
    public int ReloadPIN(int pinRole, byte[] key, int keyLen, byte[] newPin, int newLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.ReloadPIN(handle, pinRole, key, keyLen, newPin, newLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 卡重置，清除安全状态
     *
     * @return {@link JNIAPI#XKR_OK} 成功
     */
    public int CardReset() {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.CardReset(handle);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 创建目录
     * 条件:创建应用目录时，具有当前目录下，创建目录的权限
     * 创建根目录时，当前卡文件系统必须为空。
     *
     * @param dir [in]目录属性结构
     * @return {@link JNIAPI#XKR_OK} 成功
     */
    public int CreateDir(XDJA_DIR dir) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.CreateDir(handle, dir);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * 选择文件或目录
     *
     * @param fid [in] 文件或目录id
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int SelectFile(byte[] fid) {
        if (fid == null) {
            return RET_PARAM_ERROR;
        }
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SelectFile(handle, fid);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 获取当前目录剩余容量
     *
     * @param size [in] 剩余容量,单位字节Byte
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int GetDirSize(int[] size) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.GetDirSize(handle, size);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 删除文件
     *
     * @param fid [in] 文件ID，全0表示删除目录下所有文件
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int DeleteFile(byte[] fid) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.DeleteFile(handle, fid);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 创建文件
     * 条件:具有当前目录下，创建文件的权限
     *
     * @param file [in] 文件属性结构
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int CreateFile(XDJA_FILE file) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.CreateFile(handle, file);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 取文件属性
     *
     * @param fid  [in] 文件ID
     * @param file [in] 文件属性结构
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int GetFileInfo(byte[] fid, XDJA_FILE file) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.GetFileInfo(handle, fid, file);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 读文件
     * 条件：具有读该文件的权限
     *
     * @param fid     [in]文件ID
     * @param readPos [in]起始位置
     * @param readLen [in]要读取的长度
     * @param dataOut [out]读取内容缓冲区
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int ReadFile(byte[] fid, int readPos, int readLen, byte[] dataOut) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.ReadFile(handle, fid, readPos, readLen, dataOut);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 写文件
     * 条件：具有写该文件的权限
     *
     * @param fid      [in]文件ID
     * @param writePos [in]起始位置
     * @param writeLen [in]写入内容的长度
     * @param dataIn   [in]写入内容
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int WriteFile(byte[] fid, int writePos, int writeLen, byte[] dataIn) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.WriteFile(handle, fid, writePos, writeLen, dataIn);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * 读RSA公钥
     * 条件：具有读RSA公钥的权限
     *
     * @param pubfid [in] 文件ID
     * @param pubKey [out] RSA公钥
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int ReadRsaPubKey(byte[] pubfid, XDJA_RSA_PUBKEY pubKey) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.ReadRsaPubKey(handle, pubfid, pubKey);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * 写RSA公钥
     * 条件：具有写RSA公钥的权限
     *
     * @param pubfid [in] 文件ID
     * @param pubKey [out] RSA公钥
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int WriteRsaPubKey(byte[] pubfid, XDJA_RSA_PUBKEY pubKey) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.WriteRsaPubKey(handle, pubfid, pubKey);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 写RSA私钥
     * 条件：具有写RSA私钥的权限
     *
     * @param prifid [in] 文件ID
     * @param priKey [out] RSA公钥
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int WriteRsaPriKey(byte[] prifid, XDJA_RSA_PRIKEY priKey) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.WriteRsaPriKey(handle, prifid, priKey);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 读sm2公钥
     * 条件：具有读sm2公钥的权限
     *
     * @param pubfid [in]文件ID
     * @param pubKey [out]sm2公钥
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int ReadSm2PubKey(byte[] pubfid, XDJA_SM2_PUBKEY pubKey) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.ReadSm2PubKey(handle, pubfid, pubKey);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 写SM2公钥
     * 条件：具有写SM2公钥的权限
     *
     * @param pubfid [in] 文件ID
     * @param pubKey [out] SM2公钥
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int WriteSm2PubKey(byte[] pubfid, XDJA_SM2_PUBKEY pubKey) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.WriteSm2PubKey(handle, pubfid, pubKey);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 写SM2私钥
     * 条件：具有写SM2私钥的权限
     *
     * @param prifid [in] 文件ID
     * @param priKey [out] SM2私钥
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int WriteSm2PriKey(byte[] prifid, XDJA_SM2_PRIKEY priKey) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.WriteSm2PriKey(handle, prifid, priKey);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 读证书
     * 条件：具有读证书的权限
     *
     * @param fid     [in]证书文件ID
     * @param certBuf [out]证书信息缓冲区，长度最小为2048
     * @param certLen [out]实际证书信息长度
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int ReadCert(byte[] fid, byte[] certBuf, int[] certLen) {
        if (fid == null || certBuf == null || certBuf.length < 2048) {
            return RET_PARAM_ERROR;
        }
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.ReadCert(handle, fid, certBuf, certLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 写证书
     * 条件：具有写证书的权限
     *
     * @param fid     [in]证书文件ID
     * @param certBuf [in]证书信息
     * @param certLen [in]证书信息长度
     * @return 错误码
     * {@link JNIAPI#XKR_OK }             成功
     * {@link JNIAPI#XKR_NO_POWER}        权限不够
     * {@link JNIAPI#XKR_FILE_NOT_EXIST}  文件不存在
     */
    public int WriteCert(byte[] fid, byte[] certBuf, int certLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.WriteCert(handle, fid, certBuf, certLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 卡内产生随机数
     *
     * @param len    [in]需要获取的随机数长度
     * @param random [out]输出的随机数缓冲区
     * @return 错误码
     * {@link JNIAPI#XKR_OK} 成功
     */
    public int GenRandom(int len, byte[] random) {
        if (random == null) {
            return RET_PARAM_ERROR;
        }
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.GenRandom(handle, len, random);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * DES加解密，卡外送入密钥 (分组长度8)
     *
     * @param tmpKey  [in] 密钥，8字节
     * @param dataIn  [in] 输入数据，数据长度为8整数倍
     * @param inLen   [in] 输入数据长度
     * @param flag    [in] 指示加密、解密与运算模式。
     * @param dataOut [out]输出数据
     * @param IV      [in] 采用EBC模式此参数无效，可置为NULL;CBC模式时为初始向量，8字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }             成功
     */
    public int DESKEY(byte[] tmpKey, byte[] dataIn, int inLen, int flag, byte[] dataOut, byte[] IV) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            //非CBC模式下若IV==null,则为IV赋一个长度为0的数组，不然binder会报错
            if (IV == null) {
                IV = new byte[0];
            }
            return binder.DESKEY(handle, tmpKey, dataIn, inLen, flag, dataOut, IV);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 3DES加解密，卡外送入密钥 (分组长度8)
     *
     * @param tmpKey  [in] 密钥，8字节
     * @param dataIn  [in] 输入数据，数据长度为8整数倍
     * @param inLen   [in] 输入数据长度
     * @param flag    [in] 指示加密、解密与运算模式。
     * @param dataOut [out]输出数据
     * @param IV      [in] 采用EBC模式此参数无效，可置为NULL;CBC模式时为初始向量，8字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }             成功
     */
    public int DES3KEY(byte[] tmpKey, int tmpkeyLen, byte[] dataIn, int inLen, int flag, byte[] dataOut, byte[] IV) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            //非CBC模式下若IV==null,则为IV赋一个长度为0的数组，不然binder会报错
            if (IV == null) {
                IV = new byte[0];
            }
            return binder.DES3KEY(handle, tmpKey, tmpkeyLen, dataIn, inLen, flag, dataOut, IV);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * AES加解密，卡外送入密钥 (分组长度8)
     *
     * @param tmpKey  [in] 密钥，8字节
     * @param dataIn  [in] 输入数据，数据长度为8整数倍
     * @param inLen   [in] 输入数据长度
     * @param flag    [in] 指示加密、解密与运算模式。
     * @param dataOut [out]输出数据
     * @param IV      [in] 采用EBC模式此参数无效，可置为NULL;CBC模式时为初始向量，8字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }             成功
     */
    public int AESKEY(byte[] tmpKey, int keyLen, byte[] dataIn, int inLen, int flag, byte[] dataOut, byte[] IV) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            //非CBC模式下若IV==null,则为IV赋一个长度为0的数组，不然binder会报错
            if (IV == null) {
                IV = new byte[0];
            }
            return binder.AESKEY(handle, tmpKey, keyLen, dataIn, inLen, flag, dataOut, IV);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * 创建SM1密钥
     * 条件:具有当前目录下，增加密钥的权限
     *
     * @param keyAttr [in] SM1密钥属性结构
     * @return 错误码
     * {@link JNIAPI#XKR_OK }             成功
     */
    public int CreateKey(XDJA_KEY_ATTR keyAttr) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.CreateKey(handle, keyAttr);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 导入SM1密钥
     * 条件：具有更新该密钥的权限
     *
     * @param type   [in]导入方式(4bit) 和 导入密钥类型(4bit)。本版本支持明文导入。
     * @param dataIn [in]密钥数据。密钥长度由密钥类型自动决定。在本版本中，SM1密钥为16字节
     * @param kID    [in]密钥ID
     * @return 错误码
     * {@link JNIAPI#XKR_OK }             成功
     */
    public int ImportKey(int type, byte[] dataIn, byte kID) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.ImportKey(handle, type, dataIn, kID);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM1加解密运算，使用卡内密钥
     *
     * @param dataIn  [in]被运算的数据
     * @param inLen   [in]输入数据长度
     * @param flag    [in]指示加密、解密与运算模式。
     * @param dataOut [out]SM1运算结果
     * @param kID     [in]密钥ID
     * @param IV      [in,out]采用ECB模式此参数无效，可置为NULL;CBC模式时为初始向量，16字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }             成功
     */
    public int SM1(byte[] dataIn, int inLen, int flag, byte[] dataOut, byte kID, byte[] IV) {
        if (dataIn == null || dataIn.length < inLen || dataOut == null || dataOut.length < inLen) {
            return RET_PARAM_ERROR;
        }
        //如果是CBC模式，那么IV一定不能为null
        if (flag == JNIAPI.CBC_DECRYPT || flag == JNIAPI.CBC_ENCRYPT || flag == JNIAPI.CBC_MODE) {
            if (IV == null || IV.length != 16) {
                return RET_PARAM_ERROR;
            }
        }
        //非CBC模式下若IV==null,则为IV赋一个长度为0的数组，不然binder会报错
        if (IV == null) {
            IV = new byte[0];
        }

        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SM1(handle, dataIn, inLen, flag, dataOut, kID, IV);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * @param tmpKey [in] SM1密钥 16字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }             成功
     * @ 导入临时SM1密钥
     */
    public int ImportTmpKey(byte[] tmpKey) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.ImportTmpKey(handle, tmpKey);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM1加解密，使用临时密钥
     *
     * @param dataIn  [in] 被运算的数据。
     * @param inLen   [in] 输入数据长度。CBC模式时，含IV长度。
     * @param flag    [in] 指示加密、解密与运算模式。
     * @param dataOut [out]SM1运算结果。
     * @param IV      [in] 输入iv输出iv 在CBC时有效
     * @return 错误码
     * {@link JNIAPI#XKR_OK }             成功
     */
    public int TmpSM1(byte[] dataIn, int inLen, int flag, byte[] dataOut, byte[] IV) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            //非CBC模式下若IV==null,则为IV赋一个长度为0的数组，不然binder会报错
            if (IV == null) {
                IV = new byte[0];
            }
            return binder.TmpSM1(handle, dataIn, inLen, flag, dataOut, IV);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM1加解密，卡外送入密钥
     *
     * @param tmpKey  [in]SM1密钥 16字节
     * @param dataIn  [in]被运算的数据。
     * @param inLen   [in]输入数据长度。CBC模式时，含IV长度。
     * @param flag    [in]指示加密、解密与运算模式。标识符 0x00 ECB解密 0x01 ECB加密 0x10 CBC解密 0x11 CBC加密
     * @param dataOut [out]SM1运算结果。
     * @param IV      [in] 采用ECB模式此参数无效，可置为NULL;CBC模式时为初始向量，16字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }             成功
     */
    public int SM1KEY(byte[] tmpKey, byte[] dataIn, int inLen, int flag, byte[] dataOut, byte[] IV) {
        if (dataIn == null || dataIn.length < inLen || dataOut == null || dataOut.length < inLen) {
            return RET_PARAM_ERROR;
        }
        //如果是CBC模式，那么IV一定不能为null
        if (flag == JNIAPI.CBC_DECRYPT || flag == JNIAPI.CBC_ENCRYPT || flag == JNIAPI.CBC_MODE) {
            if (IV == null || IV.length != 16) {
                return RET_PARAM_ERROR;
            }
        }
        //非CBC模式下若IV==null,则为IV赋一个长度为0的数组，不然binder会报错
        if (IV == null) {
            IV = new byte[0];
        }

        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SM1KEY(handle, tmpKey, dataIn, inLen, flag, dataOut, IV);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SSF33加解密，卡外送入密钥 (分组长度16)
     *
     * @param tmpKey  [in] 密钥，16字节
     * @param dataIn  [in] 输入数据，数据长度为16整数倍
     * @param inLen   [in] 输入数据长度
     * @param flag    [in] 指示加密、解密与运算模式。
     * @param dataOut [out]输出数据
     * @param IV      [in] 采用EBC模式此参数无效，可置为NULL;CBC模式时为初始向量，16字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }             成功
     */
    public int SSF33(byte[] tmpKey, byte[] dataIn, int inLen, int flag, byte[] dataOut, byte[] IV) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            //非CBC模式下若IV==null,则为IV赋一个长度为0的数组，不然binder会报错
            if (IV == null) {
                IV = new byte[0];
            }
            return binder.SSF33(handle, tmpKey, dataIn, inLen, flag, dataOut, IV);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * 卡外SM4加解密，卡外送入密钥 (分组长度16)
     *
     * @param tmpKey  [in]密钥，16字节
     * @param dataIn  [in]输入数据，数据长度为16整数倍
     * @param inLen   [in]输入数据长度
     * @param flag    [in]指示加密、解密与运算模式。
     * @param dataOut [out]输出数据
     * @param IV      [in]采用EBC模式此参数无效，可置为NULL;CBC模式时为初始向量，16字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }             成功
     */
    public int SM4KEY(byte[] tmpKey, byte[] dataIn, int inLen, int flag, byte[] dataOut, byte[] IV) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            //非CBC模式下若IV==null,则为IV赋一个长度为0的数组，不然binder会报错
            if (IV == null) {
                IV = new byte[0];
            }
            return binder.SM4KEY(handle, tmpKey, dataIn, inLen, flag, dataOut, IV);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * 卡内SM4加解密，卡外送入密钥 (分组长度16)
     *
     * @param tmpKey  [in] 密钥，16字节
     * @param dataIn  [in] 输入数据，数据长度为16整数倍
     * @param inLen   [in] 输入数据长度
     * @param flag    [in] 指示加密、解密与运算模式。
     * @param dataOut [out]输出数据
     * @param IV      [in] 采用EBC模式此参数无效，可置为NULL;CBC模式时为初始向量，16字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int SM4KEYEx(byte[] tmpKey, byte[] dataIn, int inLen, int flag, byte[] dataOut, byte[] IV) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            //非CBC模式下若IV==null,则为IV赋一个长度为0的数组，不然binder会报错
            if (IV == null) {
                IV = new byte[0];
            }
            return binder.SM4KEYEx(handle, tmpKey, dataIn, inLen, flag, dataOut, IV);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM6加解密，卡外送入密钥 (分组长度16)
     * （SCB2）
     *
     * @param tmpKey  [in] 密钥，32字节
     * @param dataIn  [in] 输入数据，数据长度为16整数倍
     * @param inLen   [in] 输入数据长度
     * @param flag    [in] 指示加密、解密与运算模式。
     * @param dataOut [out]输出数据
     * @param IV      [in] 采用EBC模式此参数无效，可置为NULL;CBC模式时为初始向量，16字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int SM6KEY(byte[] tmpKey, byte[] dataIn, int inLen, int flag, byte[] dataOut, byte[] IV) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            //非CBC模式下若IV==null,则为IV赋一个长度为0的数组，不然binder会报错
            if (IV == null) {
                IV = new byte[0];
            }
            return binder.SM6KEY(handle, tmpKey, dataIn, inLen, flag, dataOut, IV);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * 卡内产生RSA密钥对
     * 条件：密钥对不出卡时要求有写指定公私钥文件的权限
     *
     * @param bits   [in]RSA公钥模数长度，1024或2048
     * @param pubfid [in]公钥文件ID，为0x00 0x00时表示公钥导出卡外
     * @param prifid [in]私钥文件ID，公私钥文件ID均为0x00 0x00时私钥可导出卡外
     * @param pubkey [in]RSA公钥结构，公钥文件ID为0x00 0x00时有效
     * @param prikey [in]RSA私钥结构，公私钥文件ID均为0x00 0x00时有效
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int GenRSAKeyPair(int bits, byte[] pubfid, byte[] prifid, XDJA_RSA_PUBKEY pubkey, XDJA_RSA_PRIKEY prikey) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.GenRSAKeyPair(handle, bits, pubfid, prifid, pubkey, prikey);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * RSA公钥运算 （RSA签名验证）
     * 传入数据填充或解填充由调用者进行
     *
     * @param pubfid  [in]RSA公钥文件ID,为0x00 0x00时使用外部公钥
     * @param pubkey  [in]RSA公钥结构，公钥文件ID为0x00 0x00时有效并且不能为空
     * @param dataIn  [in]输入数据
     * @param inLen   [in]输入数据长度，RSA1024为128，RSA2048为256(目前只支持RSA1024)
     * @param dataOut [out]输出数据
     * @param outLen  [out]输出结果长度，128或256
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int RSAPubKeyCalc(byte[] pubfid, XDJA_RSA_PUBKEY pubkey, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        if (pubfid == null || dataIn == null || dataIn.length < inLen || dataOut == null || outLen == null) {
            return RET_PARAM_ERROR;
        }

        if (pubfid[0] == 0x00 && pubfid[1] == 0x00) {
            if (pubkey == null) {
                return RET_PARAM_ERROR;
            }
        } else if (pubkey == null) {
            pubkey = new XDJA_RSA_PUBKEY();
        }

        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.RSAPubKeyCalc(handle, pubfid, pubkey, dataIn, inLen, dataOut, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * RSA私钥运算 （RSA签名）
     * 传入数据由调用者进行填充。
     *
     * @param prifid  [in]私钥ID
     * @param dataIn  [in]输入数据
     * @param inLen   [in]输入数据长度，RSA1024为128，RSA2048为256
     * @param dataOut [out]输出数据
     * @param outLen  [out]输出结果长度，128或256
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     * {@link JNIAPI#XKR_NO_POWER}  权限不够
     */
    public int RSAPriKeyCalc(byte[] prifid, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {

        if (prifid == null || dataIn == null || dataIn.length < inLen || dataOut == null || outLen == null) {
            return RET_PARAM_ERROR;
        }

        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.RSAPriKeyCalc(handle, prifid, dataIn, inLen, dataOut, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * RSA私钥运算 （RSA签名）
     * 传入数据由调用者进行填充。
     *
     * @param pin     [in] pin码
     * @param pinLen  [in] 	   pin密码
     * @param pinRole [in]  pin角色
     * @param prifid  [in]  私钥ID
     * @param dataIn  [in]  输入数据
     * @param inLen   [in]  输入数据长度，RSA1024为128，RSA2048为256(暂不支持)
     * @param dataOut [out] 输出数据
     * @param outLen  [out] 输出结果长度，128或256
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     * {@link JNIAPI#XKR_NO_POWER}  权限不够
     */
    public int RSAPriKeyCalcEx(byte[] pin, int pinLen, int pinRole, byte[] prifid, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.RSAPriKeyCalcEx(handle, pin, pinLen, pinRole, prifid, dataIn, inLen, dataOut, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * RSA数据签名
     *
     * @param bits     [in]  RSA公钥模数
     * @param prifid   [in]  私钥ID
     * @param dataType [in]  数数据类型SIGN_DATA_TYPE
     * @param dataIn   [in]  数据
     * @param inLen    [in]  摘要数据数据长度必须为20；输入数据长度
     * @param dataOut  [out] 输出签名数据
     * @param outLen   [out] 输出数据长度 128或256
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     * {@link JNIAPI#XKR_NO_POWER}  权限不够
     */
    public int RSASign(int bits, byte[] prifid, int dataType, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.RSASign(handle, bits, prifid, dataType, dataIn, inLen, dataOut, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * RSA数据签名验证
     *
     * @param bits          [in]   RSA公钥模数
     * @param pubfid        [in]   公钥ID
     * @param pubkey        [in]   签名用公钥，当pubkeyid为0x00 0x00时使用
     * @param dataType      [in]   数据类型SIGN_DATA_TYPE
     * @param dataIn        [in]   输入数据
     * @param inLen         [in]   摘要数据长度必须为20
     * @param signatureData [in]   验签数据
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int RSASignVerify(int bits, byte[] pubfid, XDJA_RSA_PUBKEY pubkey, int dataType, byte[] dataIn, int inLen, byte[] signatureData) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.RSASignVerify(handle, bits, pubfid, pubkey, dataType, dataIn, inLen, signatureData);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 产生信封
     * 产生16字节的随机数作为会话密钥，用P1P2指定的公钥加密，将加密结果送出卡外，同时保存随机数在临时密钥区域。
     *
     * @param pubfid  [in]  公钥文件ID
     * @param pubkey  [in]  外部公钥,当公钥文件ID为全0时有效
     * @param alg     [in]  会话密钥算法,取值1、2、3、4分别表示SM1、DES、3DES、SM4
     * @param dataOut [out] 信封数据
     * @param outLen  [out] 信封数据长度
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int PackEnvelope(byte[] pubfid, XDJA_RSA_PUBKEY pubkey, int alg, byte[] dataOut, int[] outLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.PackEnvelope(handle, pubfid, pubkey, alg, dataOut, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 拆信封
     * 将送入的数据用P1P2指定的私钥进行解密，将解密后的结果保存在临时区域。
     *
     * @param prifid [in]   私钥文件ID
     * @param alg    [in]   会话密钥算法,取值1、2、3、4分别表示SM1、DES、3DES、SM4
     * @param dataIn [in]   信封数据
     * @param inLen  [in]   信封数据长度
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int UnpackEnvelope(byte[] prifid, int alg, byte[] dataIn, int inLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.UnpackEnvelope(handle, prifid, alg, dataIn, inLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM1密钥分散
     * 密钥分散只能对SM1加密密钥或者SM1解密密钥进行。其功能是将卡外送入的密钥因子用指定的密钥进行加密，并将加密结果存入临时区域。
     *
     * @param keyId       [in] 用于密钥分散的主密钥、SM1加密秘钥、解密秘钥
     * @param keyParam    [in] 分散因子
     * @param keyParamLen [in] 分散因子长度,不能超过16
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int KeyDisperse(int keyId, byte[] keyParam, int keyParamLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.KeyDisperse(handle, keyId, keyParam, keyParamLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 设置SM2算法身份标识
     *
     * @param sm2Id [in] 身份标识
     * @param inLen [in] 身份标识长度
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int SetSM2Id(byte[] sm2Id, int inLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SetSM2Id(handle, sm2Id, inLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * 获取SM2算法身份标识
     *
     * @param sm2Id  [out]  身份标识
     * @param outLen [out]  身份标识长度
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int GetSM2Id(byte[] sm2Id, int[] outLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.GetSM2Id(handle, sm2Id, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 设置SM2算法参数
     *
     * @param param [in]   参数结构体指针
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int SetSM2Param(XDJA_SM2_PARAM param) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SetSM2Param(handle, param);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * 取得SM2算法参数
     *
     * @param param [out] 参数结构体指针
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int GetSM2Param(XDJA_SM2_PARAM param) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.GetSM2Param(handle, param);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * 生成SM2密钥对
     *
     * @param pubfid [in]   公钥文件ID,为0x00 0x00时表示公钥导出卡外
     * @param prifid [in]   私钥文件ID,公私钥ID均为0x00 0x00时表示私钥导出卡外
     * @param pubkey [out]  SM2公钥结构，公钥文件ID为0x00 0x00时有效并且不能为空
     * @param prikey [out]  SM2私钥结构，公私钥文件ID均为0x00 0x00时有效并且不能为空
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int GenSM2KeyPair(byte[] pubfid, byte[] prifid, XDJA_SM2_PUBKEY pubkey, XDJA_SM2_PRIKEY prikey) {
        if (pubfid[0] == 0x00 && pubfid[1] == 0x00) {
            if (pubkey == null) {
                return RET_PARAM_ERROR;
            }
        } else if (pubkey == null) {
            pubkey = new XDJA_SM2_PUBKEY();
        }

        if (prifid[0] == 0x00 && prifid[1] == 0x00) {
            if (prikey == null) {
                return RET_PARAM_ERROR;
            }
        } else if (prikey == null) {
            prikey = new XDJA_SM2_PRIKEY();
        }

        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.GenSM2KeyPair(handle, pubfid, prifid, pubkey, prikey);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }


    /**
     * SM2公钥加密（国密标准）
     * 国密密文结构：  x(32B) | y(32B) | 明文HASH(32B) | 密文长度(4B) | 密文
     *
     * @param pubfid  [in]      SM2公钥ID，两字节，为0x00 0x00表示公钥随数据传入
     * @param pubkey  [in]      公钥,当pubkeyid为0x00 0x00时有效
     * @param dataIn  [in]      明文数据,最大长度不超过155
     * @param inLen   [in]      数据长度
     * @param dataOut [out]     加密后密文,缓冲长度至少为inLen+97
     * @param outLen  [out]     加密后数据长度
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int SM2Encrypt(byte[] pubfid, XDJA_SM2_PUBKEY pubkey, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        if (pubfid == null || dataIn == null || dataIn.length < inLen || inLen > 155 || dataOut == null
                || dataOut.length < inLen + 97 || outLen == null) {
            return RET_PARAM_ERROR;
        }

        if (pubfid[0] == 0x00 && pubfid[1] == 0x00) {
            if (pubkey == null) {
                return RET_PARAM_ERROR;
            }
        } else if (pubkey == null) {
            pubkey = new XDJA_SM2_PUBKEY();
        }


        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SM2Encrypt(handle, pubfid, pubkey, dataIn, inLen, dataOut, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM2私钥解密（国密标准）
     *
     * @param pubfid  [in]    SM2私钥ID 两字节，第二个字节有效，为0x00 0x00表示私钥随数据传入
     * @param pubkey  [in]    密文数据,最大长度不超过255
     * @param inLen   [in]    数据长度
     * @param dataOut [out]   解密后的明文数据,缓冲区至少为dlen-100
     * @param outLen  [out]   解密后数据长度
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     * {@link JNIAPI#XKR_NO_POWER}  权限不够
     */
    public int SM2EncryptGM(byte[] pubfid, XDJA_SM2_PUBKEY pubkey, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SM2EncryptGM(handle, pubfid, pubkey, dataIn, inLen, dataOut, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM2私钥解密
     *
     * @param prifid  [in]    SM2私钥ID 两字节
     * @param dataIn  [in]    密文数据,最大长度不超过SM2_BLOCK_MAX+97
     * @param inLen   [in]    数据长度
     * @param dataOut [out]   解密后的明文数据,缓冲区至少为dlen-97
     * @param outLen  [out]   解密后数据长度
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     * {@link JNIAPI#XKR_NO_POWER}  权限不够
     */
    public int SM2Decrypt(byte[] prifid, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SM2Decrypt(handle, prifid, dataIn, inLen, dataOut, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM2私钥解密（国密标准）
     *
     * @param prifid  [in]     SM2私钥ID 两字节，第二个字节有效，为0x00 0x00表示私钥随数据传入
     * @param dataIn  [in]     密文数据,最大长度不超过255
     * @param inLen   [in]     数据长度
     * @param dataOut [out]    解密后的明文数据,缓冲区至少为dlen-100
     * @param outLen  [out]    解密后数据长度
     * @return 错误码
     * <p>
     * {@link JNIAPI#XKR_OK }   成功
     * {@link JNIAPI#XKR_NO_POWER}  权限不够
     */
    public int SM2DecryptGM(byte[] prifid, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SM2DecryptGM(handle, prifid, dataIn, inLen, dataOut, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM2数据签名
     *
     * @param pubfid   [in]   公钥ID ,当datatype=1 {@link com.xdja.SafeKey.JNIAPI#SIGN_NOHASH}时,pubfid有效
     * @param prifid   [in]   私钥ID
     * @param dataType [in]   数据类型SIGN_DATA_TYPE,具体意义参见{@link com.xdja.SafeKey.JNIAPI#SIGN_HASH}
     *                 {@link com.xdja.SafeKey.JNIAPI#SIGN_NOHASH}。<br>
     * @param dataIn   [in]   数据
     * @param inLen    [in]   摘要数据数据长度必须为32
     * @param dataOut  [out]  输出签名数据 缓冲区长度必须大于64字节
     * @param outLen   [out]  输出数据长度 当pdataout为NULL，此值必须为0
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int SM2Sign(byte[] pubfid, byte[] prifid, int dataType, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen) {
        if (dataType == JNIAPI.SIGN_NOHASH && pubfid == null) {
            return RET_PARAM_ERROR;
        } else if (pubfid == null) {
            pubfid = new byte[0];
        }

        if (prifid == null || dataIn == null || dataIn.length < inLen || dataOut == null ||
                dataOut.length < 64 || outLen == null) {
            return RET_PARAM_ERROR;
        }
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SM2Sign(handle, pubfid, prifid, dataType, dataIn, inLen, dataOut, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM2数据签名验证
     *
     * @param pubfid        [in]  公钥ID
     * @param dataType      [in]  数据类型SIGN_DATA_TYPE,数据类型SIGN_DATA_TYPE,具体意义参见{@link com.xdja.SafeKey.JNIAPI#SIGN_HASH}
     *                      {@link com.xdja.SafeKey.JNIAPI#SIGN_NOHASH}。<br>
     * @param pubkey        [in]  签名用公钥，pubfid为0x00 0x00时有效并且不能为空
     * @param dataIn        [in]  数据
     * @param inLen         [in]  摘要数据长度必须为32
     * @param signatureData [in]  验签数据
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     * {@link JNIAPI#XKR_NO_POWER}  权限不够
     */
    public int SM2SignVerify(byte[] pubfid, int dataType, XDJA_SM2_PUBKEY pubkey, byte[] dataIn, int inLen, byte[] signatureData) {
        if (pubfid == null || dataIn == null || dataIn.length < inLen || signatureData == null) {
            return RET_PARAM_ERROR;
        }
        if (pubfid[0] == 0x00 && pubfid[1] == 0x00) {
            if (pubkey == null) {
                return RET_PARAM_ERROR;
            }
        } else if (pubkey == null) {
            pubkey = new XDJA_SM2_PUBKEY();
        }

        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SM2SignVerify(handle, pubfid, dataType, pubkey, dataIn, inLen, signatureData);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SHA1运算
     *
     * @param dataIn  [in] 输入数据
     * @param inLen   [in]   输入数据长度
     * @param dataOut [out] 输出运算结果 20个字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int SHA1(byte[] dataIn, int inLen, byte[] dataOut) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SHA1(handle, dataIn, inLen, dataOut);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM3运算
     *
     * @param dataIn  [in]   输入数据
     * @param inLen   [in]   输入数据长度
     * @param dataOut [out]  输出运算结果 32字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int SM3(byte[] dataIn, int inLen, byte[] dataOut) {
        if (dataIn == null || dataIn.length < inLen || dataOut == null || dataOut.length < 32) {
            return RET_PARAM_ERROR;
        }
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SM3(handle, dataIn, inLen, dataOut);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM2协商密钥初始化
     *
     * @param pubfid  [in]     SM2公钥ID 两字节,第二个字节有效,第一字节必须为0x00
     * @param dataOut [out]    产生协商数据
     * @param outLen  [out]    输出数据长度
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int SM2KeyGenInit(byte[] pubfid, byte[] dataOut, int[] outLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SM2KeyGenInit(handle, pubfid, dataOut, outLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM2协商密钥 计算步骤
     *
     * @param pubfid   [in]      SM2公钥ID 两字节,第一节字必须为0x00,第二个字节有效 当dictflag=SM2_KEY_GENERATE_DICT_SEND 时 公钥ID为0x000x00
     * @param prifid   [in]      SM2私钥ID 两字节,第一节字必须为0x00,第二个字节有效
     * @param dataIn   [in]      输入数据
     *                 当dictflag=SM2_KEY_GENERATE_DICT_SEND 时，输入数据为 响应方的ID（TLV格式）||响应方公钥x坐标（TLV格式）||响应方公钥y坐标（TLV格式）||响应方临时公钥x坐标（TLV格式）||响应方临时公钥y坐标(TLV格式)
     *                 当dictflag=SM2_KEY_GENERATE_DICT_RECV 时，输入数据为 发起方的ID（TLV格式）||发起方公钥x坐标（TLV格式）||发起方公钥y坐标（TLV格式）||发起方临时公钥x坐标（TLV格式）||发起方临时公钥y坐标(TLV格式)
     * @param inLen    [in]      输入数据长度
     * @param dataOut  [out]     产生协商数据
     * @param outLen   [out]     输出数据长度
     * @param dictFlag [in]      发起对象,SM2_KEY_GENERATE_DICT_SEND为发起方,SM2_KEY_GENERATE_DICT_RECV为响应方]
     * @param keyFlag  [in]      密钥存储标记 1固定位置,0临时位置
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int SM2KeyGenCompute(byte[] pubfid, byte[] prifid, byte[] dataIn, int inLen, byte[] dataOut, int[] outLen, byte dictFlag, byte keyFlag) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SM2KeyGenCompute(handle, pubfid, prifid, dataIn, inLen, dataOut, outLen, dictFlag, keyFlag);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * SM2协商密钥验证
     *
     * @param dataIn [in]      输入数据
     * @param inLen  [in]      输入数据长度 固定32字节
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     */
    public int SM2KeyGenVerify(byte[] dataIn, int inLen) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.SM2KeyGenVerify(handle, dataIn, inLen);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    /**
     * 在线通过url激活卡  <br>
     * 注意：<br>
     * 1. apk使用此接口时，需要申请联网权限
     * uses-permission android:name="android.permission.INTERNET"<br>
     * 2. apk使用此接口时，不要在UI主线程使用，可能会造成卡死或崩溃问题<br>
     * 3. apk使用此接口时，在接口内部会锁定卡所有卡操作，直到此过程结束，
     * 在此期间，调用操作卡的接口都会等待。
     *
     * @param url [in]激活的地址，比如： https://11.12.110.205:8099/scas-web/api
     * @return 错误码
     * {@link JNIAPI#XKR_OK }   成功
     * 其他：表示错误，请参见其他错误码定义
     */
    public int ActivateCardByURL(String url) {
        try {
            if (getBinder() == null) {
                return JNIAPI.XKR_NO_KEY;
            }
            return binder.ActivateCardByURL(handle, url);
        } catch (Exception e) {
            e.printStackTrace();
            return handleException(e);
        }
    }

    public Bundle callMethod(Bundle bundle) {
        try {
            if (getBinder() == null) {
                Bundle result = new Bundle();
                result.putInt("ret", JNIAPI.XKR_NO_KEY);
                return result;
            }
            return binder.callMethod(bundle);
        } catch (Exception e) {
            e.printStackTrace();
            Bundle result = new Bundle();
            result.putInt("ret", RET_EXCEPTION);
            return result;
        }
    }

    private int handleException(Exception e) {
        e.printStackTrace();
        return RET_EXCEPTION;
    }

    protected IMultiJniApi getBinder() {
        if (binder == null) {
            Pair<Integer, JarJniApiProxy> pair = JarMultiJniApiManager.getInstance().make(context, cardId);
            if (pair != null && pair.first == 0) {
                this.binder = pair.second.binder;
                binderDeathLink();
            }
        }
        return binder;
    }

    /**
     * binder死亡监听
     */
    private void binderDeathLink() {
        try {
            binder.asBinder().linkToDeath(new IBinder.DeathRecipient() {
                @Override
                public void binderDied() {
                    binder = null;
                }
            }, Binder.FLAG_ONEWAY);
            Messenger messenger = new Messenger(handler);
            Bundle bundle = new Bundle();
            bundle.putString("method", "SetCardChangeListener");
            bundle.putBinder("Binder", messenger.getBinder());
            binder.callMethod(bundle);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    Handler handler = new Handler(Looper.getMainLooper()) {
        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);
            if (msg.what == 1) {
                cardId = new String(((XDJA_DEVINFO) msg.obj).cardid);
            }
        }
    };
}
