package com.xdja.multichip.jniapi;

import android.os.Bundle;
import android.util.Pair;

import com.xdja.SafeKey.JNIAPI;
import com.xdja.multichip.param.CertBean;
import com.xdja.multichip.param.ParamKeywords;

import java.io.ByteArrayInputStream;
import java.security.PublicKey;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Arrays;

import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.GET_SAFE_CARD_SN_ERROR;
import static com.xdja.multichip.jniapi.JarMultiJniApiErrorCode.RET_PARAM_ERROR;

/**
 * @author: zhangxiaolong@xdja.com <br/>
 * date:   2017/11/9 <br/>
 */

public class JarJniApiProxyExtra {

    /**
     * SM2签名
     *
     * @param proxy     卡操作代理
     * @param role      pin码角色
     * @param pin       pin码
     * @param containNo 容器编号[0-9]
     * @param certType  证书类型[1加密证书, 2签名证书]
     * @param dataType  要签名的数据类型，
     *                  参看：{@link JNIAPI#SIGN_HASH}：表示数据已做hash,
     *                  {@link JNIAPI#SIGN_NOHASH}： 表示数据未做hash
     * @param dataIn    要签名的数据
     * @param inLen     要签名数据的长度
     * @param dataOut   [out]签名后的数据
     * @param outLen    [in,out]签名后数据的长度，数据的长度在outLen[0]中
     * @return 0：成功，其他：失败
     */
    public static int SM2Sign(JarJniApiProxy proxy,
                              int role,
                              String pin,
                              int containNo,
                              int certType,
                              int dataType,
                              byte[] dataIn,
                              int inLen,
                              byte[] dataOut,
                              int[] outLen) {
        if (proxy == null) {
            throw new NullPointerException("Proxy is null!");
        }
        byte[] pubfid = new byte[2];
        byte[] prifid = new byte[2];

        if (certType == 1) {
            certType = Arithmetic.EXC_CERT;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_CERT;
        } else {
            throw new ArithmeticException("certType is error!");
        }

        byte[] certId = Arithmetic.convertToFid(containNo, certType);
        pubfid[0] = 0x00;
        pubfid[1] = (byte) (certId[1] + 1);

        prifid[0] = 0x00;
        prifid[1] = (byte) (certId[1] + 2);

        int ret;
        ret = proxy.SM2Sign(pubfid, prifid, dataType, dataIn, inLen, dataOut, outLen);
        if (ret == JNIAPI.XKR_NO_POWER) {
            ret = proxy.VerifyPIN(role, pin.getBytes(), pin.length());
            if (ret == JNIAPI.XKR_OK) {
                ret = proxy.SM2Sign(pubfid, prifid, dataType, dataIn, inLen, dataOut, outLen);
            }
        }
        return ret;
    }

    /**
     * 获取证书SN
     *
     * @param proxy     卡操作代理
     * @param containNo 容器编号[0-9]
     * @param certType  证书类型[1加密证书, 2签名证书]
     * @return 其中：Pair.first 是错误码
     * Pair.second 返回安全卡硬件ID号的对象<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public Pair<Integer, String> getSn(JarJniApiProxy proxy, int containNo, int certType) {
        int ret;
        if (certType == 1) {
            certType = Arithmetic.EXC_CERT;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_CERT;
        } else {
            return Pair.create(RET_PARAM_ERROR, "");
        }
        byte[] certId = Arithmetic.convertToFid(containNo, certType);

        byte[] certBuf = new byte[2 * 1024];
        int[] certLen = new int[2];
        ret = proxy.ReadCert(certId, certBuf, certLen);
        if (ret != JNIAPI.XKR_OK) {
            return Pair.create(ret, "");
        }

        X509Certificate x509Cert = Arithmetic.getX509Certificate(certBuf, certLen[0]);
        if (x509Cert != null) {
            byte[] certSn = x509Cert.getSerialNumber().toByteArray();
            System.arraycopy(certSn, 0, certBuf, 0, certSn.length);
            certLen[0] = certSn.length;
        } else {
            return Pair.create(GET_SAFE_CARD_SN_ERROR, "");
        }

        String stringCertSn = "";
        for (int i = 0; i < certLen[0]; i++) {
            stringCertSn += String.format("%02x", certBuf[i]);
        }
        return Pair.create(ret, stringCertSn);
    }

    /**
     * 得到公钥证书中的公钥的byte值
     *
     * @param proxy     卡操作代理
     * @param containNo 容器编号[0-9]
     * @param certType  证书类型[1加密证书, 2签名证书]
     * @return 其中：Pair.first 是错误码
     * Pair.second 公钥byte值<br/>
     * <p>
     * 只有当fist为0的时候,second才有意义。
     */
    public static Pair<Integer, byte[]> getPubKey(JarJniApiProxy proxy, int containNo, int certType) {
        int ret;
        if (certType == 1) {
            certType = Arithmetic.EXC_CERT;
        } else if (certType == 2) {
            certType = Arithmetic.SIGN_CERT;
        } else {
            return Pair.create(RET_PARAM_ERROR, null);
        }
        byte[] certId = Arithmetic.convertToFid(containNo, certType);

        byte[] certBuf = new byte[2 * 1024];
        int[] certLen = new int[1];
        ret = proxy.ReadCert(certId, certBuf, certLen);
        if (ret != JNIAPI.XKR_OK) {
            return Pair.create(ret, null);
        }

        byte[] pubKeyBytes = getPubKey(certBuf, certLen[0]);
        if (pubKeyBytes == null) {
            return Pair.create(-1, null);
        } else {
            return Pair.create(0, pubKeyBytes);
        }
    }

    /**
     * 导入证书
     *
     * @return
     */
    public static Pair<Integer, String> importCert(JarJniApiProxy proxy, CertBean bean) {
        Bundle bundle = new Bundle();
        bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_importCert);
        bundle.putAll(bean.getBundle());
        Bundle result = proxy.callMethod(bundle);
        if (result != null) {
            int ret = result.getInt(CertBean.KEY_RET_INT);
            String info = result.getString(CertBean.KEY_INFO_STRING);
            return Pair.create(ret, info);
        }
        return Pair.create(JarMultiJniApiErrorCode.RET_EXCEPTION, "");
    }

    /**
     * 创建容器
     *
     * @param proxy         proxy
     * @param containerId   容器号，只能是0到9
     * @param containerName 容器名，可以为空
     * @return
     */
    public static int createContainer(JarJniApiProxy proxy,
                                      int role,
                                      String pin,
                                      int containerId,
                                      String containerName) {
        try {
            Bundle bundle = new Bundle();
            bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_createContainer);
            bundle.putInt(ParamKeywords.KEY_int_role, role);
            bundle.putString(ParamKeywords.KEY_String_pin, pin);
            bundle.putInt(ParamKeywords.KEY_int_containerId, containerId);
            bundle.putString(ParamKeywords.KEY_containerName, containerName);
            Bundle result = proxy.callMethod(bundle);
            if (result != null) {
                int ret = result.getInt(ParamKeywords.KEY_int_ret);
                return ret;
            }
            return JarMultiJniApiErrorCode.RET_EXCEPTION;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return JarMultiJniApiErrorCode.RET_EXCEPTION;
    }

    /**
     * 在容器中产生SM2签名公私钥对
     *
     * @param proxy       proxy
     * @param containerId 容器号
     * @return
     */
    public static int genSignSM2KeyPair(JarJniApiProxy proxy,
                                        int containerId) {
        int ret;

        try {
            Bundle bundle = new Bundle();
            bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_genSM2KeyPair);
            bundle.putInt(ParamKeywords.KEY_int_containerId, containerId);

            Bundle result = proxy.callMethod(bundle);
            if (result == null) {
                return JarMultiJniApiErrorCode.RET_EXCEPTION;
            }
            ret = result.getInt(ParamKeywords.KEY_int_ret);
            return ret;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return JarMultiJniApiErrorCode.RET_EXCEPTION;
    }

        /**
     * 在容器中产生SM2签名公私钥对
     *
     * @param proxy       proxy
     * @param containerId 容器号
     * @return
     */
    public static int genExchangeSM2KeyPair(JarJniApiProxy proxy,
                                        int containerId) {
        int ret;

        try {
            Bundle bundle = new Bundle();
            bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_genExchangeSM2KeyPair);
            bundle.putInt(ParamKeywords.KEY_int_containerId, containerId);

            Bundle result = proxy.callMethod(bundle);
            if (result == null) {
                return JarMultiJniApiErrorCode.RET_EXCEPTION;
            }
            ret = result.getInt(ParamKeywords.KEY_int_ret);
            return ret;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return JarMultiJniApiErrorCode.RET_EXCEPTION;
    }


    /**
     * 在容器中产生RSA签名公私钥
     *
     * @param proxy
     * @param containerId
     * @return
     */
    public static int genSignRSAKeyPair(JarJniApiProxy proxy,
                                        int containerId,
                                        int bits) {
        int ret;
        try {
            Bundle bundle = new Bundle();
            bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_genRSAKeyPair);
            bundle.putInt(ParamKeywords.KEY_int_containerId, containerId);
            bundle.putInt(ParamKeywords.KEY_int_bits, bits);
            Bundle result = proxy.callMethod(bundle);
            if (result == null) {
                return JarMultiJniApiErrorCode.RET_EXCEPTION;
            }
            ret = result.getInt(ParamKeywords.KEY_int_ret);
            return ret;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return JarMultiJniApiErrorCode.RET_EXCEPTION;
    }

    /**
     * 在容器中产生RSA签名公私钥
     *
     * @param proxy
     * @param containerId
     * @return
     */
    public static int genExchangeRSAKeyPair(JarJniApiProxy proxy,
                                        int containerId,
                                        int bits) {
        int ret;
        try {
            Bundle bundle = new Bundle();
            bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_genExchangeRSAKeyPair);
            bundle.putInt(ParamKeywords.KEY_int_containerId, containerId);
            bundle.putInt(ParamKeywords.KEY_int_bits, bits);
            Bundle result = proxy.callMethod(bundle);
            if (result == null) {
                return JarMultiJniApiErrorCode.RET_EXCEPTION;
            }
            ret = result.getInt(ParamKeywords.KEY_int_ret);
            return ret;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return JarMultiJniApiErrorCode.RET_EXCEPTION;
    }


    /**
     * 清空容器内容
     * 目前只支持容器 0，并且角色只能为0x01
     *
     * @param proxy
     * @param role        容器对应的角色
     * @param pin         role对应的pin码
     * @param containerId 容器号
     * @return
     */
    public static int clearContainer(JarJniApiProxy proxy,
                                     int role,
                                     String pin,
                                     int containerId) {
        int ret;
        try {
            Bundle bundle = new Bundle();
            bundle.putString(ParamKeywords.KEY_String_method, ParamKeywords.KEY_METHOD_clearContainer);
            bundle.putInt(ParamKeywords.KEY_int_containerId, containerId);
            bundle.putInt(ParamKeywords.KEY_int_role, role);
            bundle.putString(ParamKeywords.KEY_String_pin, pin);
            Bundle result = proxy.callMethod(bundle);
            if (result == null) {
                return JarMultiJniApiErrorCode.RET_EXCEPTION;
            }
            ret = result.getInt(ParamKeywords.KEY_int_ret);
            return ret;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return JarMultiJniApiErrorCode.RET_EXCEPTION;
    }

    /**
     * 内部使用接口
     *
     * @param certByte
     * @param certLen
     * @return
     */
    private static byte[] getPubKey(byte[] certByte, int certLen) {
        byte[] pubKeyBytes = null;
        try {
            CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
            X509Certificate cert = (X509Certificate) certFactory.generateCertificate(
                    new ByteArrayInputStream(certByte, 0, certLen));
            try {
                PublicKey publicKey = cert.getPublicKey();
                String alg = publicKey.getAlgorithm();
                byte[] encoded = publicKey.getEncoded();
                if ("RSA".equals(alg)) {
                    // RSA的公钥 0 到 21 是文件头，从22开始是； RSA
                    pubKeyBytes = Arrays.copyOfRange(encoded, 22, encoded.length);
                } else if ("1.2.840.10045.2.1".equals(alg)) {
                    // SM2的公钥 0 到 25 是文件头，从26开始是;  1.2.840.10045.2.1
                    pubKeyBytes = Arrays.copyOfRange(encoded, 26, encoded.length);
                } else {
                    pubKeyBytes = null;
                }
            } catch (Exception e) {
                e.printStackTrace();
                //如果解析异常，则通过查找证书oid方式查找公钥信息
                int start;
                //rsa证书oid
                byte[] rsaHeader = {48, -127, -97, 48, 13, 6, 9, 42, -122, 72, -122, -9, 13, 1,
                        1, 1, 5, 0, 3, -127, -115, 0};
                start = indexOf(certByte, rsaHeader, 0);
                if (start > 0) {
                    start = start + rsaHeader.length;
                    pubKeyBytes = Arrays.copyOfRange(certByte, start, start + 140);
                } else {
                    //sm2证书oid
                    byte[] sm2Header = {48, 89, 48, 19, 6, 7, 42, -122, 72, -50, 61, 2, 1, 6, 8,
                            42, -127, 28, -49, 85, 1, -126, 45, 3, 66, 0};
                    start = indexOf(certByte, sm2Header, 0);
                    if (start > 0) {
                        start = start + sm2Header.length;
                        pubKeyBytes = Arrays.copyOfRange(certByte, start, start + 65);
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return pubKeyBytes;
    }

    /**
     * 从 origin 中 检索有没有 search
     *
     * @param origin 长数组
     * @param search 要检索的数组
     * @param start  从长数组中检索的位置
     * @return 大于0表示检索的有
     */
    private static int indexOf(byte[] origin, byte[] search, int start) {
        if (origin == null || origin.length == 0 || search == null || search.length == 0) {
            return -10;
        }
        int originLen = origin.length;
        int searchLen = search.length;
        if (originLen < searchLen) {
            return -1;
        }
        if ((start + searchLen) > originLen) {
            return -1;
        }
        int searchStart;
        searchStart = start;
        int result = -1;

        for (int i = searchStart; i <= (originLen - searchLen); i++) {
            int j = 0;
            for (; j < searchLen; j++) {
                if (origin[i + j] == search[j]) {
                    continue;
                } else {
                    break;
                }
            }
            if (j == searchLen) {
                result = i;
                break;
            }
        }
        return result;
    }
}
