package com.xdja.skfapi;

/**
 * author: zhangxiaolong@xdja.com <br/>
 * date:   2017/3/27 <br/>
 */

/**
 * 智能密码钥匙密码应用接口（国密标准GM/T 0016-2012）<p>
 * XDJA Java版 1.0.0<p>
 */

public class SkfApi {

    static {
        try {
            System.loadLibrary("skfproxy");
        } catch (Throwable t) {
        }
    }

    /**
     * 等待设备插拔事件<br>
     * 功能描述：该函数等待设备插入或者拔除事件。szDevName 返回发生事件的设备名称。
     *
     * @param szDevName     [OUT]发生事件的设备名称。
     * @param pulDevNameLen [IN,OUT]输入/输出参数，当输入时表示缓冲区长度，输出时表示设备名称的有效长度,长度包含字符串结束符。
     * @param pulEvent      [OUT]事件类型。1 表示插入，2 表示拔出。
     * @return SAR_OK:  成功。
     * 其他：   错误码。
     */
    public native int WaitForDevEvent(byte[] szDevName, int[] pulDevNameLen, int[] pulEvent);

    /**
     * @brief 取消等待设备插拔事件
     * @note 功能描述：该函数取消等待设备插入或者拔除事件。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int CancelWaitForDevEvent();

    /**
     * @brief 枚举设备
     * @note 功能描述：获得当前系统中的设备列表。
     * @param[IN] bPresent     为TRUE表示取当前设备状态为存在的设备列表。为FALSE表示取当前驱动支持的设备列表。
     * @param[OUT] szNameList   设备列表名称。如果参数为NULL，将由pulSize返回所需要的空间大小。每个设备的名称以单个'\0'结束，以双'\0'表示列表结束。
     * @param[IN,OUT] pulSize      输入时表示设备名称列表的缓冲区长度,输出时表示szNameList所占用的空间大小。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int EnumDev(int bPresent, byte[] szNameList, int[] pulSize);

    /**
     * @brief 连接设备
     * @note 功能描述：通过设备名连接设备，返回设备的句柄。
     * @param[IN] szName      设备名称。
     * @param[OUT] phDev       返回设备操作句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ConnectDev(byte[] szName, long[] phDdev);

    /**
     * @brief 断开连接
     * @note 功能描述：断开一个已经连接的设备，病释放句柄。
     * @param[IN] hDev       连接设备时返回的设备句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 备注：如果该设备已被锁定，函数应首先解锁该设备。断开连接操作并不影响设备的权限状态。
     */
    public native int DisConnectDev(long hDev);

    /**
     * @brief 获取设备状态
     * @note 功能描述：获取设备是否存在的状态。
     * @param[IN] szDevName      设备名称。
     * @param[OUT] pulDevState    返回设备状态。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int GetDevState(byte[] szDevName, int[] pulDevState);

    /**
     * @brief 设置设备标签
     * @note 功能描述：设置设备标签。
     * @param[IN] hDdev       连接设备时返回的设备句柄。
     * @param[IN] szLabel     设备标签字符串。该字符串应小于 32 字节。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int SetLabel(long hDev, byte[] szLabel);

    /**
     * @brief 获取设备信息
     * @note 功能描述：获取设备的一些特征信息，包括设备标签、厂商信息、支持的算法等。
     * @param[IN] hDdev       连接设备时返回的设备句柄。
     * @param[OUT] pDevInfo    返回的设备信息。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int GetDevInfo(long hDev, DevInfo pDevInfo);

    /**
     * @brief 锁定设备
     * @note 功能描述：获得设备的独占使用权。
     * @param[IN] hDdev       连接设备时返回的设备句柄。
     * @param[IN] pDevInfo    超时时间，单位为毫秒。如果为 0xFFFFFFFF 表示无限等待。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int LockDev(long hDev, int ulTimeOut);

    /**
     * @brief 解锁设备
     * @note 功能描述：释放对设备的独占使用权。
     * @param[IN] hDdev       连接设备时返回的设备句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int UnlockDev(long hDev);

    /**
     * @brief 设备命令传输
     * @note 将命令直接发送给设备，并返回结果。
     * @param[IN] hDdev         设备句柄。
     * @param[IN] pbCommand     设备命令。
     * @param[IN] ulCommandLen  命令长度。
     * @param[OUT] pbData        返回结果数据。
     * @param[IN,OUT] pulDataLen    输入时表示结果数据缓冲区长度，输出时表示结果数据实际长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int Transmit(long hDev, byte[] pbCommand, int ulCommandLen, byte[] pbData, int[] pulDataLen);

///访问控制系列函数

    /**
     * @brief 修改设备认证密钥
     * @param[IN] hDdev       连接设备时返回的设备句柄。
     * @param[IN] pbKeyValue  密钥值。
     * @param[IN] ulKeyLen    密钥长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：设备认证成功后才能使用。
     */
    public native int ChangeDevAuthKey(long hDev, byte[] pbKeyValue, int ulKeyLen);

    /**
     * @brief 设备认证
     * @note 认证过程：被认证方获取8字节随机数RND,填充0至16字节得到D0,加密得到结果D1,设备收到D1验证是否正确。
     * @param[IN] hDdev       连接设备时返回的设备句柄。
     * @param[IN] pbAuthData  认证数据。
     * @param[IN] ulLen       认证数据长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int DevAuth(long hDev, byte[] pbAuthData, int ulLen);

    /**
     * @brief 修改PIN
     * @note 功能描述：调用该函数可以修改Administrator PIN和User PIN。
     * 如果原PIN码错误导致验证失败,该函数辉返回相应PIN的剩余重试次数，当剩余次数为0时，表示PIN已经锁死。
     * @param[IN] hApplication     应用句柄。
     * @param[IN] ulPINType       PIN类型, 可为ADMIN_TYPE或USER_TYPE。
     * @param[IN] szOldPin        原PIN值。
     * @param[IN] szNewPin        原PIN值。
     * @param[IN] pulRetryCount   出错后的重试次数。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ChangePIN(long hApplication, int ulPINType, byte[] szOldPin, byte[] szNewPin, int[] pulRetryCount);

    /**
     * @brief 获取PIN信息
     * @note 功能描述：包括最大重试次数、当前剩余重试次数，以及当前PIN是否为出厂默认PIN码。
     * @param[IN] hApplication          应用句柄。
     * @param[IN] ulPINType            PIN类型, 可为ADMIN_TYPE或USER_TYPE。
     * @param[OUT] pulMaxRetryCount     最大重试次数。
     * @param[OUT] pulRemainRetryCount  当前剩余重试次数，当为0时表示已锁死。
     * @param[OUT] pbDefaultPin         是否为出厂默认PIN码。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int GetPINInfo(long hApplication, int ulPINType, int[] pulMaxRetryCount, int[] pulRemainRetryCount, int[] pbDefaultPin);

    /**
     * @brief 校验PIN
     * @note 功能描述：校验成功后，会获得相应的权限，如果PIN码错误，会返回PIN码重试次数，当次数为0时表示PIN码已锁死。
     * @param[IN] hApplication     应用句柄。
     * @param[IN] ulPINType       PIN类型, 可为ADMIN_TYPE或USER_TYPE。
     * @param[IN] pulRetryCount   PIN码。
     * @param[OUT] pulRetryCount   出错后的重试次数。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int VerifyPIN(long hApplication, int ulPINType, byte[] szPIN, int[] pulRetryCount);

    /**
     * @brief 解锁PIN
     * @note 功能描述：当用户的 PIN 码锁死后，通过调用该函数来解锁用户 PIN 码。 解锁后，用户 PIN 码被设置成新值，用户 PIN 码的重试次数也恢复到原值。
     * @param[IN] hApplication    应用句柄。
     * @param[IN] szAdminPIN      管理员 PIN 码。
     * @param[IN] szNewUserPIN    新的用户 PIN 码。
     * @param[IN] pulRetryCount   管理员 PIN 码错误时，返回剩余重试次数。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：验证完管理员 PIN 才能够解锁用户 PIN 码，如果输入的 Administrator PIN不正确或者已经锁死，会调用失败，并返回 Administrator PIN 的重试次数。
     */
    public native int UnblockPIN(long hApplication, byte[] szAdminPIN, byte[] szNewUserPIN, int[] pulRetryCount);

    /**
     * @brief 清除应用安全状态
     * @param[IN] hApplication     应用句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ClearSecureState(long hApplication);

///----------------------------应用管理系列函数-------------------------------

    /**
     * @brief 创建应用
     * @param[IN] hDdev                连接设备时返回的设备句柄。
     * @param[IN] szAppName            应用名称。
     * @param[IN] szAdminPin           管理员PIN。
     * @param[IN] dwAdminPinRetryCount 管理员PIN最大重试次数。
     * @param[IN] szUserPin            用户PIN。
     * @param[IN] dwUserPinRetryCount  用户PIN最大重试次数。
     * @param[IN] dwCreateFileRights   在该应用下创建文件和容器的权限。
     * @param[IN] phApplication        应用的句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：需要设备权限
     */
    public native int CreateApplication(long hDev, byte[] szAppName, byte[] szAdminPin, int dwAdminPinRetryCount, byte[] szUserPin, int dwUserPinRetryCount, int dwCreateFileRights, long[] phApplication);

    /**
     * @brief 枚举应用
     * @param[IN] hDdev         连接设备时返回的设备句柄。
     * @param[OUT] szAppName     返回应用名称列表,如果参数为空，将由pulSize返回所需内存大小。每个应用以单个'\0'结束,以双'\0'表示列表结束。
     * @param[IN,OUT] pulSize       输入时表示缓冲区长度，输出时返回占用空间大小。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int EnumApplication(long hDev, byte[] szAppName, int[] pulSize);

    /**
     * @brief 删除应用
     * @param[IN] hDdev         连接设备时返回的设备句柄。
     * @param[IN] szAppName     应用名称。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：需要设备权限
     */
    public native int DeleteApplication(long hDev, byte[] szAppName);

    /**
     * @brief 打开应用
     * @param[IN] hDdev         连接设备时返回的设备句柄。
     * @param[IN] szAppName     应用名称。
     * @param[OUT] phApplication 应用的句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int OpenApplication(long hDev, byte[] szAppName, long[] phApplication);

    /**
     * @brief 关闭应用
     * @param[IN] hApplication    应用句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 备注：此函数不影响应用安全状态。
     */
    public native int CloseApplication(long hApplication);

///-----------------------------文件管理系列函数---------------------------------

    /**
     * @brief 创建文件
     * @note 功能描述：创建文件时要指定文件的名称、大小，以及读写权限。
     * @param[IN] hApplication    应用句柄。
     * @param[IN] szFileName      文件名称，长度不得大于32个字节。
     * @param[IN] ulFileSize      文件大小。
     * @param[IN] ulReadRights    文件读权限。
     * @param[IN] ulWriteRights   文件写权限。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：创建文件需要应用指定的创建文件权限。
     */
    public native int CreateFile(long hApplication, byte[] szFileName, int ulFileSize, int ulReadRights, int ulWriteRights);

    /**
     * @brief 删除文件
     * @note 功能描述：文件删除后，文件中写入的所有信息将丢失。文件在设备中占用的空间将被释放。
     * @param[IN] hApplication    要删除的文件所在的应用句柄。
     * @param[IN] szFileName      要删除的文件的名称。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：删除一个文件应具有对该文件的创建权限。
     */
    public native int DeleteFile(long hApplication, byte[] szFileName);

    /**
     * @brief 枚举文件
     * @param[IN] hApplication    应用句柄。
     * @param[OUT] szFileList      返回文件名称列表，,如果参数为空，将由pulSize返回所需内存大小。每个文件名称以单个'\0'结束,以双'\0'表示列表结束。
     * @param[IN,OUT] pulSize         输入时表示缓冲区长度，输出时返回占用空间大小。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int EnumFiles(long hApplication, byte[] szFileList, int[] pulSize);

    /**
     * @brief 获取文件属性
     * @param[IN] hApplication    应用句柄。
     * @param[IN] szFileName      文件名称。
     * @param[IN] pFileInfo       文件信息。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int GetFileInfo(long hApplication, byte[] szFileName, FileAttribute pFileInfo);

    /**
     * @brief 读文件
     * @param[IN] hApplication    应用句柄。
     * @param[IN] szFileName      文件名称。
     * @param[IN] ulOfffset       文件读取偏移位置。
     * @param[IN] ulSize          要读取文件的长度。
     * @param[OUT] pbOutData       返回数据的缓冲区。
     * @param[IN,OUT] pulOutLen       输入时表示缓冲区长度，输出时表示实际读取返回的数据大小。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：须具备该文件的读权限。
     */
    public native int ReadFile(long hApplication, byte[] szFileName, int ulOfffset, int ulSize, byte[] pbOutData, int[] pulOutLen);

    /**
     * @brief 写文件
     * @param[IN] hApplication    应用句柄。
     * @param[IN] szFileName      文件名称。
     * @param[IN] ulOfffset       写入文件的偏移量。
     * @param[IN] pbData          写入数据的缓冲区。
     * @param[IN] ulSize          写入数据的大小。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：须具备该文件的读权限。
     */
    public native int WriteFile(long hApplication, byte[] szFileName, int ulOfffset, byte[] pbData, int ulSize);

///-----------------------------容器管理系列函数----------------------------------

    /**
     * @brief 创建容器
     * @note 功能描述：在应用下建立制度名称的容器并返回容器句柄。
     * @param[IN] hApplication    应用句柄。
     * @param[IN] szContainerName 容器名称,ASCII字符串，最大长度不超过64字节。
     * @param[IN] pbContainer     返回所建立的容器句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：需要用户权限
     */
    public native int CreateContainer(long hApplication, byte[] szContainerName, long[] pbContainer);

    /**
     * @brief 删除容器
     * @param[IN] hApplication    应用句柄。
     * @param[IN] szContainerName 指向删除容器的名称。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：需要用户权限
     */
    public native int DeleteContainer(long hApplication, byte[] szContainerName);

    /*
    * @brief 枚举容器
    * @note  功能描述：枚举应用下的所有容器并返回容器名称列表。
    *
    * @param[IN]     hApplication    应用句柄。
    * @param[OUT]     szContainerName 指向容器名称列表缓冲区，如果此参数为 NULL 时，pulSize表示返回数据所需要缓冲区的长度，如果此参数不为 NULL 时，返回容器名称列表，每个容器名以单个‘\0’为结束，列表以双‘\0’结束。
    * @param[IN,OUT] pulSize         调用前表示 szContainerName 缓冲区的长度，返回容器名称列表的长度。
    *
    * @retval SAR_OK:  成功。
    * @retval 其他：   错误码。
    */
    public native int EnumContainer(long hApplication, byte[] szContainerName, int[] pulSize);

    /**
     * @brief 打开容器
     * @param[IN] hApplication    应用句柄。
     * @param[IN] szContainerName 容器名称。
     * @param[OUT] pbContainer     返回打开的的容器句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int OpenContainer(long hApplication, byte[] szContainerName, long[] pbContainer);

    /**
     * @brief 关闭容器
     * @param[IN] hContainer     容器句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int CloseContainer(long hContainer);

    /**
     * @brief 获得容器类型
     * @param[IN] hContainer       容器句柄。
     * @param[OUT] pulContainerType 容器类型，指针指向的值为0表示未定、尚未分配类型或者空容器，为1表示RSA容器，为2表示SM2容器
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int GetContainerType(long hContainer, int[] pulContainerType);

    /**
     * @brief 导入数字证书
     * @param[IN] hContainer       容器句柄。
     * @param[IN] bSignFlag        TRUE表示签名证书，FALSE表示加密证书。
     * @param[IN] pbCert           指向证书内容缓冲区。
     * @param[IN] ulCertLen        证书长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ImportCertificate(long hContainer, int bSignFlag, byte[] pbCert, int ulCertLen);

    /**
     * @brief 导出数字证书
     * @param[IN] hContainer       容器句柄。
     * @param[IN] bSignFlag        TRUE表示签名证书，FALSE表示加密证书。
     * @param[IN] pbCert           指向证书内容缓冲区，如果此参数为NULL时，pulCertLen表示返回数据所需缓冲区长度，否则返回证书内容。
     * @param[IN/OUT] pulCertLen       输入时表示pbCert缓冲区的长度，输出时表示证书内容的长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ExportCertificate(long hContainer, int bSignFlag, byte[] pbCert, int[] pulCertLen);

///------------------------------密码服务系列函数-----------------------------------

    /**
     * @brief 生成随机数
     * @note 产生指定长度的随机数
     * @param[IN] hDev        设备句柄。
     * @param[OUT] pbRandom    返回的随机数。
     * @param[IN] ulRandomLen 随机数长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int GenRandom(long hDev, byte[] pbRandom, int ulRandomLen);

    /**
     * @brief 生成外部RSA密钥对
     * @note 由设备生成 RSA 密钥对并明文输出。
     * @param[IN] hDev       设备句柄。
     * @param[IN] ulBitsLen  密钥模长。
     * @param[OUT] pBlob      返回的私钥数据结构。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 生成的私钥只用于输出，接口内不做保留和计算。
     */
    public native int GenExtRSAKey(long hDev, int ulBitsLen, RsaPrivateKeyBlob pBlob);

    /**
     * @brief 生成RSA签名密钥对
     * @note 生成RSA签名密钥对并输出签名公钥。
     * @param[IN] hContainer       容器句柄。
     * @param[IN] ulBitsLen        密钥模长。
     * @param[OUT] pBlob            返回的RSA公钥数据结构
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：须具备用户权限。
     */
    public native int GenRSAKeyPair(long hContainer, int ulBitsLen, RsaPublicKeyBlob pBlob);

    /**
     * @brief 导入RSA加密密钥对
     * @note 导入RSA加密公私钥对
     * @param[IN] hContainer         容器句柄。
     * @param[IN] ulSymAlgId         对称算法密钥标识。
     * @param[IN] pbWrappedKey       使用该容器内签名公钥保护的对称算法密钥。
     * @param[IN] ulWrappedKeyLen    保护的对称算法密钥长度
     * @param[IN] pbEncryptedData    对称算法密钥保护的 RSA 加密私钥。私钥的格式遵循 PKCS#1 v2.1: RSA Cryptography Standard 中的私钥格式定义。。
     * @param[IN] ulEncryptedDataLen 对称算法密钥保护的 RSA 加密公私钥对长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：须具备用户权限。
     */
    public native int ImportRSAKeyPair(long hContainer, int ulSymAlgId, byte[] pbWrappedKey, int ulWrappedKeyLen, byte[] pbEncryptedData, int ulEncryptedDataLen);

    /**
     * @brief RSA签名
     * @note 使用hContainer指定容器的签名私钥，对指定数据 pbData 进行数字签名。签名后的结果存放到 pbSignature 缓冲区，设置 pulSignLen 为签名的长度。
     * @param[IN]  hContainer    用来签名的私钥所在容器句柄。
     * @param[IN] pbData         被签名的数据。
     * @param[IN] ulDataLen     签名数据长度，应不大于 RSA 密钥模长-11。
     * @param[OUT] pbSignature   存放签名结果的缓冲区指针，如果值为 NULL，用于取得签名结果长度。
     * @param[IN,OUT] pulSignLen    输入为签名结果缓冲区大小，输出为签名结果长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：须具备用户权限。
     */
    public native int RSASignData(long hContainer, byte[] pbData, int ulDataLen, byte[] pbSignature, int[] pulSignLen);

    /**
     * @brief RSA验签
     * @note 验证RSA签名。用 pRSAPubKeyBlob 内的公钥值对待验签数据进行验签。
     * @param[IN] hDev            设备句柄。
     * @param[IN] pRSAPubKeyBlob  公钥数据结构。
     * @param[IN] pbData          待验证签名的数据。
     * @param[IN] ulDataLen       数据长度，应不大于公钥模长-11。
     * @param[IN] pbSignature     待验证的签名值。
     * @param[IN] ulSignLen       签名值长度，必须为公钥模长。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int RSAVerify(long hDev, RsaPublicKeyBlob pRSAPubKeyBlob, byte[] pbData, int ulDataLen, byte[] pbSignature, int ulSignLen);

    /**
     * @brief RSA生成并导出会话密钥
     * @note 生成会话密钥并用外部公钥加密输出。
     *
     * @param[IN]      hContainer      容器句柄。
     * @param[IN]     ulAlgId          会话密钥算法标识。
     * @param[IN]     pPubKey          加密会话密钥的 RSA 公钥数据结构。
     * @param[OUT]      pbData          导出的加密会话密钥密文，按照 PKCS#1v1.5 要求封装。
     * @param[IN,OUT]     pulDataLen       返回导出数据长度。
     * @param[OUT]     phSessionKey     导出的密钥句柄。
     *
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int RSAExportSessionKey(long hContainer, int ulAlgId, RsaPublicKeyBlob pPubKey, byte[] pbData, int[] pulDataLen, long[] phSessionKey);

    /**
     * @brief RSA外来公钥运算
     * @note 使用外部传入的 RSA 公钥对输入数据做公钥运算并输出结果。
     * @param[IN] hDev            设备句柄。
     * @param[IN] pRSAPubKeyBlob  RSA 公钥数据结构。
     * @param[IN] pbInput         指向待运算的原始数据缓冲区。
     * @param[IN] ulInputLen      待运算原始数据的长度，必须为公钥模长。
     * @param[OUT] pbOutput        指向 RSA 公钥运算结果缓冲区，如果该参数为NULL，则由 pulOutputLen 返回运算结果的实际长度。
     * @param[IN,OUT] pulOutputLen    调用前表示 pbOutput 缓冲区的长度，返回RSA 公钥运算结果的实际长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ExtRSAPubKeyOperation(long hDev, RsaPublicKeyBlob pRSAPubKeyBlob, byte[] pbInput, int ulInputLen, byte[] pbOutput, int[] pulOutputLen);

    /**
     * @brief RSA外来私钥运算
     * @note 直接使用外部传入的 RSA 私钥对输入数据做私钥运算并输出结果。
     * @param[IN] hDev            设备句柄。
     * @param[IN] pRSAPriKeyBlob  RSA 私钥数据结构,需要填充公钥信息。
     * @param[IN] pbInput         指向待运算数据缓冲区。
     * @param[IN] ulInputLen      待运算数据的长度，必须为公钥模长。
     * @param[OUT] pbOutput        RSA 私钥运算结果，如果该参数为 NULL，则由pulOutputLen 返回运算结果的实际长度。
     * @param[IN,OUT] pulOutputLen    调用前表示 pbOutput 缓冲区的长度，返回RSA 私钥运算结果的实际长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ExtRSAPriKeyOperation(long hDev, RsaPrivateKeyBlob pRSAPriKeyBlob, byte[] pbInput, int ulInputLen, byte[] pbOutput, int[] pulOutputLen);

    /**
     * @brief 生成ECC签名密钥对
     * @note 生成ECC签名密钥对并输出签名公钥
     * @param[IN] hContainer        密钥容器句柄。
     * @param[IN] ulAlgId           算法标识，只支持SGD_SM2_1算法。
     * @param[OUT] pBlob             返回ECC公钥数据结构。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：需要用户权限
     */
    public native int GenECCKeyPair(long hContainer, int ulAlgId, EccPublicKeyBlob pBlob);

    /**
     * @brief 导入ECC公私钥对
     * @param[IN] hContainer        密钥容器句柄。
     * @param[IN] pEenvlopedKeyBlob 受保护的加密密钥对。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：需要用户权限
     */
    public native int ImportECCKeyPair(long hContainer, EnvelopedKeyBlob pEenvlopedKeyBlob);

    /**
     * @brief ECC签名
     * @param[IN] hContainer        密钥容器句柄。
     * @param[IN] pbData            待签名的数据。
     * @param[IN] ulDataLen         待签名的数据长度，必须小于密钥模长。
     * @param[OUT] pSignature        签名值。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：需要用户权限。
     * 输入数据位待签数据的杂凑值,使用SM2算法时,该输入数据位待签名数据经过SM2签名预处理的结果。
     */
    public native int ECCSignData(long hContainer, byte[] pbData, int ulDataLen, EccSignatureBlob pSignature);

    /**
     * @brief ECC验签
     * @note 输入数据位待签数据的杂凑值, 使用SM2算法时, 该输入数据位待签名数据经过SM2签名预处理的结果。
     * @param[IN] hDev            设备句柄。
     * @param[IN] pECCPubKeyBlob  ECC公钥数据结构。
     * @param[IN] pbData          待验证签名的数据。
     * @param[IN] ulDataLen       数据长度。
     * @param[IN] pSignature      待验证签名值。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ECCVerify(long hDev, EccPublicKeyBlob pECCPubKeyBlob, byte[] pbData, int ulDataLen, EccSignatureBlob pSignature);

    /**
     * @brief ECC生成并到处会话密钥
     * @note 功能描述：生成会话密钥并用外部公钥加密导出。
     * @param[IN] hContainer        容器句柄。
     * @param[IN] ulAlgId           会话密钥算法标识。
     * @param[IN] pPubKey           外部输入的公钥结构。
     * @param[OUT] pData             会话密钥密文。
     * @param[OUT] phSessionKey      会话密钥句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ECCExportSessionKey(long hContainer, int ulAlgId, EccPublicKeyBlob pPubKey, EccCipherBlob pData, long[] phSessionKey);

    /**
     * @brief ECC外来公钥加密
     * @note 功能描述:使用外部传入的 ECC 公钥对输入数据做加密运算并输出结果。
     * @param[IN] hDev            设备句柄。
     * @param[IN] pECCPubkeyBlob  ECC公钥数据结构。
     * @param[IN] pbPlainText     待加密的明文数据。
     * @param[IN] ulPlainTextLen  待加密的明文数据长度。
     * @param[OUT] pCipherText     密文数据。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ExtECCEncrypt(long hDev, EccPublicKeyBlob pECCPubkeyBlob, byte[] pbPlainText, int ulPlainTextLen, EccCipherBlob pCipherText);

    /**
     * @brief ECC外来私钥解密
     * @note 功能描述:使用外部传入的 ECC 私钥对输入数据做解密运算并输出结果。
     * @param[IN] hDev            设备句柄。
     * @param[IN] pECCPriKeyBlob  ECC 私钥数据结构。
     * @param[IN] pCipherText     待解密的密文数据。
     * @param[OUT] pbPlainText     返回明文数据，如果该参数为 NULL，则由pulPlainTextLen 返回明文数据的实际长度。
     * @param[IN,OUT] pulPlainTextLen 调用前表示pbPlainText 缓冲区的长度，返回明文数据的实际长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ExtECCDecrypt(long hDev, EccPrivateKeyBlob pECCPriKeyBlob, EccCipherBlob pCipherText, byte[] pbPlainText, int[] pulPlainTextLen);

    /**
     * @brief ECC外来私钥签名
     * @note 功能描述:使用外部传入的 ECC 私钥对输入数据做签名运算并输出结果。
     * @param[IN] hDev            设备句柄。
     * @param[IN] pECCPriKeyBlob  ECC 私钥数据结构。
     * @param[IN] pbData          待签名数据。
     * @param[IN] ulDataLen       待签名数据的长度。
     * @param[OUT] pSignature      签名值。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ExtECCSign(long hDev, EccPrivateKeyBlob pECCPriKeyBlob, byte[] pbData, int ulDataLen, EccSignatureBlob pSignature);

    /**
     * @brief ECC外来公钥验签
     * @note 功能描述:外部使用传入的ECC公钥做签名验证。
     * @param[IN] hDev           设备句柄。
     * @param[IN] pECCPubKeyBlob ECC公钥数据结构。
     * @param[IN] pbData         待验证数据。
     * @param[IN] ulDataLen      待验证数据的长度。
     * @param[IN] pSignature     签名值。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int ExtECCVerify(long hDev, EccPublicKeyBlob pECCPubKeyBlob, byte[] pbData, int ulDataLen, EccSignatureBlob pSignature);

    /**
     * @brief ECC生成密钥协商参数并输出
     * @note 使用 ECC 密钥协商算法，为计算会话密钥而产生协商参数，返回临时 ECC 密钥对的公钥及协商句柄。
     * @param[IN] hContainer           容器句柄。
     * @param[IN] ulAlgId              会话密钥算法标识。
     * @param[OUT] pTempECCPubKeyBlob   发起方临时 ECC 公钥。
     * @param[IN] pbID                 发起方的ID。
     * @param[IN] ulIDLen              发起方 ID 的长度，不大于 32。
     * @param[OUT] phAgreementHandle    返回的密钥协商句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 为协商会话密钥，协商的发起方应首先调用本函数
     */
    public native int GenerateAgreementDataWithECC(long hContainer, int ulAlgId, EccPublicKeyBlob pTempECCPubKeyBlob, byte[] pbID, int ulIDLen, long[] phAgreementHandle);

    /**
     * @brief ECC产生协商数据并计算会话密钥
     * @note 使用 ECC 密钥协商算法，产生协商参数并计算会话密钥，输出临时 ECC 密钥对公钥，并返回产生的密钥句柄。
     * @param[IN] hContainer                容器句柄。
     * @param[IN] ulAlgId                   会话密钥算法标识。
     * @param[IN] pSponsorECCPubKeyBlob     发起方的 ECC 公钥。
     * @param[IN] pSponsorTempECCPubKeyBlob 发起方的临时 ECC 公钥。
     * @param[OUT] pTempECCPubKeyBlob        响应方的临时 ECC 公钥。
     * @param[IN] pbID                      响应方的 ID。
     * @param[IN] ulIDLen                   响应方 ID 的长度，不大于 32。
     * @param[IN] pbSponsorID               发起方的 ID。
     * @param[IN] ulSponsorIDLen            发起方 ID 的长度，不大于 32。
     * @param[OUT] phKeyHandle              返回的对称算法密钥句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 本函数由响应方调用
     */
    public native int GenerateAgreementDataAndKeyWithECC(long hContainer, int ulAlgId, EccPublicKeyBlob pSponsorECCPubKeyBlob, EccPublicKeyBlob pSponsorTempECCPubKeyBlob, EccPublicKeyBlob pTempECCPubKeyBlob, byte[] pbID, int ulIDLen, byte[] pbSponsorID, int ulSponsorIDLen, long[] phKeyHandle);

    /**
     * @brief ECC计算会话密钥
     * @note 使用 ECC 密钥协商算法，使用自身协商句柄和响应方的协商参数计算会话密钥，同时返回会话密钥句柄。
     * @param[IN] hAgreementHandle    密钥协商句柄。
     * @param[IN] pECCPubKeyBlob      外部输入的响应方 ECC 公钥。
     * @param[IN] pTempECCPubKeyBlob  外部输入的响应方临时 ECC 公钥。
     * @param[IN] pbID                响应方的 ID。
     * @param[IN] ulIDLen             响应方 ID 的长度，不大于 32。
     * @param[OUT] phKeyHandle         返回的密钥句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 协商的发起方获得响应方的协商参数后调用本函数，计算会话密钥。
     */
    public native int GenerateKeyWithECC(long hAgreementHandle, EccPublicKeyBlob pECCPubKeyBlob, EccPublicKeyBlob pTempECCPubKeyBlob, byte[] pbID, int ulIDLen, long[] phKeyHandle);

    /**
     * @brief 导出公钥
     * @note 功能描述：导出容器中的签名公钥或加密公钥
     * @param[IN] hContainer        密钥容器句柄。
     * @param[IN] bSignFlag         TRUE表示导出签名公钥，FALSE表示导出加密公钥。
     * @param[OUT] pbBlob            指向RSA公钥结构或者ECC公钥结构，如果此参数为NULL，由pulBlobLen返回的长度
     * @param[IN,OUT] pulBlobLen        输入时表示pbBlob缓冲区长度，输出时表示导出公钥结构大小。
     * @retval SAR_OK:              成功。
     * @retval SAR_ BAD_SIGNATURE： 验证失败。
     * @retval 其他：               错误码。
     */
    public native int ExportPublicKey(long hContainer, int bSignFlag, byte[] pbBlob, int[] pulBlobLen);

    /**
     * @brief 导入会话密钥
     * @note 功能描述：导入会话密钥密文，使用容器中加密私钥解密。
     * @param[IN] hContainer        密钥容器句柄。
     * @param[IN] ulAlgId           会话密钥算法标识。
     * @param[IN] pbWrapedData      要导入的会话密钥密文。当容器为ECC类型时，此参数为ECCCIPHERBLOB密文数据,当容器为RSA类型时，此参数为RSA公钥加密后的数据。
     * @param[IN] ulWrapedLen       会话密钥密文长度。
     * @param[OUT] phKey             返回会话密钥句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 权限要求：用户权限。
     */
    public native int ImportSessionKey(long hContainer, int ulAlgId, byte[] pbWrapedData, int ulWrapedLen, long[] phKey);

    /**
     * @brief 明文导入会话密码
     * @note  功能描述：设置明文对称密钥，返回密钥句柄。
     *
     * @param[IN]      hDev       设备句柄。
     * @param[IN]      pbKey      指向会话密钥值的缓冲区。
     * @param[IN]      ulAlgID    会话密钥算法标识。
     * @param[OUT]      phKey      返回会话密钥句柄。
     *
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int SetSymmKey(long hDev, byte[] pbKey, int ulAlgID, long[] phKey);

    /**
     * @brief 加密初始化
     * @note 功能描述：设置数据加密的算法相关参数
     * @param[IN] hKey         加密密钥句柄。
     * @param[IN] EncryptParam 分组密钥算法相关参数。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int EncryptInit(long hKey, BlockCipherParam EncryptParam);

    /**
     * @brief 单组数据加密
     * @note 功能描述：单一分组数据的加密操作。在调用Encrypt之前必须先调用EncryptInit初始化。
     * @param[IN] hKey            加密密钥句柄。
     * @param[IN] pbData          待加密数据。
     * @param[IN] ulDataLen       待加密数据长度。
     * @param[OUT] pbEncryptedData 加密后的数据缓冲区，可以为NULL，用于获得加密后数据长度。
     * @param[IN,OUT] pulEncryptedLen 输入时表示缓冲区长度，输出时表示结果数据实际长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int Encrypt(long hKey, byte[] pbData, int ulDataLen, byte[] pbEncryptedData, int[] pulEncryptedLen);

    /**
     * @brief 多组数据加密
     * @note 功能描述：对多个分组数据进行加密。在调用EncryptUpdate之前必须先调用EncryptInit，在调用EncryptUpdate之后，必须调用EncryptFinal。
     * @param[IN] hKey            加密密钥句柄。
     * @param[IN] pbData          待加密数据。
     * @param[IN] ulDataLen       待加密数据长度。
     * @param[OUT] pbEncryptedData 加密后的数据缓冲区指针。
     * @param[OUT] pulEncryptedLen 返回加密后的数据长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int EncryptUpdate(long hKey, byte[] pbData, int ulDataLen, byte[] pbEncryptedData, int[] pulEncryptedLen);

    /**
     * @brief 结束加密
     * @note 功能描述：结束多个分组的数据加密，返回剩余加密结果。先调用EncryptInit初始化，再调用EncryptUpdate对多个分组加密，最后调用EncryptFinal结束多个分组数据加密。
     * @param[IN] hKey            加密密钥句柄。
     * @param[OUT] pbEncryptedData 加密结果的缓冲区。
     * @param[OUT] pulEncryptedLen 加密结果的长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int EncryptFinal(long hKey, byte[] pbEncryptedData, int[] pulEncryptedLen);

    /**
     * @brief 解密初始化
     * @note 功能描述：设置数据解密的算法相关参数
     * @param[IN] hKey         解密密钥句柄。
     * @param[IN] EncryptParam 分组密钥算法相关参数。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int DecryptInit(long hKey, BlockCipherParam EncryptParam);

    /**
     * @brief 单组数据解密
     * @note 功能描述：单一分组数据的解密操作。在调用Decrypt之前必须先调用DecryptInit初始化。
     * @param[IN] hKey            解密密钥句柄。
     * @param[IN] pbEncryptedData 待解密数据。
     * @param[IN] ulEncryptedLen  待解密数据长度。
     * @param[OUT] pbData          解密后的数据缓冲区，可以为NULL，用于获得加密后数据长度。
     * @param[IN,OUT] pulDataLen      输入时表示缓冲区长度，输出时表示结果数据实际长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int Decrypt(long hKey, byte[] pbEncryptedData, int ulEncryptedLen, byte[] pbData, int[] pulDataLen);

    /**
     * @brief 分组数据解密
     * @note 功能描述：对多个分组数据进行解密。在调用DecryptUpdate之前必须先调用DecryptInit，在调用DecryptUpdate之后，必须调用DecryptFinal。
     * @param[IN] hKey            解密密钥句柄。
     * @param[IN] pbEncryptedData 待解密数据。
     * @param[IN] ulEncryptedLen  待解密数据长度。
     * @param[OUT] pbData          解密后的数据缓冲区指针。
     * @param[IN,OUT] pulDataLen      输入时表示缓冲区长度，输出时表示结果数据实际长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int DecryptUpdate(long hKey, byte[] pbEncryptedData, int ulEncryptedLen, byte[] pbData, int[] pulDataLen);

    /**
     * @brief 结束多个分组的解密
     * @note 功能描述：先调用DecryptInit初始化解密操作，再调用DecryptUpdate对多个分组进行解密，最后调用DecryptFinal结束多个分组数据的解密。
     * @param[IN] hKey                解密密钥句柄。
     * @param[OUT] pbDecryptedData     指向解密结果的缓冲区。
     * @param[IN,OUT] pulDecryptedDataLen 输入时表示缓冲区长度，输出时表示解密结果长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int DecryptFinal(long hKey, byte[] pbDecryptedData, int[] pulDecryptedDataLen);

    /**
     * @brief 密码杂凑初始化
     * @param[IN] hDev        设备句柄。
     * @param[IN] ulAlgID     密码杂凑算法标识。
     * @param[IN] pPubKey     签名者公钥。当ulAlgID为SGD_SM3时有效。
     * @param[IN] pucID       签名者ID值。当ulAlgID为SGD_SM3时有效。
     * @param[IN] ulIDLen     签名者ID的长度。当ulAlgID为SGD_SM3时有效。
     * @param[OUT] phHash      密码杂凑对象句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 当ulAlgID为SGD_SM3且ulIDLen不为0的情况下，pPubKey、pucID有效，执行SM2算法签名预处理1操作。
     */
    public native int DigestInit(long hDev, int ulAlgID, EccPublicKeyBlob pPubKey, byte[] pucID, int ulIDLen, long[] phHash);

    /**
     * @brief 单组数据密码杂凑
     * @note 功能描述：对单一分组的消息进行密码杂凑计算。调用Digest之前必须调用DigestInit初始化。
     * @param[IN] hDev        。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 调用 Digest 之前，必须调用 DigestInit 初始化杂凑计算操作。Digest等价于多次调用 DigestUpdate 之后再调用 DigestFinal。
     */
    public native int Digest(long hHash, byte[] pbData, int ulDataLen, byte[] phHashData, int[] pulHashLen);

    /**
     * @brief 分组数据密码杂凑
     * @note 功能描述：对多个分组数据进行密码杂凑运算。调用DigestUpdate之前必须调用DigestInit；调用DigestUpdate之后必须调用DigestFinal。
     * @param[IN] hHash        密码杂凑对象句柄。
     * @param[IN] pbData       指向消息数据的缓冲区。
     * @param[IN] ulDataLen    消息数据的长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 调用 DigestUpdate 之前，必须调用 DigestInit 初始化杂凑计算操作；调用 DigestUpdate 之后，必须调用DigestFinal 结束杂凑计算操作。
     */
    public native int DigestUpdate(long hHash, byte[] pbData, int ulDataLen);

    /**
     * @brief 结束密码杂凑
     * @note 功能描述：结束多个分组的密码杂凑计算，将结果保存到指定的缓冲区。
     * @param[IN] hHash        密码杂凑对象句柄。
     * @param[OUT] pHashData    返回的密码杂凑结果缓冲区指针，如果此参数为NULL，由pulHashLen返回杂凑结果长度。
     * @param[IN,OUT] pulHashLen   输入时表示缓冲区长度，输出时表示密码杂凑结果的长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark DigestFinal 必须用于 DigestUpdate 之后。
     */
    public native int DigestFinal(long hHash, byte[] pHashData, int[] pulHashLen);

    /**
     * @brief 消息鉴别码运算初始化
     * @note 功能描述：初始化消息鉴别码计算操作，设置计算消息鉴别码的密钥参数，并返回消息鉴别码句柄。
     * @param[IN] hKey       计算消息鉴别码的密钥句柄。
     * @param[IN] pMacParam  消息认证计算相关参数，包括初始向量、初始向量长度、填充方法等。
     * @param[OUT] phMac      消息鉴别码对象句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 消息鉴别码计算采用分组加密算法的 CBC 模式，将加密结果的最后一块作为计算结果。待计算数据的长度必须是分组加密算法块长的倍数，接口内部不作数据填充。
     */
    public native int MacInit(long hKey, BlockCipherParam pMacParam, long[] phMac);

    /**
     * @brief 单组数据消息鉴别码运算
     * @note 功能描述：Mac 计算单一分组数据的消息鉴别码。
     * @param[IN] hMac       消息鉴别码句柄。
     * @param[IN] pbData     指向待计算数据的缓冲区。
     * @param[IN] ulDataLen  待计算数据的长度。
     * @param[OUT] pbMacData  指向计算后的 Mac 结果，如果此参数为 NULL 时，由 pulMacLen返回计算后 Mac 结果的长度。
     * @param[IN,OUT] pulMacLen  调用时表示 pbMacData 缓冲区的长度，返回计算 Mac 结果的长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 调用 Mac 之前，必须调用 MacInit 初始化消息鉴别码计算操作。Mac等价于多次调用 MacUpdate 之后再调用 MacFinal。
     */
    public native int Mac(long hMac, byte[] pbData, int ulDataLen, byte[] pbMacData, int[] pulMacLen);

    /**
     * @brief 多组数据消息鉴别码运算
     * @note 功能描述：计算多个分组数据的消息鉴别码。
     * @param[IN] hMac       消息鉴别码句柄。
     * @param[IN] pbData     指向待计算数据的缓冲区。
     * @param[IN] ulDataLen  待计算数据的长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark 调用 MacUpdate 之前，必须调用 MacInit 初始化消息鉴别码计算操作；调用 MacUpdate 之后，必须调用 MacFinal 结束多个分组数据的消息鉴别码计算操作。
     */
    public native int MacUpdate(long hMac, byte[] pbData, int ulDataLen);

    /**
     * @brief 结束消息鉴别码运算
     * @note 功能描述：结束多个分组数据的消息鉴别码计算操作。
     * @param[IN] hMac           消息鉴别码句柄。
     * @param[OUT] pbMacData      指向消息鉴别码的缓冲区，当此参数为 NULL 时，由pulMacDataLen 返回消息鉴别码返回的长度。
     * @param[OUT] pulMacDataLen  调用时表示消息鉴别码缓冲区的最大长度，返回消息鉴别码的长度。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     * @remark MacFinal 必须用于 MacUpdate 之后。
     */
    public native int MacFinal(long hMac, byte[] pbMacData, int[] pulMacDataLen);

    /**
     * @brief 关闭密码对象句柄
     * @note 功能描述：关闭会话密钥、密码杂凑对象、MAC对象、ECC密钥协商等句柄
     * @param[IN] hHandle        要关闭的对象句柄。
     * @retval SAR_OK:  成功。
     * @retval 其他：   错误码。
     */
    public native int CloseHandle(long hHandle);

}
