/**
 * Xdjacrypto.java
 *
 * @author zhangxiaolong@xdja.com
 * 2015-7-20
 */
package com.xdja.crypto;

import android.text.TextUtils;
import android.util.Pair;

import com.xdja.xdjacrypto.SM2PrivateKey;
import com.xdja.xdjacrypto.SM2PublicKey;
import com.xdja.xdjacrypto.SM2Signature;
import com.xdja.xdjacrypto.XCT_CERT_INFO;
import com.xdja.xdjacrypto.XDJA_RSA_PRIKEY;
import com.xdja.xdjacrypto.XDJA_RSA_PUBKEY;
import com.xdja.xdjacrypto.XdjaCrypto;
import com.xdja.xdjacrypto.oct_string;

import java.io.ByteArrayInputStream;
import java.security.PublicKey;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Arrays;

public class XDJACrypto {
    private static XDJACrypto mInstance;
    private XdjaCrypto crypto = null;
    private long[] hHandle = new long[2];

    private final String version = "1.0.3100";

    /**
     * 主要是私钥加密时的填充方式
     */
    public final static byte FLAG_PKCS1_PADDING_1 = 1;
    /**
     * 主要是公钥加密时的填充方式
     */
    public final static byte FLAG_PKCS1_PADDING_2 = 2;

    /**
     * 得到版本号
     *
     * @return 版本号
     */
    public String getVersion() {
        return version;
    }

    private XDJACrypto() {
        crypto = new XdjaCrypto();
    }

    public XDJACrypto(String name) {
        crypto = new XdjaCrypto();
    }

    /**
     * 得到实例
     *
     * @return 实例
     */
    public static XDJACrypto getInstance() {
        if (mInstance == null) {
            synchronized (XDJACrypto.class) {
                if (mInstance == null) {
                    mInstance = new XDJACrypto();
                }
            }
        }

        return mInstance;
    }

    //↓------------------------------SM2-------------------------------------------↓
    //↓                                                                            ↓
    //↓------------------------------SM2-------------------------------------------↓

    /**
     * SM2_signature签名
     *
     * @param pub     [in]sm2签名公钥
     * @param priv    [in]sm2签名私钥
     * @param message [in]需要签名的数据
     * @param mlen    [in]数据长度
     * @param value   [out]签名结果
     * @return 0-成功   -1-失败
     */
    public int SM2Signature(SM2PublicKey pub, SM2PrivateKey priv, byte[] message, int mlen, SM2Signature value) {
        return crypto.SM2Signature(pub, priv, message, mlen, value);
    }

    /**
     * SM2_verify验签
     *
     * @param pub     [in]sm2签名公钥
     * @param message [in]需要签名的数据
     * @param mlen    [in]数据长度
     * @param value   [in]签名结果
     * @return 0-成功   -1-失败
     */
    public int SM2Verify(SM2PublicKey pub, byte[] message, int mlen, SM2Signature value) {
        return crypto.SM2Verify(pub, message, mlen, value);
    }

    /**
     * SM2_public_encrypt公钥加密
     *
     * @param pub  [in]sm2公钥
     * @param from [in]需要加密的数据
     * @param flen [in]数据长度
     * @param to   [out]加密的密文数据
     * @param tlen [out]密文数据长度
     * @return 0-成功   -1-失败
     */
    public int SM2PublicEncrypt(SM2PublicKey pub, byte[] from, int flen, byte[] to, int[] tlen) {
        return crypto.SM2PublicEncrypt(pub, from, flen, to, tlen);
    }

    /**
     * SM2公钥加密（新国密标准）
     *
     * @param pub  [in]sm2公钥
     * @param from [in]需要加密的数据
     * @param flen [in]数据长度
     * @param to   [out]加密的密文数据
     * @param tlen [out]密文数据长度
     * @return 0-成功   -1-失败
     */
    public int SM2PublicEncryptGM(SM2PublicKey pub, byte[] from, int flen, byte[] to, int[] tlen) {
        return crypto.SM2PublicEncryptGM(pub, from, flen, to, tlen);
    }

    /**
     * SM2_private_decrypt私钥解密
     *
     * @param priv [in]sm2私钥
     * @param from [in]需要解密的数据
     * @param flen [in]数据长度
     * @param to   [out]解密的明文数据
     * @param tlen [out]明文数据长度
     * @return 0-成功   -1-失败
     */
    public int SM2PrivateDecrypt(SM2PrivateKey priv, byte[] from, int flen, byte[] to, int[] tlen) {
        return crypto.SM2PrivateDecrypt(priv, from, flen, to, tlen);
    }

    /**
     * SM2私钥解密（新国密标准）
     *
     * @param priv [in]sm2私钥
     * @param from [in]需要解密的数据
     * @param flen [in]数据长度
     * @param to   [out]解密的明文数据
     * @param tlen [out]明文数据长度
     *             返回值：0-成功   -1-失败
     */
    public int SM2PrivateDecryptGM(SM2PrivateKey priv, byte[] from, int flen, byte[] to, int[] tlen) {
        return crypto.SM2PrivateDecryptGM(priv, from, flen, to, tlen);
    }

    /**
     * SM2_exchange_random生成临时公钥和私钥
     *
     * @param tpub  [out]sm2临时公钥
     * @param tpriv [out]sm2临时私钥
     * @return 0-成功   -1-失败
     */
    public int SM2ExchangeRandom(SM2PublicKey tpub, SM2PrivateKey tpriv) {
        return crypto.SM2ExchangeRandom(tpub, tpriv);
    }

    /**
     * SM2_exchange_cale密钥协商，生成共享会话密钥
     *
     * @param mytpriv  [in]用SM2_exchange_random()请求方临时私钥
     * @param mytpub   [in]用SM2_exchange_random()请求方临时公钥
     * @param peertpub [in]用SM2_exchange_random()应答方临时私钥
     * @param mypub    [in]请求方的公钥
     * @param mypriv   [in]请求方的私钥
     * @param peerpub  [in]应答方的公钥
     * @param keylen   [in]共享会话密钥长度
     * @param role     [in]角色RESPONDER或INITIATOR
     * @param key      [out]共享会话密钥
     * @param hash     [out]哈希验证标识
     * @return 0-成功   -1-失败
     */
    public int SM2ExchangeCale(SM2PrivateKey mytpriv, SM2PublicKey mytpub, SM2PublicKey peertpub, SM2PublicKey mypub,
                               SM2PrivateKey mypriv, SM2PublicKey peerpub, int keylen, int role, byte[] key, oct_string hash) {
        return crypto.SM2ExchangeCale(mytpriv, mytpub, peertpub, mypub, mypriv, peerpub, keylen, role, key, hash);
    }

    /**
     * SM2_exchange_verify密钥协商验证
     *
     * @param s1 [in]请求方验证标识
     * @param s2 [in]应答方验证标识
     * @return true-成功   false-失败
     */
    public boolean SM2ExchangeVerify(oct_string s1, oct_string s2) {
        return crypto.SM2ExchangeVerify(s1, s2);
    }

    //↑------------------------------SM2--------------------------------------↑
    //↑                                                                       ↑
    //↑------------------------------SM2--------------------------------------↑

    //↓------------------------------SHA1--------------------------------------↓
    //↓                                                                        ↓
    //↓------------------------------SHA1--------------------------------------↓

    /**
     * SHA1运算初始化
     *
     * @param hHandle 初始化产生的运算句柄，保存在hHandle[0]中
     * @return XCR_OK正确，其他错误
     */
    public int SHA1Init(long[] hHandle) {
        return crypto.SHA1Init(hHandle);
    }

    /**
     * SHA1运算过程
     *
     * @param handle 运算句柄
     * @param datain 待运算的数据块
     * @param in_len 数据块长度
     * @return XCR_OK正确，其他错误
     */
    public int SHA1Update(long handle, byte[] datain, int in_len) {
        return crypto.SHA1Update(handle, datain, in_len);
    }

    /**
     * SHA1运算结束，返回运算数据
     *
     * @param handle  运算句柄
     * @param dataout 运算返回的数据，长度为32
     * @return XCR_OK正确，其他错误
     */
    public int SHA1Final(long handle, byte[] dataout) {
        return crypto.SHA1Final(handle, dataout);
    }


    //↑------------------------------SHA1--------------------------------------↑
    //↑                                                                        ↑
    //↑------------------------------SHA1--------------------------------------↑

    //↓------------------------------SM3--------------------------------------↓
    //↓                                                                       ↓
    //↓------------------------------SM3--------------------------------------↓

    /**
     * SM3运算初始化
     *
     * @return XCR_OK正确，其他错误
     */
    public int SM3Init() {
        if (hHandle == null) {
            hHandle = new long[2];
        }
        return crypto.SM3Init(hHandle);
    }

    /**
     * SM3运算过程
     *
     * @param datain [in]待运算的数据块
     * @param in_len [in]数据块长度
     * @return XCR_OK正确，其他错误
     */
    public int SM3Update(byte[] datain, int in_len) {
        return crypto.SM3Update(hHandle[0], datain, in_len);
    }

    /**
     * SM3运算结束，返回运算数据
     *
     * @param dataout [out]运算返回的数据，长度为32
     * @return XCR_OK正确，其他错误
     */
    public int SM3Final(byte[] dataout) {
        int ret = crypto.SM3Final(hHandle[0], dataout);
        hHandle = null;
        return ret;
    }

    //↑------------------------------SM3--------------------------------------↑
    //↑                                                                       ↑
    //↑------------------------------SM3--------------------------------------↑


    //↓------------------------------RSA--------------------------------------↓
    //↓                                                                       ↓
    //↓------------------------------RSA--------------------------------------↓

    /**
     * RSA公钥计算
     *
     * @param in     要计算的数据。当key的长度为128时，in长度为128；当key的长度为256时，in的长度为256
     * @param inLen
     * @param pubKey
     * @param out    跟in的长度一致
     * @param outLen
     * @return 0：成功，其他：失败
     */
    public int RSAPubKeyCalc(byte[] in, int inLen, XDJA_RSA_PUBKEY pubKey, byte[] out, int[] outLen) {
        return crypto.XCF_RSAPubKeyCalc(in, inLen, pubKey, out, outLen);
    }

    /**
     * RSA 私钥计算
     *
     * @param in     要计算的数据。当key的长度为128时，in长度为128；当key的长度为256时，in的长度为256
     * @param inLen
     * @param priKey
     * @param out    出参，调用接口时不能为空，跟in的长度一致
     * @param outLen 出参，调用接口时不能为空，
     * @return 0：成功，其他：失败
     */
    public int RSAPriKeyCalc(byte[] in, int inLen, XDJA_RSA_PRIKEY priKey, byte[] out, int[] outLen) {
        return crypto.XCF_RSAPriKeyCalc(in, inLen, priKey, out, outLen);
    }

    /**
     * RSA Gen Key Pair
     *
     * @param bits   要生成的RSA的长度字节，1024或2048
     * @param pubKey 出参，调用接口时，不能为空
     * @param priKey 出参，调用接口时，不能为空
     * @return 0:成功，其他：失败
     */
    public int genRSAKeyPair(int bits, XDJA_RSA_PUBKEY pubKey, XDJA_RSA_PRIKEY priKey) {
        return crypto.XCF_GenRSAKeyPair(bits, pubKey, priKey);
    }

    /**
     * 使用SHA1算法的RSA算法签名
     *
     * @param priKey
     * @param dataIn 不需要填充
     * @return
     */
    public Pair<Integer, byte[]> RSASignWITHSHA1(XDJA_RSA_PRIKEY priKey, byte[] dataIn) {
        int ret;
        long[] handle = new long[1];
        ret = SHA1Init(handle);
        if (ret != 0) {
            return Pair.create(ret, null);
        }

        ret = SHA1Update(handle[0], dataIn, dataIn.length);
        if (ret != 0) {
            return Pair.create(ret, null);
        }

        byte[] sha1Out = new byte[20];
        ret = SHA1Final(handle[0], sha1Out);
        if (ret != 0) {
            return Pair.create(ret, null);
        }

        byte[] pcksOut = new byte[priKey.bits / 8];
        ret = PKCS1Padding(sha1Out, sha1Out.length, FLAG_PKCS1_PADDING_1, priKey.bits, pcksOut);
        if (ret != 0) {
            return Pair.create(ret, null);
        }

        byte[] calcOut = new byte[pcksOut.length];
        int[] outLen = new int[1];
        ret = RSAPriKeyCalc(pcksOut, pcksOut.length, priKey, calcOut, outLen);
        return Pair.create(ret, Arrays.copyOfRange(calcOut, 0, outLen[0]));
    }

    /**
     * RSA验签算法
     *
     * @param pubKey
     * @param dataIn   不需要填充
     * @param signData 验签数据
     * @return
     */
    public int RSASignVerifyWITHSHA1(XDJA_RSA_PUBKEY pubKey, byte[] dataIn, byte[] signData) {
        byte[] calcOut = new byte[signData.length];
        int[] calcOutLen = new int[1];
        int ret;
        ret = RSAPubKeyCalc(signData, signData.length, pubKey, calcOut, calcOutLen);
        if (ret != 0) {
            return ret;
        }

        long[] handle = new long[1];
        ret = SHA1Init(handle);
        if (ret != 0) {
            return ret;
        }

        ret = SHA1Update(handle[0], dataIn, dataIn.length);
        if (ret != 0) {
            return ret;
        }

        byte[] sha1Out = new byte[20];
        ret = SHA1Final(handle[0], sha1Out);
        if (ret != 0) {
            return ret;
        }

        byte[] pcksOut = new byte[pubKey.bits / 8];
        ret = PKCS1Padding(sha1Out, sha1Out.length, FLAG_PKCS1_PADDING_1, pubKey.bits, pcksOut);
        if (ret != 0) {
            return ret;
        }

        if (Arrays.equals(pcksOut, calcOut)) {
            return 0;
        }
        return -1;
    }

    //↑------------------------------RSA--------------------------------------↑
    //↑                                                                       ↑
    //↑------------------------------RSA--------------------------------------↑

    /**
     * AES加解密运算
     *
     * @param tmpkey     [in]加解密密钥
     * @param tmpkey_len [in]加解密密钥长度，支持16、24、32
     * @param mode       [in]运算模式,0x00表示ECB解密, 0x01表示ECB加密, 0x10 表示CBC解密, 0x11表示CBC加密
     * @param in         [in]待运算数据
     * @param in_len     [in]待运算数据长度，8的倍数
     * @param out        [out]运算得到的返回数据，长度等于in_len
     * @param iv         [in/out]运算IV，8个字节
     * @return XCR_OK正确，其他错误
     */
    public int AES(byte[] tmpkey, int tmpkey_len, int mode, byte[] in, int in_len, byte[] out, byte[] iv) {
        return crypto.AES(tmpkey, tmpkey_len, mode, in, in_len, out, iv);
    }

    /**
     * SM4对称算法(iv不传出)
     *
     * @param tmpkey  [in]密钥
     * @param flag    [in]加解密方式  0x00表示ECB解密, 0x01表示ECB加密, 0x10 表示CBC解密, 0x11表示CBC加密
     * @param datain  [in]输入数据
     * @param inlen   [in]输入数据长度
     * @param dataout [in,out]输出数据
     * @param iv      [in]iv
     * @return 0-成功   -1-失败
     */
    public int SM4(byte[] tmpkey, int flag, byte[] datain, int inlen, byte[] dataout, byte[] iv) {
        return crypto.SM4(tmpkey, flag, datain, inlen, dataout, iv);
    }

    /**
     * SM4对称算法（iv由接口传出）
     *
     * @param tmpkey  [in]		密钥
     * @param flag    [in]		加解密方式
     * @param datain  [in]		输入数据
     * @param inlen   [in]		输入数据长度
     * @param dataout [out]	    输出数据
     * @param iv      [in,out]	iv
     *                返回值：0-成功   -1-失败
     */
    public int SM4New(byte[] tmpkey, int flag, byte[] datain, int inlen, byte[] dataout, byte[] iv) {
        return crypto.SM4New(tmpkey, flag, datain, inlen, dataout, iv);
    }

    /**
     * 解析DER编码证书信息
     *
     * @param cert      [in]DER编码证书buffer
     * @param cert_len  [in]DER编码证书buffer
     * @param cert_info [out]解析后证书的详细信息
     * @return 没有任何返回值
     */
    public int GetCertInfo(byte[] cert, int cert_len, XCT_CERT_INFO cert_info) {
        return crypto.GetCertInfo(cert, cert_len, cert_info);
    }

    /**
     * PKCS1填充
     *
     * @param in      输入数据
     * @param inLen   输入数据长度
     * @param version 填充版本1或2；
     *                1：一般用来做签名时的填充（用私钥加密时）；
     *                2：一般用来做加密时的填充（用公钥加密时）；
     *                加密或填充时，用1或2也没有严格的区分。
     * @param bits    填充之后的比特数1028或2048
     * @param out     输出数据
     * @return
     */
    public int PKCS1Padding(byte[] in, int inLen, byte version, int bits, byte[] out) {
        return crypto.PKCS1Padding(in, inLen, version, bits, out);
    }

    /**
     * PKCS1去填充
     *
     * @param in     输入数据
     * @param inLen  输入数据长度
     * @param out    输出数据
     * @param outLen 输出数据长度
     * @return
     */
    public int PKCS1UnPadding(byte[] in, int inLen, byte[] out, int[] outLen) {
        return crypto.PKCS1UnPadding(in, inLen, out, outLen);
    }


    /**
     * 内部使用方法。
     * 用于替换原so库中的方法
     *
     * @param certByte
     * @param cert_len
     * @param cert_info
     * @return
     */
    private int selfGetCertInfo(byte[] certByte, int cert_len, XCT_CERT_INFO cert_info) {
        try {
            CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
            X509Certificate cert = (X509Certificate) certFactory.generateCertificate(
                    new ByteArrayInputStream(certByte, 0, cert_len));

            //issuerDn CN=XdjaSubCa_sm2, O=XinDaJieAn Corporation, C=cn
            //subjectDn CN=78646a6178646a613652313032156c4d, OU=tf, O=XinDaJieAn Corporation, C=cn
            String issuerDN = cert.getIssuerDN().toString();
            String subjectDN = cert.getSubjectDN().toString();

            byte[] tmpBytes;
            tmpBytes = getStr(subjectDN, "CN").getBytes();
            System.arraycopy(tmpBytes, 0, cert_info.subjectCN, 0, tmpBytes.length);

            System.arraycopy(subjectDN.getBytes(), 0, cert_info.subjectDN, 0, subjectDN.length());

            tmpBytes = getStr(subjectDN, "O").getBytes();
            System.arraycopy(tmpBytes, 0, cert_info.subjectOrg, 0, tmpBytes.length);

            tmpBytes = getStr(issuerDN, "CN").getBytes();
            System.arraycopy(tmpBytes, 0, cert_info.issuerCN, 0, tmpBytes.length);

            tmpBytes = getStr(issuerDN, "O").getBytes();
            System.arraycopy(tmpBytes, 0, cert_info.issuerOrg, 0, tmpBytes.length);

            tmpBytes = cert.getSerialNumber().toString(16).getBytes();
            System.arraycopy(tmpBytes, 0, cert_info.SN, 0, tmpBytes.length);

            tmpBytes = (cert.getNotAfter().getTime() + "").getBytes();
            System.arraycopy(tmpBytes, 0, cert_info.before, 0, tmpBytes.length);

            tmpBytes = (cert.getNotBefore().getTime() + "").getBytes();
            System.arraycopy(tmpBytes, 0, cert_info.after, 0, tmpBytes.length);

            try {
                PublicKey publicKey = cert.getPublicKey();
                String alg = publicKey.getAlgorithm();
                byte[] encoded = publicKey.getEncoded();
                if (alg.equals("RSA")) {
                    // RSA的公钥 0 到 21 是文件头，从22开始是； RSA
                    tmpBytes = Arrays.copyOfRange(encoded, 22, encoded.length);
                } else if (alg.equals("1.2.840.10045.2.1")) {
                    // SM2的公钥 0 到 25 是文件头，从26开始是;  1.2.840.10045.2.1
                    tmpBytes = Arrays.copyOfRange(encoded, 26, encoded.length);
                } else {
                    tmpBytes = null;
                }
            } catch (Exception e) {
                e.printStackTrace();

                //如果解析异常，则通过查找证书oid方式查找公钥信息
                int start;

                //rsa证书oid
                byte[] rsaHeader = {48, -127, -97, 48, 13, 6, 9, 42, -122, 72, -122, -9, 13, 1,
                        1, 1, 5, 0, 3, -127, -115, 0};
                start = indexOf(certByte, rsaHeader, 0);
                if (start > 0) {
                    start = start + rsaHeader.length;
                    tmpBytes = Arrays.copyOfRange(certByte, start, start + 140);
                } else {

                    //sm2证书oid
                    byte[] sm2Header = {48, 89, 48, 19, 6, 7, 42, -122, 72, -50, 61, 2, 1, 6, 8,
                            42, -127, 28, -49, 85, 1, -126, 45, 3, 66, 0};
                    start = indexOf(certByte, sm2Header, 0);
                    if (start > 0) {
                        start = start + sm2Header.length;
                        tmpBytes = Arrays.copyOfRange(certByte, start, start + 65);
                    }
                }
            }

            if (tmpBytes != null) {
                System.arraycopy(tmpBytes, 0, cert_info.pubkey, 0, tmpBytes.length);
                cert_info.setPubkeyLen(tmpBytes.length);
            } else {
                return -1;
            }
        } catch (Exception e) {
            e.printStackTrace();
            return -1;
        }
        return 0;
    }

    /**
     * 内部使用方法
     * 用于获取字符串中的 key 等于的值。
     * 比如:字符串 CN=78646a6178646a613652313032156c4d, OU=tf, O=XinDaJieAn Corporation, C=cn
     * 想要获取字符串中的CN对应的值，使用此方法后得到的值为 78646a6178646a613652313032156c4d
     *
     * @param str
     * @param key
     * @return
     */
    private String getStr(String str, String key) {
        if (TextUtils.isEmpty(str)) {
            return "";
        }
        str = str + ",";
        String result;
        int start = str.indexOf(key + "=");
        int end = str.indexOf(",", start);
        result = str.substring(start + key.length() + 1, end);
        return result;
    }

    /**
     * 从 origin 中 检索有没有 search
     *
     * @param origin 长数组
     * @param search 要检索的数组
     * @param start  从长数组中检索的位置
     * @return 大于0表示检索的有
     */
    private static int indexOf(byte[] origin, byte[] search, int start) {
        if (origin == null || origin.length == 0 || search == null || search.length == 0) {
            return -10;
        }
        int originLen = origin.length;
        int searchLen = search.length;
        if (originLen < searchLen) {
            return -1;
        }
        if ((start + searchLen) > originLen) {
            return -1;
        }
        int searchStart;
        searchStart = start;
        int result = -1;

        for (int i = searchStart; i <= (originLen - searchLen); i++) {
            int j = 0;
            for (; j < searchLen; j++) {
                if (origin[i + j] == search[j]) {
                    continue;
                } else {
                    break;
                }
            }
            if (j == searchLen) {
                result = i;
                break;
            }
        }
        return result;
    }

    /*
     * RSA Public Key Calculation.
     * notice:
     *		1 input/output size should be 128 when pubkey.bits is 1024, 256 when pubkey.bits is 2048.
     */
    public int XCF_RSAPubKeyCalc(byte[] input, int ilen,
                                 XDJA_RSA_PUBKEY pubkey, byte[] output, int[] olen) {
        return crypto.XCF_RSAPubKeyCalc(input, ilen, pubkey, output, olen);
    }

    /*
     * RSA Private Key Calculation.
     * notice:
     *		1 input/output size should be 128 when pubkey.bits is 1024, 256 when pubkey.bits is 2048.
     */
    public int XCF_RSAPriKeyCalc(byte[] input, int ilen,
                                 XDJA_RSA_PRIKEY prikey, byte[] output, int[] olen) {
        return crypto.XCF_RSAPriKeyCalc(input, ilen, prikey, output, olen);
    }

    /**
     * 根ca验证签发证书
     *
     * @param cabuf
     * @param calen
     * @param certbuf
     * @param certlen
     * @return
     */
    public int CaVerifyUerCert(byte[] cabuf, int calen, byte[] certbuf, int certlen) {
        return crypto.CaVerifyUerCert(cabuf, calen, certbuf, certlen);
    }

}
