package com.xdja.xdjacrypto;


public class XdjaCrypto {
    static {
        System.loadLibrary("xdjacrypto");
    }

    //卡内常见返回值
    /**
     * 成功
     */
    public static final int XCR_OK = 0;
    /**
     * 内存申请失败
     */
    public static final int XCR_MEM_MAC = -0x00000001;
    /**
     * 无效参数
     */
    public static final int XCR_INVALID_PARM = -0x00000002;
    /**
     * 算法不支持
     */
    public static final int XCR_NOT_SUPPORT = -0x00000003;
    /**
     * P12证书解析失败
     */
    public static final int XCR_PARSE_P12 = -0x00000004;
    public static final int XCR_PARSE_X509 = -0x00000005;

    //参数定义
    public static final int SM2_MAX_BITS = 512;
    public static final int SM2_MAX_LEN = SM2_MAX_BITS / 16;
    public static final int INITIATOR = 1;
    public static final int RESPONDER = 0;
    //对称算法表示
    public static final int ECB_DECRYPT = 0x00;
    public static final int ECB_ENCRYPT = 0x01;
    public static final int CBC_DECRYPT = 0x10;
    public static final int CBC_ENCRYPT = 0x11;

    /**
     * SM3运算初始化
     *
     * @param hHandle [out] 初始化产生的运算句柄，保存在hHandle[0]中
     * @return XCR_OK正确，其他错误
     */
    public native int SM3Init(long[] hHandle);

    /**
     * SM3运算过程
     *
     * @param handle [in] 运算句柄
     * @param datain [in] 待运算的数据块
     * @param in_len [in] 数据块长度
     * @return XCR_OK正确，其他错误
     */
    public native int SM3Update(long handle, byte[] datain, int in_len);

    /**
     * SM3运算结束，返回运算数据
     *
     * @param handle  [in]   运算句柄
     * @param dataout [out] 运算返回的数据，长度为32
     * @return XCR_OK正确，其他错误
     */
    public native int SM3Final(long handle, byte[] dataout);

    /**
     * AES加解密运算
     *
     * @param tmpkey     [in]     加解密密钥
     * @param tmpkey_len [in] 加解密密钥长度，支持16、24、32
     * @param mode       [in]       运算模式,0x00表示ECB解密, 0x01表示ECB加密, 0x10 表示CBC解密, 0x11表示CBC加密
     * @param in         [in]         待运算数据
     * @param in_len     [in]     待运算数据长度，8的倍数
     * @param out        [out]       运算得到的返回数据，长度等于in_len
     * @param iv         [in/out]     运算IV，8个字节
     * @return XCR_OK正确，其他错误
     */
    public native int AES(byte[] tmpkey, int tmpkey_len, int mode, byte[] in, int in_len, byte[] out, byte[] iv);

    /**
     * PKCS1填充
     *
     * @param in      输入数据
     * @param in_len  输入数据长度
     * @param version 填充版本1或2
     * @param bits    填充之后的比特数1028或2048
     * @param out     输出数据
     * @return
     */
    public native int PKCS1Padding(byte[] in, int in_len, byte version, int bits, byte[] out);

    /**
     * PKCS1去填充
     *
     * @param in      输入数据
     * @param in_len  输入数据长度
     * @param out     输出数据
     * @param out_len 输出数据长度
     * @return
     */
    public native int PKCS1UnPadding(byte[] in, int in_len, byte[] out, int[] out_len);

    /**
     * PKCS填充
     *
     * @param type      填充类型0或非0
     * @param in        输入数据
     * @param in_len    输入数据长度
     * @param block_len 填充块大小
     * @param out       输出数据
     * @param out_len   输出数据长度
     * @return
     */
    public native int PKCS5Padding(byte type, byte[] in, int in_len, int block_len, byte[] out, int[] out_len);

    /**
     * 解析DER编码证书信息
     *
     * @param cert      [in]       DER编码证书buffer
     * @param cert_len  [in]   DER编码证书buffer
     * @param cert_info [out] 解析后证书的详细信息
     * @return 没有任何返回值
     */
    public native int GetCertInfo(byte[] cert, int cert_len, XCT_CERT_INFO cert_info);

    /**
     * SM2签名
     *
     * @param pub     [in]         sm2签名公钥
     * @param priv    [in]       sm2签名私钥
     * @param message [in]			   需要签名的数据
     * @param mlen    [in]				   数据长度
     * @param value   [out]       签名结果
     *                返回值：0-成功   -1-失败
     */
    public native int SM2Signature(SM2PublicKey pub, SM2PrivateKey priv, byte[] message, int mlen, SM2Signature value);

    /**
     * SM2验签
     *
     * @param pub     [in]         sm2签名公钥
     * @param message [in]			   需要签名的数据
     * @param mlen    [in]				   数据长度
     * @param value   [in]       签名结果
     *                返回值：0-成功   -1-失败
     */
    public native int SM2Verify(SM2PublicKey pub, byte[] message, int mlen, SM2Signature value);

    /**
     * SM2公钥加密
     *
     * @param pub  [in]    	sm2公钥
     * @param from [in]             需要加密的数据
     * @param flen [in]				数据长度
     * @param to   [out]				加密的密文数据
     * @param tlen [out]				密文数据长度
     *             返回值：0-成功   -1-失败
     */
    public native int SM2PublicEncrypt(SM2PublicKey pub, byte[] from, int flen, byte[] to, int[] tlen);

    /**
     * SM2公钥加密（新国密标准）
     *
     * @param pub  [in]    	sm2公钥
     * @param from [in]             需要加密的数据
     * @param flen [in]				数据长度
     * @param to   [out]				加密的密文数据
     * @param tlen [out]				密文数据长度
     *             返回值：0-成功   -1-失败
     */
    public native int SM2PublicEncryptGM(SM2PublicKey pub, byte[] from, int flen, byte[] to, int[] tlen);

    /**
     * SM2私钥解密
     *
     * @param priv [in]    	sm2私钥
     * @param from [in]             需要解密的数据
     * @param flen [in]				数据长度
     * @param to   [out]				解密的明文数据
     * @param tlen [out]				明文数据长度
     *             返回值：0-成功   -1-失败
     */
    public native int SM2PrivateDecrypt(SM2PrivateKey priv, byte[] from, int flen, byte[] to, int[] tlen);

    /**
     * SM2私钥解密（新国密标准）
     *
     * @param priv [in]    	sm2私钥
     * @param from [in]             需要解密的数据
     * @param flen [in]				数据长度
     * @param to   [out]				解密的明文数据
     * @param tlen [out]				明文数据长度
     *             返回值：0-成功   -1-失败
     */
    public native int SM2PrivateDecryptGM(SM2PrivateKey priv, byte[] from, int flen, byte[] to, int[] tlen);

    /**
     * SM2生成临时公钥和私钥
     *
     * @param tpub  [out]    	sm2临时公钥
     * @param tpriv [out]		sm2临时私钥
     *              返回值：0-成功   -1-失败
     */
    public native int SM2ExchangeRandom(SM2PublicKey tpub, SM2PrivateKey tpriv);

    /**
     * SM2密钥协商，生成共享会话密钥
     *
     * @param mytpriv  [in]	用SM2_exchange_random()请求方临时私钥
     * @param mytpub   [in]		用SM2_exchange_random()请求方临时公钥
     * @param peertpub [in]	用SM2_exchange_random()应答方临时私钥
     * @param mypub    [in]		请求方的公钥
     * @param mypriv   [in]	请求方的私钥
     * @param peerpub  [in]	应答方的公钥
     * @param keylen   [in]				共享会话密钥长度
     * @param role     [in]				角色RESPONDER或INITIATOR
     * @param key      [out]				共享会话密钥
     * @param hash     [out]			哈希验证标识
     *                 返回值：0-成功   -1-失败
     */
    public native int SM2ExchangeCale(SM2PrivateKey mytpriv, SM2PublicKey mytpub, SM2PublicKey peertpub,
                                      SM2PublicKey mypub, SM2PrivateKey mypriv, SM2PublicKey peerpub, int keylen, int role, byte[] key, oct_string hash);

    /**
     * SM4对称算法(iv不传出)
     *
     * @param tmpkey  [in]		密钥
     * @param flag    [in]		加解密方式
     * @param datain  [in]		输入数据
     * @param inlen   [in]		输入数据长度
     * @param dataout [in]	输出数据
     * @param iv      [in]			iv
     *                返回值：0-成功   -1-失败
     */
    public native int SM4(byte[] tmpkey, int flag, byte[] datain, int inlen, byte[] dataout, byte[] iv);

    /**
     * SM4对称算法（iv由接口传出）
     *
     * @param tmpkey  [in]		密钥
     * @param flag    [in]		加解密方式
     * @param datain  [in]		输入数据
     * @param inlen   [in]		输入数据长度
     * @param dataout [in]	    输出数据
     * @param iv      [in,out]	iv
     *                返回值：0-成功   -1-失败
     */
    public native int SM4New(byte[] tmpkey, int flag, byte[] datain, int inlen, byte[] dataout, byte[] iv);

    /**
     * SM2密钥协商验证
     * 入参：
     *
     * @param s1 [in]			请求方验证标识
     * @param s2 [in]			应答方验证标识
     *           出参：
     *           返回值：true-成功   false-失败
     */
    public boolean SM2ExchangeVerify(oct_string s1, oct_string s2) {
        boolean ret = true;
        for (int i = 0; i < XdjaCrypto.SM2_MAX_LEN; i++) {
            if (s1.data[i] != s2.data[i]) {
                ret = false;
            }
        }
        return ret;
    }

    /**
     * SHA1运算初始化
     *
     * @param hHandle [out] 初始化产生的运算句柄，保存在hHandle[0]中
     * @return XCR_OK正确，其他错误
     */
    public native int SHA1Init(long[] hHandle);

    /**
     * SHA1运算过程
     *
     * @param handle [in] 运算句柄
     * @param datain [in] 待运算的数据块
     * @param in_len [in] 数据块长度
     * @return XCR_OK正确，其他错误
     */
    public native int SHA1Update(long handle, byte[] datain, int in_len);

    /**
     * SHA1运算结束，返回运算数据
     *
     * @param handle  [in]   运算句柄
     * @param dataout [out] 运算返回的数据，长度为20
     * @return XCR_OK正确，其他错误
     */
    public native int SHA1Final(long handle, byte[] dataout);

    /**
     * RSA Gen Key Pair
     *
     * @param bits
     * @param pubkey
     * @param prikey
     * @return XCR_OK success, other error
     */
    public native int XCF_GenRSAKeyPair(int bits, XDJA_RSA_PUBKEY pubkey,
                                        XDJA_RSA_PRIKEY prikey);

    /**
     * RSA Public Key Calculation.
     *
     * @param input
     * @param ilen
     * @param pubkey
     * @param output
     * @param olen
     * @return XCR_OK success, other error
     * notice:
     * 1 input/output size should be 128 when pubkey.bits is 1024, 256 when pubkey.bits is 2048.
     */
    public native int XCF_RSAPubKeyCalc(byte[] input, int ilen,
                                        XDJA_RSA_PUBKEY pubkey, byte[] output, int[] olen);

    /**
     * RSA Private Key Calculation.
     *
     * @param input
     * @param ilen
     * @param prikey
     * @param output
     * @param olen
     * @return XCR_OK success, other error
     * notice:
     * 1 input/output size should be 128 when pubkey.bits is 1024, 256 when pubkey.bits is 2048.
     */
    public native int XCF_RSAPriKeyCalc(byte[] input, int ilen,
                                        XDJA_RSA_PRIKEY prikey, byte[] output, int[] olen);

    /**
     * @param password        口令
     * @param plen            口令长度
     * @param salt            盐值
     * @param slen            salt长度
     * @param iteration_count 迭代次数
     * @param key_length      派生密钥长度
     * @param output          派生密钥
     * @return 错误码
     * @brief　密钥派生
     * @retval 0 成功
     */
    public native int PKCS5_PBKDF2_HMAC(byte[] password, int plen, byte[] salt, int slen, int iteration_count, int key_length, byte[] output);

    /**
     * @return 错误码
     * @retval 0 成功
     */
    public native int CaVerifyUerCert(byte[] cabuf, int calen, byte[] certbuf, int certlen);
}
