package com.xdja.fwzxutils.transform;

import android.text.TextUtils;

import com.xdja.fwzxutils.base.BaseHttpRequest;
import com.xdja.fwzxutils.base.BaseInfoUtils;
import com.xdja.fwzxutils.bean.ResourceQueryBean;
import com.xdja.fwzxutils.common.Constant;
import com.xdja.fwzxutils.exception.ArmsException;
import com.xdja.fwzxutils.exception.TimeoutException;
import com.xdja.fwzxutils.utils.CErrorInfo;
import com.xdja.gprsconverter.convert.ConvertFactory;
import com.xdja.gprsconverter.protocol.ChangePwdRequest;
import com.xdja.gprsconverter.protocol.EntryRequest;
import com.xdja.gprsconverter.protocol.GetDataSourceRequest;
import com.xdja.gprsconverter.protocol.LoginRequest;
import com.xdja.gprsconverter.protocol.LoginResponse;
import com.xdja.gprsconverter.protocol.LogoutRequest;
import com.xdja.gprsconverter.protocol.QueryRequest;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;

import io.reactivex.Observable;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.ObservableSource;
import io.reactivex.annotations.NonNull;
import io.reactivex.functions.Function;

/**
 * @author yuanwanli
 * @des Gprs协议对接服务总线转换类
 * @date 2020/9/22
 */
public class GprsTransform {

    /**
     * 请求警员power
     *
     * @return 权限集合
     */
    private Observable<List<String>> getPowerFromServiceBus(String idCard, String powerId) {
        ResourceQueryBean queryBean = ResourceQueryBean.getQueryCommonBean(idCard, powerId, Constant.XML_REQ);
        return BaseHttpRequest.getInstance()
                .getListOfType(queryBean, Constant.XML_DATA, String.class)
                .onErrorReturnItem(new ArrayList<>());
    }

    /**
     * 获取警员信息
     *
     * @param loginRequest 请求入参
     * @return Observable<LoginResponse>
     */
    public Observable<LoginResponse> getPoliceInfo(@NonNull LoginRequest loginRequest, String idCard) {
        String gprsLoginResId = BaseInfoUtils.getGprsLoginResId();
        String powerId = BaseInfoUtils.getGprsPowerId();
        Observable<LoginResponse> loginResponseObservable = getGprsRequest(gprsLoginResId, loginRequest, LoginResponse.class);
        if (!TextUtils.isEmpty(powerId)) {
            Observable<List<String>> powerFromServiceBus = getPowerFromServiceBus(idCard, powerId);
            return Observable.zip(loginResponseObservable, powerFromServiceBus, (loginResponse, powers) -> {
                StringBuilder stringBuilder = new StringBuilder();
                for (String power : powers) {
                    stringBuilder.append(power).append("#");
                }
                loginResponse.setPower(stringBuilder.toString());
                return loginResponse;
            });
        } else {
            return loginResponseObservable;
        }
    }

    /**
     * 通用grps请求对接服务总线
     *
     * @param resourceId         接口id
     * @param requestType        入参
     * @param classOfT           返回类型的class
     * @param <RequestType>      入参类型
     * @param <ResponseType>出参类型
     * @return Observable<ResponseType>
     */
    public <RequestType, ResponseType> Observable<ResponseType> getGprsRequest(@NonNull String resourceId, @NonNull RequestType requestType, @NonNull Class<ResponseType> classOfT) {
        return Observable.create((ObservableOnSubscribe<ResourceQueryBean>) emitter -> {
            String xml = "";
            if (BaseInfoUtils.controlId != null && BaseInfoUtils.controlName != null) {
                xml = generateXML(requestType);
            } else {
                xml = ConvertFactory.generateXML(requestType);
            }
            String reqType = getReqType(xml);
            ResourceQueryBean queryBean = ResourceQueryBean.getQueryCommonBean(xml, resourceId, Constant.XML_REQ);
            List<String> heads = new ArrayList<>();
            heads.add("controlId : " + BaseInfoUtils.controlId);
            heads.add("controlName : " + BaseInfoUtils.controlName);
            heads.add("reqType : " + reqType);
            heads.add("messageId : " + queryBean.getMessageId());
            queryBean.setHeads(heads);
            emitter.onNext(queryBean);
            emitter.onComplete();
        }).flatMap((Function<ResourceQueryBean, ObservableSource<String>>) queryBean -> BaseHttpRequest.getInstance().getResponse(queryBean, Constant.XML_DATA)
        ).flatMap((Function<String, ObservableSource<ResponseType>>) response ->
                Observable.create((ObservableOnSubscribe<ResponseType>) emitter -> {
                    try {
                        // 解析返回结果
                        CErrorInfo.parseErrorMsg(response);
                        if (CErrorInfo.isError && !CErrorInfo.message.contains("修改密码成功")) {
                            if (response != null && response.startsWith("0x00000000")) {
                                //查询成功，但未找到记录，此种情况不算错误
                                ResponseType responseType = classOfT.newInstance();
                                emitter.onNext(responseType);
                                emitter.onComplete();
                            } else {
                                ArmsException throwable = new ArmsException(CErrorInfo.code, CErrorInfo.message);
                                emitter.onError(throwable);
                                return;
                            }
                            //超时处理
                            if ("0x13014034".equals(CErrorInfo.code) || CErrorInfo.message.contains("登陆超时")) {
                                emitter.onError(new TimeoutException());
                            }
                        } else {
                            if (response.contains("00 未找到记录")) {
                                ResponseType responseType = classOfT.newInstance();
                                emitter.onNext(responseType);
                            } else {
                                emitter.onNext(ConvertFactory.parserResponse(response, classOfT));
                            }
                            emitter.onComplete();
                        }
                    } catch (Throwable throwable) {
                        emitter.onError(throwable);
                    }
                }));
    }

    private String getReqType(String reqStr) {
        String reqType = "";
        try {
            XmlPullParser xmlPullParser = XmlPullParserFactory.newInstance().newPullParser();
            xmlPullParser.setInput(new StringReader(reqStr));
            int eventType = xmlPullParser.getEventType();
            while (eventType != XmlPullParser.END_DOCUMENT) {
                if (eventType == XmlPullParser.START_TAG) {
                    String name = xmlPullParser.getName();
                    if (name.equals("ReqType")) {
                        reqType = xmlPullParser.nextText();
                        break;
                    }
                }
                eventType = xmlPullParser.next();
            }
        } catch (XmlPullParserException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return reqType;
    }

    private <T> String generateXML(T t) {
        if (t instanceof QueryRequest) {
            ((QueryRequest) t).setAppKey(BaseInfoUtils.controlId);
            ((QueryRequest) t).setReqDigest(BaseInfoUtils.controlName);
            return ConvertFactory.generateQueryXML((QueryRequest) t);
        } else if (t instanceof EntryRequest) {
            ((EntryRequest) t).setAppKey(BaseInfoUtils.controlId);
            ((EntryRequest) t).setReqDigest(BaseInfoUtils.controlName);
            return ConvertFactory.generateEntryXML((EntryRequest) t);
        } else if (t instanceof LoginRequest) {
            ((LoginRequest) t).setAppKey(BaseInfoUtils.controlId);
            ((LoginRequest) t).setReqDigest(BaseInfoUtils.controlName);
            return ConvertFactory.generateLoginXML((LoginRequest) t);
        } else if (t instanceof LogoutRequest) {
            ((LogoutRequest) t).setAppKey(BaseInfoUtils.controlId);
            ((LogoutRequest) t).setReqDigest(BaseInfoUtils.controlName);
            return ConvertFactory.generateLogoutXML((LogoutRequest) t);
        } else if (t instanceof ChangePwdRequest) {
            ((ChangePwdRequest) t).setAppKey(BaseInfoUtils.controlId);
            ((ChangePwdRequest) t).setReqDigest(BaseInfoUtils.controlName);
            return ConvertFactory.generateChangePwdXML((ChangePwdRequest) t);
        } else if (t instanceof GetDataSourceRequest) {
            ((GetDataSourceRequest) t).setAppKey(BaseInfoUtils.controlId);
            ((GetDataSourceRequest) t).setReqDigest(BaseInfoUtils.controlName);
            return ConvertFactory.generateGetDataSourceXML((GetDataSourceRequest) t);
        } else if (t instanceof String) {
            return (String) t;
        }
        return null;
    }
}











