package com.xdja.fwzxutils.utils;

import com.xdja.fwzxutils.base.BaseInfoUtils;
import com.xdja.fwzxutils.base.BaseLoginUtils;
import com.xdja.fwzxutils.bean.CredentialBean;
import com.xdja.fwzxutils.bean.LoginBean;
import com.xdja.fwzxutils.bean.ResourceQueryBean;
import com.xdja.fwzxutils.bean.ResourceResponseBean;
import com.xdja.fwzxutils.bean.UserCredentialBean;
import com.xdja.fwzxutils.transform.GprsTransform;
import com.xdja.fwzxutils.transform.HttpTransform;
import com.xdja.gprsconverter.protocol.LoginRequest;
import com.xdja.gprsconverter.protocol.LoginResponse;

import java.util.List;
import java.util.concurrent.TimeUnit;

import io.reactivex.Observable;
import io.reactivex.ObservableSource;
import io.reactivex.annotations.NonNull;
import io.reactivex.functions.Function;
import io.reactivex.schedulers.Schedulers;

/**
 * @author yuanwanli
 * @des 服务总线工具类
 * @date 2020/9/27
 */
public class ServiceBusUtils {
    /**
     * 静态内部类实现单例
     *
     * @return ServiceBusLoginUtils
     */
    public static ServiceBusUtils getInstance() {
        return Holder.singleton;
    }

    /**
     * 不需要gprs的登录
     *
     * @return LoginBean 包含用户凭证,应用凭证以及寻址的资源列表
     * {@link LoginBean#getUserInfo()}可以拿到警号,身份证号，姓名，userId，orgId
     * {@link UserCredentialBean.CredentialBean.LoadBean.UserInfoBean}
     */
    public Observable<LoginBean> login() {
        BaseLoginUtils loginUtils = new BaseLoginUtils();
        return loginUtils.getCredential()
                .concatMap((Function<CredentialBean, ObservableSource<LoginBean>>) credentialBean ->
                        loginUtils.getResourceAddress(credentialBean.getAppCredential()).map(resourceBeans -> {
                            LoginBean loginBean = new LoginBean();
                            loginBean.setAppCredential(credentialBean.getAppCredential());
                            loginBean.setUserCredential(credentialBean.getUserCredential());
                            loginBean.setResourceBeanList(resourceBeans);
                            return loginBean;
                        }))
                .subscribeOn(Schedulers.io());
    }

    /**
     * 需要gprs的登录
     *
     * @param loginRequest gprs登录请求的bean
     * @return LoginResponse gprs登录返回的bean
     */
    public Observable<LoginResponse> getGprsLogin(LoginRequest loginRequest) {
        GprsTransform gprsTransform = new GprsTransform();
        return login()
                .concatMap((Function<LoginBean, ObservableSource<LoginResponse>>) loginBean -> gprsTransform.getPoliceInfo(loginRequest, loginBean.getUserInfo().getSfzh()))
                .timeout(60, TimeUnit.SECONDS)
                .subscribeOn(Schedulers.io());
    }

    /**
     * gprs业务请求接口
     *
     * @param requestType    入参请求bean
     * @param classOfT       出参的class
     * @param <RequestType>  入参泛型
     * @param <ResponseType> 出参泛型
     * @return Observable<ResponseType>
     */
    public <RequestType, ResponseType> Observable<ResponseType> getGprsBusiness(@NonNull RequestType requestType, @NonNull Class<ResponseType> classOfT) {
        GprsTransform gprsTransform = new GprsTransform();
        return gprsTransform.getGprsRequest(BaseInfoUtils.getGprsBusinessResId(), requestType, classOfT).subscribeOn(Schedulers.io());
    }

    /**
     * 一般的json请求
     * 入参出参tag跟后台协商为"req_json"，"res_json"
     *
     * @param resourceId     接口id
     * @param jsonObject     入参bean
     * @param classOfT       出参class
     * @param <ResponseType> 出参泛型
     * @return Observable<ResponseType>
     */
    public <ResponseType> Observable<ResponseType> getJsonRequest(@NonNull String resourceId, @NonNull Object jsonObject, @NonNull Class<ResponseType> classOfT) {
        HttpTransform httpTransform = new HttpTransform();
        return httpTransform.getJsonRequest(resourceId, jsonObject, classOfT).subscribeOn(Schedulers.io());
    }

    /**
     * 一般的json请求 自定义入参出参tag
     *
     * @param resourceId     接口id
     * @param requestTag     入参tag
     * @param responseTag    出参tag
     * @param jsonObject     入参bean
     * @param classOfT       出参class
     * @param <ResponseType> 出参泛型
     * @return Observable<ResponseType>
     */
    public <ResponseType> Observable<ResponseType> getJsonRequestWithTag(@NonNull String resourceId, @NonNull String requestTag, @NonNull String responseTag, @NonNull Object jsonObject, @NonNull Class<ResponseType> classOfT) {
        HttpTransform httpTransform = new HttpTransform();
        return httpTransform.getJsonRequestWithTag(resourceId, requestTag, responseTag, jsonObject, classOfT).subscribeOn(Schedulers.io());
    }

    /**
     * 按照服务总线标准协议对接的方式(不建议)
     * 标准方式是传参的时候,入参参数封装{@link ResourceQueryBean.ParameterBean.ConditionBean.KeyValueListBean}
     *
     * @param resourceQueryBean 入参bean
     * @return Observable<List < ResourceResponseBean.DataBean.DataListBean>>
     */
    public Observable<List<ResourceResponseBean.DataBean.DataListBean>> requestThirdApi(ResourceQueryBean resourceQueryBean) {
        HttpTransform httpTransform = new HttpTransform();
        return httpTransform.requestThirdApi(resourceQueryBean).subscribeOn(Schedulers.io());
    }

    private static class Holder {
        private static ServiceBusUtils singleton = new ServiceBusUtils();
    }
}
