package com.xdja.fwzxutils.base;

import android.text.TextUtils;

import com.blankj.utilcode.util.CacheDoubleStaticUtils;
import com.blankj.utilcode.util.GsonUtils;
import com.xdja.fwzxutils.bean.ResourceQueryBean;
import com.xdja.fwzxutils.bean.ResourceResponseBean;
import com.xdja.fwzxutils.exception.ArmsException;
import com.xdja.fwzxutils.exception.TimeoutException;
import com.xdja.fwzxutils.net.RetrofitFactory;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

import io.reactivex.Observable;
import io.reactivex.ObservableSource;
import io.reactivex.annotations.NonNull;
import io.reactivex.functions.Function;

/**
 * @author yuanwanli
 * @des
 * @date 2020/9/22
 */
public class BaseHttpRequest {
    private BaseHttpRequest() {
    }

    /**
     * 静态内部类实现单例
     *
     * @return ServiceBusLoginUtils
     */
    public static BaseHttpRequest getInstance() {
        return BaseHttpRequest.Holder.singleton;
    }

    /**
     * 请求集合数据
     *
     * @param queryBean      请求bean
     * @param tag            终端后台定义用来解析的标识
     * @param classOfT       数据类型class
     * @param <ResponseType> 数据bean
     * @return List<Bean>
     */
    public <ResponseType> Observable<List<ResponseType>> getListOfType(@NonNull ResourceQueryBean queryBean, @NonNull String tag, @NonNull Class<ResponseType> classOfT) {
        return getResponse(queryBean, tag)
                .flatMap((Function<String, ObservableSource<List<ResponseType>>>) response -> Observable.create(e -> {
                    try {
                        Type listType = GsonUtils.getListType(classOfT);
                        List<ResponseType> powers = GsonUtils.fromJson(response, listType);
                        if (powers == null) {
                            e.onError(new Throwable("json解析字段失败"));
                        } else {
                            e.onNext(powers);
                            e.onComplete();
                        }
                    } catch (Throwable throwable) {
                        e.onError(throwable);
                    }
                }));
    }

    /**
     * @param queryBean 请求bean
     * @return 终端根据需要自行解析服务总线返回数据
     */
    public Observable<List<ResourceResponseBean.DataBean.DataListBean>> getResponse(@NonNull ResourceQueryBean queryBean) {
        return Observable.create(e -> {
            try {
                String resourceId = queryBean.getParameter().getDataObjId();
                String url = CacheDoubleStaticUtils.getString(resourceId);
                queryBean.getHeads().add("resourceId : " + resourceId);
                queryBean.getHeads().add("url : " + url);
                if (url == null || url.isEmpty()) {
                    String errorMessage = "联网请求失败,请求地址为空";
                    DefaultFormatPrinter.printRequest(queryBean, errorMessage, "");
                    e.onError(new Throwable("联网请求失败,请求地址为空"));
                    return;
                }
                // 执行网络请求
                ResourceResponseBean responseBean = RetrofitFactory.getInstance().getServiceBusApi().getRequest(url, queryBean).execute().body();
                if (responseBean == null) {
                    String errorMessage = "联网请求失败,ResourceResponseBean为空";
                    DefaultFormatPrinter.printResponse(null, null, errorMessage, null);
                    e.onError(new Throwable(errorMessage));
                } else {
                    String messageId = responseBean.getMessageId();
                    List<String> heads = new ArrayList<>();
                    heads.add("messageId : " + messageId);
                    if (responseBean.isSuccess()) {
                        DefaultFormatPrinter.printResponse(heads, GsonUtils.toJson(responseBean.getData().getDataList()), null, null);
                        e.onNext(responseBean.getData().getDataList());
                        e.onComplete();
                    } else {
                        String errorMessage = responseBean.getMessage();
                        DefaultFormatPrinter.printResponse(null, null, errorMessage, null);
                        if (responseBean.isTimeout()){
                            e.onError(new TimeoutException());
                        }else {
                            e.onError(new ArmsException(responseBean.getCode(), errorMessage));
                        }
                    }
                }
            } catch (Throwable throwable) {
                e.onError(throwable);
            }
        });
    }

    /**
     * @param queryBean 请求bean
     * @param tag       前后台标识
     * @return 终端实际需要的String字符串(可为json或者xml)
     */
    public Observable<String> getResponse(@NonNull ResourceQueryBean queryBean, @NonNull String tag) {
        return Observable.create(e -> {
            try {
                String resourceId = queryBean.getParameter().getDataObjId();
                if (TextUtils.isEmpty(resourceId)) {
                    String errorMessage = "resourceId为空";
                    DefaultFormatPrinter.printRequest(queryBean, errorMessage, tag);
                    e.onError(new Throwable(errorMessage));
                    return;
                }
                String url = CacheDoubleStaticUtils.getString(resourceId);
                queryBean.getHeads().add("messageId : " + queryBean.getMessageId());
                queryBean.getHeads().add("resourceId : " + resourceId);
                queryBean.getHeads().add("url : " + url);
                if (url == null || url.isEmpty()) {
                    String errorMessage = "联网请求失败,请求地址为空";
                    DefaultFormatPrinter.printRequest(queryBean, errorMessage, tag);
                    e.onError(new Throwable(errorMessage));
                    return;
                }
                DefaultFormatPrinter.printRequest(queryBean, null, tag);
                // 执行网络请求
                ResourceResponseBean responseBean = RetrofitFactory.getInstance().getServiceBusApi().getRequest(url, queryBean).execute().body();
                if (responseBean == null) {
                    String errorMessage = "联网请求失败,ResourceResponseBean为空";
                    DefaultFormatPrinter.printResponse(null, null, errorMessage, tag);
                    e.onError(new Throwable(errorMessage));
                } else {
                    String messageId = responseBean.getMessageId();
                    List<String> heads = new ArrayList<>();
                    heads.add("messageId : " + messageId);
                    if (responseBean.isSuccess()) {
                        List<ResourceResponseBean.DataBean.DataListBean> dataList = responseBean.getData().getDataList();
                        String response = null;
                        if (dataList != null && dataList.size() > 0) {
                            List<ResourceResponseBean.DataBean.DataListBean.FieldValuesBean> fieldValues = dataList.get(0).getFieldValues();
                            for (ResourceResponseBean.DataBean.DataListBean.FieldValuesBean bean : fieldValues) {
                                if (tag.equals(bean.getField())) {
                                    response = bean.getValue();
                                    break;
                                }
                            }
                            if (response == null) {
                                String errorMessage = "解析字段失败,约定返回为空";
                                DefaultFormatPrinter.printResponse(null, null, errorMessage, tag);
                                e.onError(new Throwable(errorMessage));
                            } else {
                                DefaultFormatPrinter.printResponse(heads, response, null, tag);
                                e.onNext(response);
                                e.onComplete();
                            }
                        } else {
                            String errorMessage = "解析字段失败,dataList字段为空";
                            DefaultFormatPrinter.printResponse(null, null, errorMessage, tag);
                            e.onError(new Throwable(errorMessage));
                        }
                    } else {
                        String errorMessage = responseBean.getMessage();
                        DefaultFormatPrinter.printResponse(null, null, errorMessage, tag);
                        if (responseBean.isTimeout()){
                            e.onError(new TimeoutException());
                        }else {
                            e.onError(new ArmsException(responseBean.getCode(), errorMessage));
                        }
                    }
                }
            } catch (Throwable throwable) {
                e.onError(throwable);
            }
        });
    }

    private static class Holder {
        private static BaseHttpRequest singleton = new BaseHttpRequest();
    }
}
