package com.xdja.fwzxutils.net.cache;

import android.support.annotation.Nullable;

import java.util.LinkedHashMap;
import java.util.Map;

public class LruCache<K, V> implements Cache<K, V> {
    private final LinkedHashMap<K, V> cache = new LinkedHashMap(100, 0.75F, true);
    private final int initialMaxSize;
    private int maxSize;
    private int currentSize = 0;

    public LruCache(int size) {
        this.initialMaxSize = size;
        this.maxSize = size;
    }

    public synchronized void setSizeMultiplier(float multiplier) {
        if (multiplier < 0.0F) {
            throw new IllegalArgumentException("Multiplier must be >= 0");
        } else {
            this.maxSize = Math.round((float) this.initialMaxSize * multiplier);
            this.evict();
        }
    }

    protected int getItemSize(V item) {
        return 1;
    }

    protected void onItemEvicted(K key, V value) {
    }

    @Override
    public synchronized int getMaxSize() {
        return this.maxSize;
    }

    @Override
    public synchronized int size() {
        return this.currentSize;
    }

    @Override
    public synchronized boolean containsKey(K key) {
        return this.cache.containsKey(key);
    }

    @Override
    @Nullable
    public synchronized V get(K key) {
        return this.cache.get(key);
    }

    @Override
    @Nullable
    public synchronized V put(K key, V value) {
        int itemSize = this.getItemSize(value);
        if (itemSize >= this.maxSize) {
            this.onItemEvicted(key, value);
            return null;
        } else {
            V result = this.cache.put(key, value);
            if (value != null) {
                this.currentSize += this.getItemSize(value);
            }

            if (result != null) {
                this.currentSize -= this.getItemSize(result);
            }

            this.evict();
            return result;
        }
    }

    @Override
    @Nullable
    public synchronized V remove(K key) {
        V value = this.cache.remove(key);
        if (value != null) {
            this.currentSize -= this.getItemSize(value);
        }

        return value;
    }

    @Override
    public void clear() {
        this.trimToSize(0);
    }

    protected synchronized void trimToSize(int size) {
        while (this.currentSize > size) {
            Map.Entry<K, V> last = (Map.Entry) this.cache.entrySet().iterator().next();
            V toRemove = last.getValue();
            this.currentSize -= this.getItemSize(toRemove);
            K key = last.getKey();
            this.cache.remove(key);
            this.onItemEvicted(key, toRemove);
        }

    }

    private void evict() {
        this.trimToSize(this.maxSize);
    }
}
