package com.xdja.servicebus.provider;

import android.content.ContentProvider;
import android.content.ContentValues;
import android.content.UriMatcher;
import android.database.Cursor;
import android.net.Uri;
import android.os.Bundle;
import android.os.NetworkOnMainThreadException;
import android.text.TextUtils;
import android.util.Log;

import com.google.gson.Gson;
import com.xdja.servicebus.http.RetrofitFactory;
import com.xdja.servicebus.http.entity.RequestUrlEntity;
import com.xdja.servicebus.http.entity.UrlEntity;
import com.xdja.servicebus.utils.UrlConstant;

import java.io.EOFException;
import java.io.UnsupportedEncodingException;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import okhttp3.Headers;
import retrofit2.Call;
import retrofit2.Response;

/**
 * @author yuanwanli
 * @des 服务总线的内容提供者
 * 由于需要异步联网，故采用insert插入数据，联网请求成功使用
 * contentObserver通知调用端再query
 * @date 2019/4/9
 */
public class ServiceBusProvider extends ContentProvider {
    private final static String AUTHORITY = "com.ydjw.rsb.getResourceAddress";
    private static UriMatcher mUriMatcher;
    private static final int URL_DIR = 0;
    /**
     * 接口版本号
     */
    private static final String VERSION = "version";
    /**
     * 接口唯一标识
     */
    private static final String MESSAGE_ID = "messageId";
    /**
     * 应用凭证
     */
    private static final String APP_CREDENTIAL = "appCredential";

    /**
     * 利用静态代码块初始化UriMatcher
     * 在UriMatcher中包含了多个Uri,每个Uri代表一种操作
     * 当调用UriMatcher.match(Uri uri)方法时就会返回该uri对应的code;
     * 比如此处的URL_DIR
     */
    static {
        mUriMatcher = new UriMatcher(UriMatcher.NO_MATCH);
        mUriMatcher.addURI(AUTHORITY, null, URL_DIR);
    }

    @Override
    public boolean onCreate() {
        Log.d("ServiceBusProvider", "onCreate");
        return true;
    }

    @Override
    public Cursor query(Uri uri, String[] projection, String selection, String[] selectionArgs, String sortOrder) {
        return null;
    }

    /**
     * 在自定义ContentProvider中必须覆写getType(Uri uri)方法.
     * 该方法用于获取Uri对象所对应的MIME类型.
     * <p>
     * 一个Uri对应的MIME字符串遵守以下三点:
     * 1 必须以vnd开头
     * 2 如果该Uri对应的数据可能包含多条记录,那么返回字符串应该以"vnd.android.cursor.dir/"开头
     * 3 如果该Uri对应的数据只包含一条记录,那么返回字符串应该以"vnd.android.cursor.item/"开头
     */
    @Override
    public String getType(Uri uri) {
        throw new IllegalArgumentException("unknown uri" + uri.toString());
    }

    @Override
    public Uri insert(final Uri uri, ContentValues values) {
        return uri;
    }

    @Override
    public int delete(Uri uri, String selection, String[] selectionArgs) {
        return 0;
    }

    @Override
    public int update(Uri uri, ContentValues values, String selection, String[] selectionArgs) {
        return 0;
    }

    @Override
    public synchronized Bundle call(String method, String arg, final Bundle extras) {
        final CountDownLatch downLatch = new CountDownLatch(1);
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final Bundle bundle = new Bundle();
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                String version = extras.getString(VERSION);
                String appCredential = extras.getString(APP_CREDENTIAL);
                String messageId = extras.getString(MESSAGE_ID);
                int resultVersion = 0;

                try {
                    appCredential = URLEncoder.encode(appCredential, "UTF-8");
                } catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                    bundle.putInt("version", resultVersion);
                    bundle.putString("message", "应用凭证 encode失败:" + e.getMessage());
                    bundle.putString("messageId", messageId);
                    bundle.putInt("resultCode", 41002);
                    bundle.putString("resourceList", null);
                }
                List<RequestUrlEntity> list = new ArrayList<>();
                String url = UrlConstant.getRsbSearchResourceUrl();
                Call<List<UrlEntity>> call = RetrofitFactory.getInstance()
                        .getUrlConverterApi()
                        .queryResourceAddress(url, appCredential, messageId, version, list);

                try {
                    resultVersion = Integer.parseInt(version);
                    Response<List<UrlEntity>> execute = call.execute();
                    bundle.putInt("version", resultVersion);
                    if (execute.isSuccessful()) {
                        /*请求成功*/
                        Headers headers = execute.headers();
                        String code = headers.get("code");
                        bundle.putInt("resultCode", Integer.parseInt(code));
                        String resultMessageId = headers.get("messageId");
                        bundle.putString("messageId", resultMessageId);
                        String message = headers.get("message");
                        bundle.putString("message", message);

                        if ("0".equals(code)) {
                            /*请求返回成功*/
                            List<UrlEntity> body = execute.body();
                            if (body != null) {
                                String json = new Gson().toJson(body);
                                String result = json.replace("regionalismCode", "resourceRegionalismCode");
                                bundle.putString("resourceList", result);
                            } else {
                                bundle.putString("resourceList", null);
                            }
                        } else {
                            /*请求成功，返回失败*/
                            bundle.putString("resourceList", null);
                        }
                    } else {
                        /*请求失败*/
                        bundle.putString("message", "服务器请求失败,请检查网络或者sod服务:" + UrlConstant.getRsbSearchResourceUrl());
                        bundle.putInt("resultCode", 41003);
                        bundle.putString("messageId", messageId);
                        bundle.putString("resourceList", null);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    bundle.putInt("version", resultVersion);
                    String message = "未知异常";
                    if (e instanceof NumberFormatException) {
                        bundle.putInt("resultCode", 41004);
                        message = "寻址传入的version不是阿拉伯数字";
                    } else if (e instanceof NetworkOnMainThreadException) {
                        bundle.putInt("resultCode", 41001);
                        message = "主线程调用寻址，改为子线程调用";
                    } else if (e instanceof SocketTimeoutException || e instanceof ConnectException) {
                        bundle.putInt("resultCode", 41003);
                        message = "服务器请求失败,请检查网络或者sod服务:" + UrlConstant.getRsbSearchResourceUrl();
                    } else if (e instanceof EOFException) {
                        bundle.putInt("resultCode", 41005);
                        message = "请求寻址失败，请检查应用凭证是否规范，错误信息:" + e;
                    } else {
                        if (TextUtils.isEmpty(e.getMessage())) {
                            message = "请求失败:" + e;
                        } else {
                            message = "请求失败:" + e.getMessage();
                        }
                        bundle.putInt("resultCode", 41000);
                    }
                    bundle.putString("message", message);
                    bundle.putString("messageId", messageId);
                    bundle.putString("resourceList", null);

                } finally {
                    downLatch.countDown();
                }
            }
        });
        executorService.shutdown();
        try {
            downLatch.await();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        return bundle;
    }
}
