package com.xdja.uaac.api;

import android.app.Activity;
import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Log;

import com.xdja.uaac.data.bean.ThirdAppInfo;

/**
 * Author:kongguoguang
 * Date:2018-03-07
 * Time:10:09
 * Summary:
 */

public class UaacApi {

    /**
     * 统一认证应用包名
     */
    private static String uaacPackageName;
    /**
     * 统一认证启动Activity类名
     */
    private static String uaacActivityName;
    /**
     * 票据Provider
     */
    private static final String TOKEN_PROVIDER_URI = "content://com.xdja.app.pj/";
    /**
     * 凭证Provider
     */
    private static final String CREDENTIAL_PROVIDER_URI = "content://com.ydjw.ua.getCredential";
    /**
     * Android 10 SDK版本号
     */
    private static final int ANDROID_10 = 29;
    /**
     * 票据或凭证不存在
     */
    private static final int TOKEN_CREDENTIAL_NOT_EXISTS = -5;
    /**
     * 统一认证未安装
     */
    private static final int UAAC_NOT_EXISTS = -6;
    /**
     * 关联启动未开启
     */
    private static final int CORRELATION_START_NOT_ENABLE = -7;

    private static final String KEY_PACKAGE_NAME = "packageName";
    private static final String KEY_MESSAGE_ID = "messageId";
    private static final String KEY_VERSION = "version";
    private static final String KEY_APP_ID = "appId";
    private static final String KEY_ORG_ID = "orgId";
    private static final String KEY_NETWORK_AREA_CODE = "networkAreaCode";
    private static final String KEY_RESULT_CODE = "resultCode";
    private static final String KEY_MESSAGE = "message";
    private static final String KEY_SAFE_CLIENT_STATUS = "safeClientStatus";
    private static final String KEY_BILL_STR = "billStr";
    private static final String KEY_THIRD_APP_INFO = "third_app_info";
    private static final String KEY_THIRD_APP_PACKAGE_NAME = "third_app_package_name";




    public static void getToken(@NonNull Context context, @NonNull TokenCallback callback){
        getToken(context, "", callback);
    }

    /**
     * 获取票据
     *
     * @param context     Context
     * @param packageName packageName
     * @param callback    TokenCallback
     */
    public static void getToken(@NonNull Context context, String packageName, @NonNull TokenCallback callback) {

        if (!isUaacInstalled(context) && !isUnifyAuthorizeInstalled(context) &&
                !isSafeClientInstalled(context) && !isJxInstalled(context)) {
            callback.onError("未检测到统一认证，请先安装！");
            return;
        }

        if (TextUtils.isEmpty(packageName)) {
            packageName = context.getPackageName();
        }

        Uri uri = Uri.parse(TOKEN_PROVIDER_URI + packageName);
        ContentResolver resolver = context.getContentResolver();
        Log.d("UaacApi", "Query provider 1");
        Cursor cursor = resolver.query(uri, null, null, null, null);
        if (cursor == null){

            try {
                Log.d("UaacApi", "Start activity!");
                Intent intent = new Intent("com.xdja.uaac.action.start");
                intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                context.startActivity(intent);

            }catch (Exception e){
                Log.e("UaacApi", e.getMessage());
                Log.d("UaacApi", "Start activity failed!");
                callback.onError("未开启统一认证关联启动权限！");
                return;
            }

            for (int i = 0; i < 30; i++){
                Log.d("UaacApi", "Query provider " + (i + 2));
                cursor = resolver.query(uri, null, null, null, null);
                if (cursor != null){
                    parseCursor(context, cursor, callback, packageName);
                    return;
                }

                try {
                    Thread.sleep(50);
                } catch (InterruptedException e) {
                        e.printStackTrace();
                }
            }

            callback.onError("未开启统一认证关联启动权限！");

        }else {
            parseCursor(context, cursor, callback, packageName);
        }
    }

    /**
     *
     * @return 0：注销成功；-1：无权调用注销接口
     */
    public static int logoutUaac(@NonNull Context context){
        Uri uri = Uri.parse(TOKEN_PROVIDER_URI + context.getPackageName());
        ContentResolver contentResolver = context.getContentResolver();
        int result = 0;
        try {
            result = contentResolver.delete(uri, null, null);
        }catch (Exception e){
            return result;
        }

        return result;
    }

    /**
     * 应用退出时通知统一认证
     * @param context
     */
    public static void notifyLogout(@NonNull Context context){
        Intent intent = new Intent("com.xdja.unifyauthorize.ACTION_THIRD_APP_LOGOUT");
        intent.putExtra("PACKAGE_NAME", context.getPackageName());
        context.sendBroadcast(intent);
    }

    /**
     * 判断新版统一认证是否安装
     */
    private static boolean isUaacInstalled(Context context) {
        PackageManager mPackageManager = context.getPackageManager();
        ApplicationInfo mApplicationInfo = null;
        try {
            mApplicationInfo = mPackageManager.getApplicationInfo("com.xdja.uaac", PackageManager.GET_META_DATA);
            if (mApplicationInfo != null) {
                uaacPackageName = "com.xdja.uaac";
                uaacActivityName = "com.xdja.uaac.ui.InitActivity";
                return true;
            }
        } catch (PackageManager.NameNotFoundException e) {
            return false;
        }
        return false;
    }

    /**
     * 判断老版统一认证是否安装
     *
     * @param context
     * @return
     */
    private static boolean isUnifyAuthorizeInstalled(Context context) {
        PackageManager mPackageManager = context.getPackageManager();
        ApplicationInfo mApplicationInfo = null;
        try {
            mApplicationInfo = mPackageManager.getApplicationInfo("com.xdja.unifyauthorize", PackageManager.GET_META_DATA);
            if (mApplicationInfo != null) {
                uaacPackageName = "com.xdja.unifyauthorize";
                uaacActivityName = "com.xdja.unifyauthorize.activity.LoginActivity";
                return true;
            }
        } catch (PackageManager.NameNotFoundException e) {
            return false;
        }
        return false;
    }

    /**
     * 判断安全接入是否安装
     *
     * @param context
     * @return
     */
    private static boolean isSafeClientInstalled(Context context) {
        PackageManager mPackageManager = context.getPackageManager();
        ApplicationInfo mApplicationInfo = null;
        try {
            mApplicationInfo = mPackageManager.getApplicationInfo("com.xdja.safeclient", PackageManager.GET_META_DATA);
            if (mApplicationInfo != null) {
                uaacPackageName = "com.xdja.safeclient";
                uaacActivityName = "com.xdja.safeclient.frame.SplashActivity";
                return true;
            }
        } catch (PackageManager.NameNotFoundException e) {
            return false;
        }
        return false;
    }

    /**
     * 判断警信是否安装
     *
     * @param context
     * @return
     */
    private static boolean isJxInstalled(Context context) {
        PackageManager mPackageManager = context.getPackageManager();
        ApplicationInfo mApplicationInfo = null;
        try {
            mApplicationInfo = mPackageManager.getApplicationInfo("com.xdja.jxclient", PackageManager.GET_META_DATA);
            if (mApplicationInfo != null) {
                uaacPackageName = "com.xdja.jxclient";
                uaacActivityName = "com.xdja.unifyauthorize.activity.LoginActivity";
                return true;
            }
        } catch (PackageManager.NameNotFoundException e) {
            return false;
        }
        return false;
    }



    /**
     * 解析统一认证返回的Cursor
     */
    private static void parseCursor(Context context, Cursor cursor, TokenCallback callback, String packageName){
        cursor.moveToFirst();
        int resultCode = cursor.getInt(cursor.getColumnIndex(KEY_RESULT_CODE));

        if (resultCode == 0){
            String token = cursor.getString(cursor.getColumnIndex(KEY_BILL_STR));
            int safeClientStatus;
            try {
                safeClientStatus = cursor.getInt(cursor.getColumnIndex(KEY_SAFE_CLIENT_STATUS));
            }catch (Exception e){
                safeClientStatus = 1;
            }
            callback.onSuccess(token, safeClientStatus == 0);
        } else {
            String message = cursor.getString(cursor.getColumnIndex(KEY_MESSAGE));
            callback.onError(message);

            //票据不存在，Android 10以下设备会自动弹出登录界面，Android 10需要主动拉起登录界面
            if (Build.VERSION.SDK_INT >= ANDROID_10 && resultCode == TOKEN_CREDENTIAL_NOT_EXISTS){
                Intent intent = new Intent();
                intent.setClassName(uaacPackageName, uaacActivityName);
                intent.putExtra(KEY_THIRD_APP_PACKAGE_NAME, packageName);
                if (!(context instanceof Activity)){
                    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                }
                try {
                    context.startActivity(intent);
                } catch (Exception e){
                    callback.onError("统一认证登录界面启动失败");
                }
            }
        }
        cursor.close();
    }


    public static void getCredential(final Context context, final String messageId, String version, String appId, String orgId,
                                     String networkAreaCode, String packageName, final CredentialCallback callback){
        if (!isUaacInstalled(context) && !isSafeClientInstalled(context) ) {
            callback.onResult(getErrorBundle(UAAC_NOT_EXISTS, messageId));
            return;
        }

        if (TextUtils.isEmpty(packageName)) {
            packageName = context.getPackageName();
        }

        final ThirdAppInfo thirdAppInfo = new ThirdAppInfo();
        thirdAppInfo.setMessageId(messageId);
        thirdAppInfo.setPackageName(packageName);
        thirdAppInfo.setAppId(appId);
        thirdAppInfo.setOrgId(orgId);
        thirdAppInfo.setNetworkAreaCode(networkAreaCode);
        thirdAppInfo.setVersion(version);

        final Uri uri = Uri.parse(CREDENTIAL_PROVIDER_URI);
        final ContentResolver resolver = context.getContentResolver();
        final Bundle params = new Bundle();
        params.putString(KEY_MESSAGE_ID, messageId);
        params.putString(KEY_VERSION, version);
        params.putString(KEY_APP_ID, appId);
        params.putString(KEY_ORG_ID, orgId);
        params.putString(KEY_NETWORK_AREA_CODE, networkAreaCode);
        params.putString(KEY_PACKAGE_NAME, packageName);

        try {
            Log.d("UaacApi", "Call provider");
            Bundle bundle = resolver.call(uri, "", null, params);
            callback.onResult(bundle);
            int resultCode = bundle.getInt(KEY_RESULT_CODE);
            //凭证不存在，Android 10以下设备会自动弹出登录界面，Android 10需要主动拉起登录界面
            if (Build.VERSION.SDK_INT >= ANDROID_10 && resultCode == TOKEN_CREDENTIAL_NOT_EXISTS){
                Intent intent = new Intent();
                intent.setClassName(uaacPackageName, uaacActivityName);
                intent.putExtra(KEY_THIRD_APP_INFO, thirdAppInfo);
                if (!(context instanceof Activity)){
                    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                }
                try {
                    context.startActivity(intent);
                } catch (Exception e){
                    callback.onResult(getErrorBundle(UAAC_NOT_EXISTS, messageId));
                }
            }
        } catch (Exception e){
            callback.onResult(getErrorBundle(CORRELATION_START_NOT_ENABLE, messageId));
        }
    }

    private static Bundle getErrorBundle(int errorCode, String messageId){
        Bundle bundle = new Bundle();
        bundle.putInt(KEY_RESULT_CODE, errorCode);
        if (errorCode == UAAC_NOT_EXISTS){
            bundle.putString(KEY_MESSAGE, "统一认证未安装或版本过低");
        } else if (errorCode == CORRELATION_START_NOT_ENABLE){
            bundle.putString(KEY_MESSAGE, "未开启统一认证关联启动权限！");
        }

        bundle.putString(KEY_MESSAGE_ID, messageId);
        bundle.putString(KEY_VERSION, "1");
        return bundle;
    }

}
