package com.xdja.base.system.action;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.MessageSourceAware;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.User;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.RequestMapping;

import com.xdja.base.common.action.BaseAction;
import com.xdja.base.system.SystemPropertiesConfig;
import com.xdja.base.system.entity.Function;
import com.xdja.base.system.manager.SystemManager;
import com.xdja.base.ucm.systemlog.entity.SystemLog;
import com.xdja.base.ucm.systemlog.manager.SystemLogManager;
import com.xdja.base.util.Constants;
import com.xdja.base.util.HttpSessionUtil;
import com.xdja.base.util.IPV4Util;
import com.xdja.base.util.SystemConstants;

/**
 * @ProjectName：ucm
 * @ClassName：LoginAction
 * @Description：系统登录
 * @author: mayanpei
 * @date: 2014-8-13 下午4:09:39
 * 
 */
@Controller
public class LoginAction extends BaseAction implements MessageSourceAware {

	private final Logger logger = LoggerFactory.getLogger(getClass());

	@Autowired
	private SystemManager systemManager;

	@Autowired
	private SystemLogManager logManager;

	private MessageSource messageSource;

	public void setMessageSource(MessageSource messageSource) {
		this.messageSource = messageSource;
	}

	/**
	 * @Description：系统登录
	 * @author: mayanpei
	 * @date: 2014-8-13 下午5:25:49
	 * @param model
	 *            ModelMap
	 * @return
	 */
	@RequestMapping("/index")
	public String index(ModelMap model) {
		model.put("version", SystemPropertiesConfig.get("version"));
		return "login";
	}

	/**
	 * @Description：系统登录
	 * @author: mayanpei
	 * @date: 2014-8-13 下午5:26:04
	 * @param model
	 *            ModelMap
	 * @param request
	 *            请求
	 * @return
	 */
	@RequestMapping("/login")
	public String login(ModelMap model, HttpServletRequest request) {
		try {
			
			if (SecurityContextHolder.getContext().getAuthentication() == null
					|| SecurityContextHolder.getContext().getAuthentication().getPrincipal()
							.equals(Constants.ANONYMOUS_USER)) {
				return "redirect:/index.do";
			}
			
			User user = (User) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
			
			if (user != null && request.getSession().getAttribute(Constants.LOGIN_USER) != null) {
				
				return "redirect:/homePage.do";
			}
			String username = "";
			if (user != null ) {
				username = user.getUsername();
			}
			
			com.xdja.base.system.entity.User admin = systemManager.getUserByUserName(username);
			HttpSessionUtil.getRequest().getSession().setAttribute(Constants.LOGIN_USER, admin);

			// 构造Log日志对象并保存
			SystemLog log = new SystemLog();
			log.setAdminId(admin.getId());
			log.setAdministrator(username);
			log.setLogContent("登录系统");
			log.setIp(IPV4Util.getIpAddr(request));
			log.setLogLevel(Constants.UCM_LOG_LEVEL.info.value);
			log.setLogType(Constants.UCM_LOG_TYPE.loginLog.value);
			log.setLogTypeDetail(Constants.UCM_LOG_TYPE_DETAIL.login.value);
			log.setSystem("ucm");
			logManager.addSystemLog(log); 
			

		} catch (Exception e) {
			logger.error("登录失败", e);
		}
		return "redirect:/homePage.do";
	}

	/**
	 * @Description：跳转转首页
	 * @author: mayanpei
	 * @date: 2014-8-15 上午9:26:08
	 * @param model
	 *            ModelMap
	 * @param request
	 *            请求
	 * @return
	 */
	@RequestMapping("/homePage")
	public String homePage(ModelMap model, HttpServletRequest request) {
		try {
			com.xdja.base.system.entity.User user = HttpSessionUtil.getUser();
			// 判断是否无登录直接访问
			if (user == null) {
				// 直接返回登录页面
				return "redirect:/index.do";
			}

			List<Function> list = new ArrayList<Function>();
			
			//查询系统导航菜单
			List<Function> temp = systemManager.getFunctions(user.getLoginName(), user.getType());
			for(Function f : temp) {
				if(f.getParentId() == Constants.MENU_ROOT_ID) {
					list.add(f);
				}
			} 
			
			model.addAttribute("functionList", list);
			model.addAttribute("username", user.getLoginName());
		} catch (Exception e) {
			logger.error("查询登录用户信息出错", e);
		}
		return "index";
	}

	/**
	 * @Description：系统左边
	 * @author: mayanpei
	 * @date: 2014-8-13 下午5:26:29
	 * @param functionId
	 *            菜单ID
	 * @param username
	 *            用户名
	 * @param model
	 *            ModelMap
	 * @param response
	 *            响应
	 * @return
	 */
	@RequestMapping("/left")
	public String left(Long functionId, String username, ModelMap model, HttpServletResponse response) {
		try {
			com.xdja.base.system.entity.User user = HttpSessionUtil.getUser();
			// 判断是否无登录直接访问
			if (user == null) {
				// 直接返回登录页面
				return "reload";
			}
			List<Function> list = new ArrayList<Function>();
			list = systemManager.getFunctionsByParentId(functionId, username);
			model.addAttribute("menuList", list);
		} catch (Exception e) {
			return "reload";
		}
		return "left";
	}

	/**
	 * @Description：系统登录超时
	 * @author: mayanpei
	 * @date: 2014-8-13 下午5:26:45
	 * @param model
	 *            ModelMap
	 * @param request
	 *            请求
	 * @return
	 */
	@RequestMapping("/sessionTimeout")
	public String sessionTimeout(ModelMap model, HttpServletRequest request) {
		String message = messageSource.getMessage("sessionTimeout.msg", new Object[] {}, Locale.getDefault());
		model.addAttribute("message", message);
		request.getSession().removeAttribute(Constants.LOGIN_USER);
		return "reload";
	}

	/**
	 * @Description：系统访问拒绝
	 * @author: mayanpei
	 * @date: 2014-8-13 下午5:27:04
	 * @param request
	 *            请求
	 * @param model
	 *            ModelMap
	 * @return
	 */
	@RequestMapping("/accessDenied")
	public String accessDenied(HttpServletRequest request, ModelMap model) {
		Exception e = (Exception) request.getSession().getAttribute("SPRING_SECURITY_LAST_EXCEPTION");
		logger.error(e.getMessage(), e);
		String message = messageSource.getMessage("accessDenied.msg", new Object[] {}, Locale.getDefault());
		model.addAttribute("message", message);
		return "accessDenied";
	}

	/**
	 * @Description：没有访问权限
	 * @author: mayanpei
	 * @date: 2014-8-13 下午5:27:19
	 * @param model
	 *            ModelMap
	 * @param request
	 *            请求
	 * @return
	 */
	@RequestMapping("/noAuthority")
	public String noAuthority(ModelMap model, HttpServletRequest request) {
		String message = messageSource.getMessage("noAuthority.msg", new Object[] {}, Locale.getDefault());
		model.addAttribute("message", message);
		return "noAuthority";
	}

	/**
	 * @Description： 退出
	 * @author: mayanpei
	 * @date: 2014-8-20 上午9:51:49
	 * @param model
	 *            ModelMap
	 * @param request
	 *            请求
	 * @return
	 */
	@RequestMapping("/logout")
	public String logout(ModelMap model, HttpServletRequest request) {
		if (request.getSession().getAttribute(Constants.LOGIN_USER) != null) {
			com.xdja.base.system.entity.User user = (com.xdja.base.system.entity.User) request.getSession()
					.getAttribute(Constants.LOGIN_USER);
			SystemLog log = new SystemLog();
			log.setAdminId(user.getId());
			log.setAdministrator(user.getLoginName());
			log.setLogContent("退出系统");
			log.setLogLevel(Constants.UCM_LOG_LEVEL.info.value);
			log.setLogType(Constants.UCM_LOG_TYPE.loginLog.value);
			log.setLogTypeDetail(Constants.UCM_LOG_TYPE_DETAIL.loginOut.value);
			log.setIp(IPV4Util.getIpAddr(request));
			logManager.addSystemLog(log);
		}
		request.getSession().removeAttribute(Constants.LOGIN_USER);
		return "redirect:/j_spring_security_logout";
	}

	/**
	 * @Description：api
	 * @author: mayanpei
	 * @date: 2014-8-13 下午5:27:38
	 * @param model
	 *            ModelMap
	 * @return
	 */
	@RequestMapping("/api")
	public String api(ModelMap model) {
		if (SecurityContextHolder.getContext().getAuthentication().getPrincipal().equals(Constants.ANONYMOUS_USER)) {
			return "redirect:/index.do";
		}
		return "api";
	}

	/**
	 * @Description：api
	 * @author: mayanpei
	 * @date: 2014-8-13 下午5:27:38
	 * @param model
	 *            ModelMap
	 * @return
	 */
	@RequestMapping("/config.do")
	public String config(ModelMap model) {
		SystemConstants systemConstants = SystemConstants.getInstances();
		Map<String, String> map = systemConstants.getAllSystemParam();
		Iterator<Entry<String, String>> it = map.entrySet().iterator();
		while (it.hasNext()) {
			Entry<String, String> en = (Entry<String, String>) it.next();
			logger.info("key: " + en.getKey());
			logger.info("value:  " + en.getValue());
		}
		model.put("data", systemConstants.getAllSystemParam());
		return "constants";
	}
}
