package com.xdja.base.ucm.jmx.server.service;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;

import com.xdja.base.ucm.jmx.server.bean.ClientListBean;
import com.xdja.base.ucm.jmx.server.bean.RegistReqBean;
import com.xdja.base.ucm.jmx.server.bean.RegistRespBean;
import com.xdja.base.ucm.jmx.server.bean.Status;
import com.xdja.base.ucm.jmx.server.dao.RegistDao;

/**
 * UCM认证及数据获取
 * @author liqi
 * @date 2015-03-09 15:49:22
 */
@Service
public final class UcmRegist implements UcmRegistMBean, ApplicationContextAware {
	
	private static final Logger logger = LoggerFactory.getLogger(UcmRegist.class);
	
	private static RegistDao dao;
	
//	@Autowired
//	RegistDao dao;
	
	private static final ConcurrentHashMap<String, AtomicInteger> qosMap = new ConcurrentHashMap<String, AtomicInteger>();
	
	private static final int MAX_DAY_NVOKES = 200;
	
	private static String getDate() {
		Date now = new Date();
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd"); // 可以方便地修改日期格式
		return dateFormat.format(now);
	}
	
	
	private static boolean qosReject(String client, String ip) {
		
		String key = String.format("%s_%s_%s", client, ip, getDate()); 
		AtomicInteger ai = qosMap.get(key);
		
		if (null == ai) {
		   ai = new AtomicInteger(0);
		}
		
		if (ai.incrementAndGet() > MAX_DAY_NVOKES) {
			qosMap.put(key, ai);
			return true;
		}
		
		return false;
		  
	}
	
	
	
	/**
	 * 身份检测和数据获取
	 * @author liqi
	 * @date 2015-03-09 15:49:22
	 */
	@Override
	public RegistRespBean checkAndRegisterData(RegistReqBean reqBean) {
		
		RegistRespBean result = new RegistRespBean();
		result.setData(null);
		
		try {
			
			if(null == reqBean ) { 
				
				logger.info("reqBean is null，");  
				result.setStatus(Status.PARAMS_ERROR);
				return result;
			}
			
			if(StringUtils.isBlank(reqBean.getAppId())) {
				logger.info("reqBean.appid is null，");  
				result.setStatus(Status.PARAMS_ERROR);
				return result;
			}
				
			if(StringUtils.isBlank(reqBean.getIp())) {
				logger.info("reqBean.ip is null，");  
				result.setStatus(Status.PARAMS_ERROR);
				return result;
			}
			
			if( 65535 < reqBean.getPort() || reqBean.getPort() < 1024) {
				logger.info("reqBean.getPort is error，");  
				result.setStatus(Status.PARAMS_ERROR);
				return result;
			}
			
			
			if( qosReject(reqBean.getAppId().trim(), reqBean.getIp().trim()) ) {
				
				logger.info(String.format("客户端[%s]访问[%s]应用违反QOS规则，", reqBean.getAppId(), reqBean.getIp()));
				
				result.setData(null);
				result.setStatus(Status.EXCEPTION);
				return result;
			}
			
			if(null == dao) {
				logger.error(String.format("客户端[%s]访问数据库的dao未初始化，", reqBean.getIp()));
				result.setStatus(Status.EXCEPTION);
				result.setData(null);
				return result;
			}
			
			List<ClientListBean> beanLists = dao.getList(reqBean.getAppId().trim() );
			
			logger.info( beanLists.size() + " sisieeseseserserser");
			
			//应用不存在
			if (null == beanLists || beanLists.size() == 0) {
				
				logger.info(String.format("客户端[%s]访问[%s]应用不存在此应用，", reqBean.getIp(), reqBean.getAppId()));
				
				result.setStatus(Status.APP_NOT_EXISTS);
				result.setData(null);
				return result;
			}
			
			//判断IP和端口是否匹配
			boolean ipExists = false;
			boolean portExists = false;
			for (ClientListBean bean : beanLists) { 
				if(reqBean.getIp().trim().equals(bean.getIp().trim())) {
					ipExists = true;
					if( reqBean.getPort() == Integer.parseInt(bean.getPort())) {
						portExists = true;
						break;
					}  
				} 
			}
	
			//判断ip是否存在
			if (!ipExists) {
				
				logger.info(String.format("客户端[%s]访问[%s]应用未配置此应用IP地址，", reqBean.getIp(), reqBean.getAppId()));
				
				result.setStatus(Status.IP_NOT_EXISTS);
				result.setData(null);
				return result;
			}
			//判断端口是否存在
			if (ipExists && !portExists) {
				
				logger.info(String.format("客户端[%s]访问[%s]应用未配置此应用端口[%d]，", reqBean.getIp(), reqBean.getAppId(), reqBean.getPort()));
				
				result.setStatus(Status.PORT_NOT_EXISTS); 
				result.setData(null);
				return result;
			}
			beanLists = null;
			
			List<Map<String, Object>> dataList = dao.queryForDataList(reqBean.getAppId().trim());
			
			HashMap<String, Object> resultMap = new HashMap<String, Object>(50);
			
			for (Map<String, Object> map : dataList) {
				resultMap.put((String) map.get("c_data_key"), map.get("c_data_value"));
			}
			 
			  
			result.setStatus(Status.SUCCESS);
			result.setData(resultMap);
			
			logger.info(String.format("客户端[%s]访问[%s]应用返回数据[%d]条，", reqBean.getIp(), reqBean.getAppId(), dataList.size()));
		} catch(Exception ex) {
			logger.error("客户端访问服务器出现异常");
			logger.error(ex.getMessage()); 
			result.setStatus(Status.EXCEPTION);
		}
		return result;   
	}


	@Override
	public void setApplicationContext(ApplicationContext applicationContext)
			throws BeansException {
		dao = applicationContext.getBean(RegistDao.class);
	}

}
