package com.xdja.base.ucm.role.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.stereotype.Repository;

import com.xdja.base.common.dao.BaseDao;
import com.xdja.base.system.entity.Function;
import com.xdja.base.system.entity.Role;

/**
 * @ProjectName：ucm
 * @ClassName：RoleDao
 * @Description：角色管理DAO 层实现
 * @author: mayanpei
 * @date: 2014-8-15 下午2:25:48
 * 
 */
@Repository
public class RoleDao extends BaseDao {

	/**
	 * @Description：查询角色信息
	 * @author: mayanpei
	 * @date: 2014-8-15 下午2:02:23
	 * @param name
	 *            角色名称
	 * @return
	 */
	public List<Map<String, Object>> queryRoleList(String name) {
		StringBuffer sql = new StringBuffer();
		sql.append("SELECT n_id as id,c_name AS name,c_note AS note FROM t_ucm_role");
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		if (StringUtils.isNotBlank(name)) {
			sql.append(" WHERE c_name = :name");
			sqlParam.addValue("name", name);
		}
		sql.append(" ORDER BY n_id DESC");
		return queryForList(sql.toString(), sqlParam);
	}

	/**
	 * @Description：保存角色信息
	 * @author: mayanpei
	 * @date: 2014-8-15 下午2:04:14
	 * @param role
	 *            角色
	 */
	public void saveRole(Role role) {
		String sql = "INSERT INTO t_ucm_role (c_name,c_note) VALUES(:name,:note)";
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("name", role.getName());
		sqlParam.addValue("note", role.getNote());
		executeSql(sql, sqlParam);
	}

	/**
	 * @Description：更新角色信息
	 * @author: mayanpei
	 * @date: 2014-8-15 下午2:04:25
	 * @param role
	 *            角色
	 */
	public void updateRole(Role role) {
		String sql = "UPDATE t_ucm_role set c_name = :name,c_note = :note WHERE n_id=:id";
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("name", role.getName());
		sqlParam.addValue("note", role.getNote());
		sqlParam.addValue("id", role.getId());
		executeSql(sql, sqlParam);
	}

	/**
	 * @Description：根据角色ID查询角色信息
	 * @author: mayanpei
	 * @date: 2014-8-18 下午4:00:43
	 * @param roleId
	 *            角色ID
	 * @return role
	 */
	public Role queryRoleByRoleId(Long roleId) {
		StringBuffer sql = new StringBuffer("SELECT n_id as id,c_name AS name,c_note AS note ");
		sql.append(" FROM t_ucm_role  WHERE n_id = :id");
		RowMapper<Role> mapper = new RowMapper<Role>() {
			public Role mapRow(ResultSet rs, int rowNum) throws SQLException {
				Role role = new Role();
				role.setId(rs.getLong("id"));
				role.setName(rs.getString("name"));
				role.setNote(rs.getString("note"));
				return role;
			}
		};
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("id", roleId);
		
		
		List<Role> list = query(sql.toString(), sqlParam, mapper);
		
		if (null == list || list.size() == 0 ) {
			return null;
		}
		return list.get(0);
	}

	/**
	 * @Description：删除角色信息
	 * @author: mayanpei
	 * @date: 2014-8-15 下午2:04:25
	 * @param roleId
	 *            角色ID
	 */
	public void deleteRole(Long roleId) {
		String sql = "DELETE FROM t_ucm_role WHERE n_id=:id";
		String sqlFun = "DELETE FROM  t_ucm_function_role WHERE n_role_id=:id";
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("id", roleId);
		executeSql(sql, sqlParam);
		executeSql(sqlFun, sqlParam);
	}

	/**
	 * @Description：查询角色已有权限
	 * @author: mayanpei
	 * @date: 2014-8-19 上午9:52:40
	 * @param roleId
	 *            角色ID
	 * @return
	 */
	public List<Map<String, Object>> queryByRoleId(Long roleId) {
		String sql = "SELECT t.n_function_id FROM t_ucm_function_role t WHERE t.n_role_id =:roleId";
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("roleId", roleId);
		return queryForList(sql, sqlParam);
	}

	/**
	 * @Description：保存角色权限
	 * @author: mayanpei
	 * @date: 2014-8-15 下午2:04:37
	 * @param roleId
	 *            角色ID
	 * @param funcIds
	 *            菜单ID
	 */
	public void saveRoleFunc(Long roleId, String[] funcIds) {
		String sql = "DELETE FROM t_ucm_function_role WHERE n_role_id =:roleId";
		String addSql = "INSERT INTO t_ucm_function_role(n_role_id,n_function_id) VALUES(:roleId,:funId)";
		MapSqlParameterSource parameterSource = new MapSqlParameterSource();
		parameterSource.addValue("roleId", roleId);
		update(sql, parameterSource);
		if (funcIds.length > 0) {
			MapSqlParameterSource[] parameterSources = new MapSqlParameterSource[funcIds.length];
			for (int i = 0; i < funcIds.length; i++) {
				MapSqlParameterSource parameter = new MapSqlParameterSource();
				parameter.addValue("roleId", roleId);
				parameter.addValue("funId", funcIds[i]);
				parameterSources[i] = parameter;
			}
			addOrUpdate(addSql, parameterSources);
		}
	}

	/**
	 * @Description：查询菜单权限
	 * @author: mayanpei
	 * @date: 2014-8-19 上午10:18:40
	 * @param functionId
	 *            菜单ID
	 * @param datas
	 *            菜单数据
	 * @return List<Function>
	 */
	public List<Function> queryTreeFunction(Long functionId, List<Function> datas) {

		List<Function> result = new LinkedList<Function>();

		if (datas == null) {
			datas = this.getChildren();
		}

		for (Function fun : datas) {
			if (fun.getParentId().intValue() == functionId.intValue()) {
				fun.setChildren(this.queryTreeFunction(fun.getId(), datas));
				result.add(fun);
			}
		}
		return result;
	}

	/**
	 * @Description：查询菜单
	 * @author: mayanpei
	 * @date: 2014-8-19 上午10:19:52
	 * @return
	 */
	public List<Function> getChildren() {
		StringBuffer sql = new StringBuffer("SELECT n_id ,c_name,n_parent_id");
		sql.append(" FROM t_ucm_function  ");
		RowMapper<Function> mapper = new RowMapper<Function>() {
			public Function mapRow(ResultSet rs, int rowNum) throws SQLException {
				Function function = new Function();
				function.setId(rs.getLong("n_id"));
				function.setName(rs.getString("c_name"));
				function.setParentId(rs.getLong("n_parent_id"));
				return function;
			}
		};
		return query(sql.toString(), new MapSqlParameterSource(), mapper);
	}
}
